package com.bizunited.platform.security.local.service;

import com.bizunited.platform.security.sdk.config.SimpleSecurityProperties;
import com.bizunited.platform.security.sdk.constant.Constants;
import com.bizunited.platform.security.sdk.event.AuthenticationRbacEventListener;
import com.bizunited.platform.security.sdk.vo.LoginDetails;
import com.google.common.collect.Lists;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.security.access.AccessDeniedException;
import org.springframework.security.access.ConfigAttribute;
import org.springframework.security.access.SecurityConfig;
import org.springframework.security.core.Authentication;
import org.springframework.security.core.GrantedAuthority;
import org.springframework.security.core.context.SecurityContextHolder;
import org.springframework.security.web.FilterInvocation;
import org.springframework.security.web.access.intercept.FilterInvocationSecurityMetadataSource;
import org.springframework.security.web.util.matcher.AntPathRequestMatcher;
import org.springframework.util.CollectionUtils;
import org.springframework.web.method.HandlerMethod;
import org.springframework.web.servlet.mvc.method.RequestMappingInfo;
import org.springframework.web.servlet.mvc.method.annotation.RequestMappingHandlerMapping;

import javax.servlet.http.HttpServletRequest;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;

/**
 * 该功能负责最主要的访问权限验证，既是确认当前的访问请求具备什么样的角色设定</br>
 * 请注意这里的判定逻辑是：如果没有明确设定某个URL的访问角色要求，那么默认该URL至少具备“匿名访问”角色要求
 * @author yinwenjie
 */
public class CustomFilterInvocationSecurityMetadataSource implements FilterInvocationSecurityMetadataSource {
  /**
   * 匿名用户鉴权体系
   */
  private static final SecurityConfig ANONYMOUS_CONFIG = new SecurityConfig(Constants.ANONYMOUS);
  @Autowired
  private SimpleSecurityProperties securityProperties;
  @Autowired
  private RequestMappingHandlerMapping frameworkEndpointHandler;
  @Autowired(required = false)
  private AuthenticationRbacEventListener authenticationRbacEventListener;
  
  
  /* (non-Javadoc)
   * @see org.springframework.security.access.SecurityMetadataSource#getAttributes(java.lang.Object)
   */
  @Override
  public Collection<ConfigAttribute> getAttributes(Object object) throws IllegalArgumentException {
    FilterInvocation filterInvocation = (FilterInvocation) object;
    HttpServletRequest httpRequest = filterInvocation.getHttpRequest();
    List<ConfigAttribute> configs = new ArrayList<>();
    
    /*
     * 该方法确认当前请求的URL信息，具备怎样的角色配置。
     * 请注意这里方法的判定逻辑是：如果没有明确设定某个URL的访问角色要求，那么默认该URL至少具备“匿名访问”角色要求
     * 处理过程如下：
     * 
     * 1、首先需要肯定，在以下情况出现时，不需要判定当前请求具备的角色，给定匿名角色：
     * 当前URL属于DEFAULT_IGNOREURLS常量或IgnoreUrls配置属性设定的URL（不再向后执行）；
     * 
     * 2、在以下情况出现时，当前请求所具备的角色就是当前登录者具有的角色：
     * 当前登录者具有管理员权限（不再向后执行）
     * 
     * 3、如果以上事实都不成立，那么就要求上层的RBAC模块对给定的URL信息进行角色绑定情况判定
     * 并将判定得到的角色业务编号信息进行返回
     * */
    
    // 1、========
    ArrayList<String> currentIgnoreUrls = new ArrayList<>();
    String[] ignoreUrls = securityProperties.getIgnoreUrls();
    if(ignoreUrls != null && ignoreUrls.length > 0) {
      currentIgnoreUrls.addAll(Lists.newArrayList(ignoreUrls));
    }
    currentIgnoreUrls.addAll(Lists.newArrayList(SimpleSecurityProperties.DEFAULT_IGNOREURLS));
    for (String ignoreUrl : currentIgnoreUrls) {
      AntPathRequestMatcher requestMatcher = new AntPathRequestMatcher(ignoreUrl);
      if(requestMatcher.matches(httpRequest)) {
        configs.add(ANONYMOUS_CONFIG);
        return configs;
      }
    }
    
    // 2、=======
    if(this.authenticationRbacEventListener == null) {
      throw new AccessDeniedException("not author（no authenticationRbacEventListener）!");
    }
    Set<String> ignoreMethodCheckRoleCodes = this.authenticationRbacEventListener.onRequestIgnoreMethodCheckRoles();
    Authentication authentication = SecurityContextHolder.getContext().getAuthentication();
    if(authentication == null) {
      throw new AccessDeniedException("not author（no authentication）!");
    }
    Collection<? extends GrantedAuthority> currentRoles = authentication.getAuthorities();
    Set<String> currentRoleCodes = currentRoles.stream().map(GrantedAuthority::toString).collect(Collectors.toSet());
    if(!CollectionUtils.isEmpty(ignoreMethodCheckRoleCodes) && !CollectionUtils.isEmpty(currentRoles)) {
      String[] ignoreMethodCheckArrayRoleCodes = ignoreMethodCheckRoleCodes.toArray(new String[] {});
      for (String currentRoleCode : currentRoleCodes) {
        // 如果条件成立，就说明这个用户携带了不需要进行资源鉴权的超级管理员性质的角色
        if(StringUtils.equalsAnyIgnoreCase(currentRoleCode, ignoreMethodCheckArrayRoleCodes)) {
          SecurityConfig securityConfig = new SecurityConfig(currentRoleCode);
          configs.add(securityConfig);
          return configs;
        } 
      } 
    }
    
    // 3、======
    // 首先找到和spring mvc上下文匹配的请求路径
    // 如果没有找到任何处理器，说明这个URL地址并没有加入到权限控制体系，则不做权限过滤
    // 原则就是“未明确拒绝则是允许”
    Object loginDetailsObject = authentication.getDetails();
    if(!(loginDetailsObject instanceof LoginDetails)) {
      throw new AccessDeniedException("not author（no login）!");
    }
    LoginDetails loginDetails = (LoginDetails)loginDetailsObject;
    String tenantCode = loginDetails.getTenantCode();
    Map<RequestMappingInfo , HandlerMethod> pathMapping = frameworkEndpointHandler.getHandlerMethods();
    Set<RequestMappingInfo> requestMappings = pathMapping.keySet();
    List<RequestMappingInfo> matchingResults = Lists.newArrayList();
    String currentMethod = httpRequest.getMethod();
    for (RequestMappingInfo requestMappingInfo : requestMappings) {
      RequestMappingInfo current = requestMappingInfo.getMatchingCondition(httpRequest);
      if(current != null) {
        matchingResults.add(current);
      }
    }
    // “未明确拒绝则是允许”
    if(matchingResults.isEmpty()) {
      configs.add(ANONYMOUS_CONFIG);
      return configs;
    }
    Set<String> currentCompetenceCodes = this.authenticationRbacEventListener.onRequestRoleCodes(matchingResults , tenantCode , currentMethod);
    if(CollectionUtils.isEmpty(currentCompetenceCodes)) {
      throw new AccessDeniedException("not author( no competence mapping role)!");
    }
    for (String currentCompetenceCode : currentCompetenceCodes) {
      SecurityConfig securityConfig = new SecurityConfig(currentCompetenceCode);
      configs.add(securityConfig);
    }
    return configs;
  }

  /* (non-Javadoc)
   * @see org.springframework.security.access.SecurityMetadataSource#getAllConfigAttributes()
   */
  @Override
  public Collection<ConfigAttribute> getAllConfigAttributes() {
    return Collections.emptyList();
  }

  /* (non-Javadoc)
   * @see org.springframework.security.access.SecurityMetadataSource#supports(java.lang.Class)
   */
  @Override
  public boolean supports(Class<?> clazz) {
    return true;
  }
}