package com.bizunited.platform.security.local.service;

import com.bizunited.platform.security.sdk.config.SimpleSecurityProperties;
import com.bizunited.platform.security.sdk.event.AuthenticationDecisionStrategy;
import com.bizunited.platform.security.sdk.event.AuthenticationUserEventListener;
import com.bizunited.platform.security.sdk.vo.LoginDetails;

import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.security.authentication.AuthenticationCredentialsNotFoundException;
import org.springframework.security.authentication.AuthenticationProvider;
import org.springframework.security.authentication.BadCredentialsException;
import org.springframework.security.authentication.UsernamePasswordAuthenticationToken;
import org.springframework.security.core.Authentication;
import org.springframework.security.core.AuthenticationException;
import org.springframework.security.core.authority.SimpleGrantedAuthority;
import org.springframework.security.core.userdetails.UsernameNotFoundException;

import org.springframework.stereotype.Component;
import org.springframework.util.CollectionUtils;

import java.util.ArrayList;
import java.util.List;
import java.util.Set;

/**
 * 重写的用于多种鉴权鉴权场景下，确定用户是否鉴权成功的处理器</br>
 * 其核心原理为：用户信息查询服务（权限模块使用），用来查询指定的用户基本信息（但是包括角色信息）
 * @author yinwenjie
 */
@Component
public class SimpleAuthenticationProvider implements AuthenticationProvider {
  /**
   * 日志
   */
  private static final Logger LOGGER = LoggerFactory.getLogger(SimpleAuthenticationProvider.class);
  private static final String ERROR_TYPE = "目前用户鉴权模式支持6中方式：1、典型的账户名+密码方式；2、典型的账户名 + 密码 + 校验码方式；3、用户名 + 短信验证码 + 校验码方式； 4、手机号 + 密码方式；5、手机号 + 短信验证码；6、手机号 + 短信验证码 + 校验码方式；7、业务系统自定的其它登录方式";
  private static final String ERROR_USER = "未发现指定的用户信息或其数据存储编号，也可能是因为指定的鉴权方式暂未实现";
  private static final String ERROR_ROLE = "未发现指定的用户带有任何角色信息，请检查用户权限设定";
  /**
   * 鉴权方式
   */
  @Autowired(required = false)
  private List<AuthenticationDecisionStrategy> authenticationDecisions;
  @Autowired
  private SimpleSecurityProperties simpleSecurityProperties;
  @Autowired
  private AuthenticationUserEventListener authenticationEventListener;
  
  @Override
  public Authentication authenticate(Authentication authentication) throws AuthenticationException {
    // 没有任何登录鉴权策略是有可能的，例如应用程序作为微服务的组件存在，禁止任何登录
    if(CollectionUtils.isEmpty(authenticationDecisions)) {
      throw new AuthenticationCredentialsNotFoundException("本应用系统/微服务，禁止任何形式的登录(没有任何登录策略)!!");
    }
    //拿到表单的其他信息
    LoginDetails loginDetails = (LoginDetails) authentication.getDetails();
    LOGGER.debug("当前有用户正在进行经典登录鉴权 =  " + loginDetails.toString());
    Integer type = loginDetails.getType();
    if(type == null) {
      type = simpleSecurityProperties.getDefaultLoginType();
    }
    Validate.isTrue(type > 0 && type <= 7 , ERROR_TYPE);
    
    /*
     * node:处理过程为：
     * 1、首先根据type，确定应该使用的鉴权方式，注意：如果这一步抛出异常或者返回了null，说明鉴权失败。
     * 2、接着根据返回的userid，得到这个用户的角色信息，并组装最后的token信息
     * .角色的来源有几部分（参见默认的user-local-security模块）:
     *   a、当前用户直接绑定的角色
     *   b、当前用户所属的一个或者多个用户组绑定的角色
     *   c、当前用户所属组织机构、组织机构的各父级组织结构所绑定的角色
     * .注意：以上步骤中，如果找到了ADMIN这样存在于ignoreMethodCheckRoles配置中的角色时，就不需要再继续找了。
     * .因为一旦有了这个角色，就不再受角色-功能的绑定限制了，任何功能都可以访问
     * CustomUserSecurityDetailsService读取用户详细信息时，需要这个密码
     * */
    
    // 1、==========
    boolean success = false;
    try {
      for (AuthenticationDecisionStrategy authenticationDecisionStrategy : authenticationDecisions) {
        Integer currentCode = authenticationDecisionStrategy.type().getCode();
        if(currentCode.equals(type)) {
          success = authenticationDecisionStrategy.onAuthenticate(loginDetails);
          break;
        }
      }
    } catch(RuntimeException e) {
      LOGGER.error(e.getMessage() , e);
      throw new AuthenticationCredentialsNotFoundException(e.getMessage() , e);
    }
    if(!success
        || StringUtils.isAnyBlank(loginDetails.getTenantCode() , loginDetails.getAccount() , loginDetails.getPassword())) {
      throw new UsernameNotFoundException(ERROR_USER);
    }
    
    // 2、=========
    // 查询用户角色信息
    if(authenticationEventListener != null) {
      Set<String> roleCodes = this.authenticationEventListener.onRequestRoleCodes(loginDetails.getTenantCode(), loginDetails.getAccount());
      if(CollectionUtils.isEmpty(roleCodes)) {
        throw new BadCredentialsException(ERROR_ROLE);
      }
      List<SimpleGrantedAuthority> authorities = new ArrayList<>();
      for (String roleCode : roleCodes) {
        SimpleGrantedAuthority authoritie = new SimpleGrantedAuthority(StringUtils.upperCase(roleCode));
        authorities.add(authoritie);
      }
      // 构建返回的用户登录成功的token
      return new UsernamePasswordAuthenticationToken(loginDetails.getAccount(), loginDetails.getPassword(), authorities);
    } else {
      throw new BadCredentialsException("未配置nebula-security组件的实现模块，请检查配置!!");
    }
  }

  @Override
  public boolean supports(Class<?> authentication) {
    // TODO 正常来说，这里应该根据不同的典型鉴权场景确认该provider是否可用
    return true;
  }
}