package com.bizunited.platform.task.local.service.scheduler;

import java.util.Collection;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.BlockingQueue;
import java.util.concurrent.ScheduledFuture;
import java.util.concurrent.TimeUnit;

import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.tuple.Pair;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.context.ApplicationContext;
import org.springframework.context.annotation.Lazy;
import org.springframework.context.annotation.Scope;
import org.springframework.scheduling.concurrent.ThreadPoolTaskScheduler;
import org.springframework.scheduling.support.CronTrigger;
import org.springframework.stereotype.Component;
import org.springframework.util.CollectionUtils;

import com.bizunited.platform.common.service.redis.RedisMutexService;
import com.bizunited.platform.task.local.configuration.DynamicTaskProperties;
import com.google.common.collect.Maps;

/**
 * 动态任务master独占权保持任务，每一个任务都负责都过applicationName_appCode的独占权保持
 * @author yinwenjie
 */
@Component("_DynamicMasterKeeperTask")
@Scope("prototype")
@Lazy
public class DynamicMasterKeeperTask implements Runnable  {
  /**
   * 队列中的每一个节点携带两个数据，左侧：applicationName；右侧：appCode
   */
  private BlockingQueue<Pair<String, String>> taskQueue;
  @Autowired
  private ApplicationContext applicationContext;
  @Autowired
  private RedisMutexService redisMutexService;
  @Autowired
  @Qualifier("platform_dynamicTaskLoadingThreadScheduler")
  private ThreadPoolTaskScheduler dynamicTaskLoadingThreadScheduler;
  @Autowired
  private DynamicTaskProperties dynamicTaskProperties;
  /**
   * 表示当前保持任务（线程）负责保持的applicationName_appCode信息
   */
  private Map<String , ScheduledFuture<?>> preemptedKeyMapping = Maps.newHashMap();
  /**
   * 日志
   */
  private static final Logger LOGGER = LoggerFactory.getLogger(DynamicMasterKeeperTask.class);
  
  public DynamicMasterKeeperTask(BlockingQueue<Pair<String, String>> taskQueue) {
    this.taskQueue = taskQueue;
  }

  @Override
  public void run() {
    /*
     * 处理方式为：
     * 1、首先监控通过taskQueue传达过来的保持任务；
     * 2、收到信息后，进行redis的独占权操作，如果操作成功，则激活一个周期性刷新任务DynamicLoadingTask
     * ————通过platform_dynamicTaskLoadingThreadScheduler调度线程池 
     * 
     * 注意：无论抢占是否成功，整个处理过程都不会结束，而是回到步骤1进行循环执行
     * 另外，一旦进程退出，则该保持任务负责的调度任务需要结束
     * */
    try {
      int count = 0;
      while(count++ < Integer.MAX_VALUE) {
        // 1、======
        Pair<String,String> info = this.taskQueue.take();
        String applicationName = info.getLeft();
        String appCode = info.getRight();
        String key = StringUtils.join(applicationName , "_" , appCode);
        // 如果当前没有管理这个applicationName_appCode的独占操作权，才进行后续处理
        if(this.preemptedKeyMapping.get(key) != null) {
          continue;
        }
        
        // 2、======
        String dynamicTaskLockKey = StringUtils.join(key , "_" , dynamicTaskProperties.getDynamicTaskLockKey());
        // 以下逻辑只能在获取了独占操作权才能进行，否则就忽略
        if(this.redisMutexService.tryLock(dynamicTaskLockKey, TimeUnit.SECONDS, 10)) {
          // 创建任务(每30秒运行一次)
          DynamicLoadingSchedulerTask dynamicLoadingSchedulerTask = applicationContext.getBean(DynamicLoadingSchedulerTask.class, appCode , applicationName);
          // 每30秒执行一次
          CronTrigger cronTrigger = new CronTrigger("0/30 * * * * ? ");
          ScheduledFuture<?> scheduledFuture = this.dynamicTaskLoadingThreadScheduler.schedule(dynamicLoadingSchedulerTask, cronTrigger);
          this.preemptedKeyMapping.put(key, scheduledFuture);
        } else {
          LOGGER.warn("当前应用并没有获取到指定appCode[{}]的动态任务独占操作权，不能进行这部分动态任务的周期性状态扫描" , appCode);
        }
      }
    } catch (InterruptedException e) {
      LOGGER.error(e.getMessage() , e);
      LOGGER.warn("==== 准备停止DynamicTaskMasterKeeperTask任务");
      return;
    } finally {
      Collection<ScheduledFuture<?>> scheduledFutures = preemptedKeyMapping.values();
      if(!CollectionUtils.isEmpty(scheduledFutures)) {
        for (ScheduledFuture<?> scheduledFuture : scheduledFutures) {
          scheduledFuture.cancel(true);
        }
      }
      // 视图取消锁资源，但实际上没有取消成功也无所谓
      Set<String> keys = preemptedKeyMapping.keySet();
      for (String key : keys) {
        String lockKey = StringUtils.join(key , "_" , dynamicTaskProperties.getDynamicTaskLockKey());
        this.redisMutexService.unlock(lockKey);
      }
    }
  }
}