package com.bizunited.nebula.europa.database.sdk.context.matedata;

import java.io.Serializable;
import java.util.Collection;
import java.util.Comparator;
import java.util.TreeMap;

import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.util.CollectionUtils;

import com.bizunited.nebula.europa.sdk.context.matedata.AbstractMetaData;
import com.google.common.collect.Maps;

/**
 * 针对数据库来源的数据视图，最终预执行解析出来的元数据描述。
 * 他在默认的AbstractMetaData元数据的描述基础上，增加了对SQL中参数名绑定情况的描述
 * @author yinwenjie
 *
 */
public class DatabaseMetaData extends AbstractMetaData {
  /**
   * 
   */
  private static final long serialVersionUID = -5375454613225491481L;
  /**
   * 该属性描述了SQL中由设置者依次书写的参数信息(这些信息都是标示为必须传入的查询条件)，类似“:xxxxx”</br>
   * Key:参数名
   * Value:参数的数据库类型
   */
  private TreeMap<Integer , DatabaseMetaDataParameter> parameterMappings = Maps.newTreeMap(new DatabaseQueryMetaDataComparator());
  
  private static class DatabaseQueryMetaDataComparator implements Serializable , Comparator<Integer> {
    private static final long serialVersionUID = 67331616298107342L;

    @Override
    public int compare(Integer o1, Integer o2) {
      return o1 = o2;
    }
  }
  
  public DatabaseMetaData(String sourceType) {
    super(sourceType);
  }
  
  /**
   * 该方法可以添加一个参数信息，不过添加过程有以下限制：</p>
   * 
   * 1、index必须大于0</br>
   * 2、同一个index位置只能添加一次（多次会报错）</br>
   * 3、同一个parameterName名字可以添加多次，但是每一次添加时，其parameterClassName信息必须一致</p>
   * 
   * @param index 索引位只能从1开始
   * @param parameterName 参数名称
   * @param parameterClassName 参数的java类型
   */
  public void addParameter(int index , String parameterName , String parameterClassName) {
    Validate.isTrue(index >= 1 , "查询条件的索引位必须大于等于1");
    Validate.notBlank(parameterName , "查询条件的名称，必须传入");
    Validate.notBlank(parameterClassName , "查询条件的对应类型，必须传入");
    
    // 一个索引位只能添加一次
    Validate.isTrue(parameterMappings.get(index) == null, "一个索引位只能添加一次");
    // 同一个parameterName名字可以添加多次，但是每一次添加时，其parameterClassName信息必须一致
    Collection<DatabaseMetaDataParameter> parameters = parameterMappings.values();
    if(!CollectionUtils.isEmpty(parameters)) {
      for (DatabaseMetaDataParameter item : parameters) {
        String itemName = item.getParameterName();
        String itemClassName = item.getParameterClassName();
        if(StringUtils.equals(itemName, parameterName)) {
          Validate.isTrue(StringUtils.equals(itemClassName, parameterClassName) , "同一个parameterName名字可以添加多次，但是每一次添加时，其parameterClassName信息必须一致");
        }
      }
    }
    
    // 进行添加
    DatabaseMetaDataParameter currentQueryParameter = new DatabaseMetaDataParameter();
    currentQueryParameter.setIndex(index);
    currentQueryParameter.setParameterClassName(parameterClassName);
    currentQueryParameter.setParameterName(parameterName);
    parameterMappings.put(index , currentQueryParameter);
  }
  
  public Collection<DatabaseMetaDataParameter> getParameters() {
    return parameterMappings.values();
  }
}
