package com.bizunited.nebula.europa.sdk.service.strategy;

import java.util.Set;

import javax.servlet.http.HttpServletRequest;

import org.springframework.data.domain.Pageable;

import com.bizunited.nebula.europa.sdk.context.execute.ExecuteContent;
import com.bizunited.nebula.europa.sdk.context.execute.ExecuteParameter;
import com.bizunited.nebula.europa.sdk.context.execute.OrderParameter;
import com.bizunited.nebula.europa.sdk.context.execute.RequestParameter;
import com.bizunited.nebula.europa.sdk.context.matedata.MetaData;
import com.bizunited.nebula.europa.sdk.vo.EuropaInfoVo;

/**
 * 执行策略，针对不同的数据视图源头，执行策略一定是不一样的；</br>
 * 而且同一种数据视图源头，根据源头条件，可能使用的执行策略也不一样
 * @author yinwenjie
 */
public interface ExecutionStrategy { 
  /**
   * 该方法表示这个执行策略是不是一种默认的执行策略
   * @return
   */
  public default boolean isDefault() {
    return false;
  }
  
  /**
   * 该方法判定本次执行过程是否被特定的执行策略所支持；
   * @param europaInfoVo 欧罗巴查询视图的基本信息
   * @return 如果支持，则返回true；其它情况返回false
   */
  public boolean validate(EuropaInfoVo europaInfoVo);
  /**
   * 该方法用于根据request信息构建具体的数据视图执行过程所需要的参数结构
   * @param europaInfoVo 欧罗巴查询视图的基本信息
   * @param request 本次http基本信息
   * @param pageable 可能的分页信息（要让数据视图的执行过程支持分页，处理数据视图自身要设置成支持分页外，业务信息也必须传入）
   * @return
   */
  public RequestParameter buildRequestParameter(EuropaInfoVo europaInfoVo , HttpServletRequest request , Pageable pageable);
  /**
   * 数据视图正式执行的第二步，构建元数据结构（数据视图的元数据结构在正式查询的时候，也需要被返回）
   * @param europaInfoVo 欧罗巴查询视图的基本信息
   * @return 预执行后的结果将封装成一个QueryMetaData查询元数据进行返回，这个查询元数据将被送入到可能存在的多个拦截器中进行后续执行
   */
  public MetaData buildQueryMetaData(EuropaInfoVo europaInfoVo , RequestParameter requestParameter);
  /**
   * 该方法，可以帮助该策略在正式执行数据视图时，获取原始的SQL信息
   * @param europaInfoVo
   * @return
   */
  public String sourceSql(EuropaInfoVo europaInfoVo);
  /**
   * 数据视图正式执行的第三步，构建执行时查询条件。</br>
   * 以基于数据库查询实现的数据视图执行过程为例，其执行时查询条件主要基于以下方面进行构建：</p>
   * 
   * 1、基于DatabaseExecuteParameter进行构建</br>
   * 2、基于执行拦截器进行构建</br>
   * 
   * @param metaData 在数据视图正式执行的第二步得到的元数据信息
   * @param requestParameter 在在数据视图正式执行的第一步得到的外部传参信息
   * @return
   */
  public Set<ExecuteParameter> buildExecuteParameters(MetaData metaData , EuropaInfoVo europaInfoVo , RequestParameter requestParameter);
  /**
   * 数据视图正式执行的第四步，构建执行时的排序条件。</br>
   * 以基于数据库查询实现的数据视图执行过程为例，其执行时排序条件主要基于DatabaseOrderParameter进行构建
   * 
   * @param metaData
   * @param europaInfoVo
   * @param requestParameter
   * @return
   */
  public Set<OrderParameter> buildOrderParameters(MetaData metaData , EuropaInfoVo europaInfoVo , RequestParameter requestParameter);
  /**
   * 数据视图正式的第五步，正式查询指定数据视图的内容信息。</br>
   * 特别注意，由于不同的应用系统可能采用了不同的分页技术（也可能没有采用任何分页信息），所以构建的返回内容，需要重新统一返回的分页信息
   * @param metaData 在数据视图正式执行的第二步得到的元数据信息
   * @param europaInfoVo
   * @param executeParameters 已经构建好的查询条件
   * @param orderParameters 已经构建好的排序条件
   * @param sqlType sql类型
   * @return 
   */
  public ExecuteContent buildExecuteContent(MetaData metaData , EuropaInfoVo europaInfoVo , RequestParameter requestParameter , Set<ExecuteParameter> executeParameters , Set<OrderParameter> orderParameters , Integer sqlType);
} 
