package com.bizunited.nebula.europa.sdk.service.strategy;

import org.springframework.core.Ordered;

import com.bizunited.nebula.europa.sdk.vo.EuropaInfoVo;
import com.bizunited.nebula.europa.sdk.vo.PersonalizationTemplateVo;

/**
 * 该接口是数据视图个性化展示模板使用的策略接口。</br>
 * 如果数据视图上层的某个模块（例如用户模块、角色模块或者岗位、组织模块等）
 * 
 * 注意实现的Ordered接口
 * 
 * 优先级越高，数字越小；</p>
 * 例如A角色模块，要求只要具有角色X的操作者，都可以使用V1的数据视图展示效果；
 * B岗位模块，又要求所有具有Y岗位的操作者，都可以使用V2的数据视图展示效果；</p>
 * 
 * 那么当一个操作既具有角色X又具有岗位Y时，怎么显示数据视图效果呢？这就要看优先级了，如果A模块的优先级要低于B模块的优先级（也就是A模块的sort方法返回的数字要大于B模块），
 * 那么就以B模块设定的效果为准。优先级约低的策略，也越先被执行。
 * 
 * @author yinwenjie
 */
public interface PersonalizationTemplateStrategy extends Ordered {
  /**
   * 获得当前策略的名字，例如：“forRole”
   * @return
   */
  public String getName();
  /**
   * 获得当前策略在页面上展示的名字，例如：“按照角色设定样式”
   * @return
   */
  public String getDescription();
  /**
   * 当个性化展示信息被创建，修改，且满足以下场景时，该方法会被触发：</br>
   * 1、在个性化展示信息被创建，且在key == getName()方法返回的值时，该方法会被触发</br>
   * 2、在个性化展示信息被修改，且在没有修改前的personalizationTemplate信息中，并不存在该key对应的扩展维度，该方法会被触发</br>
   * 实现该PersonalizationTemplateStrategy接口的上层业务模块可以在该方法中进行扩展维度信息的保存
   * @param europaInfoVo 当前正在被创建的个性化视图所关联的欧罗巴数据视图基本信息
   * @param personalizationTemplateVo 当前正在被创建的个性化视图信息
   * @param key 匹配到的扩展维度名称信息和值
   */
  public void onMappingCreate(EuropaInfoVo europaInfoVo , PersonalizationTemplateVo personalizationTemplateVo, String key , String value);
  /**
   * 以下种场景会触发该方法：</br>
   * 1、在个性化展示信息被修改，且个性化视图设定信息在修改前和修改后都包括key所代的扩展维度，则该方法被触发
   * 实现该PersonalizationTemplateStrategy接口的上层业务模块可以在该方法中进行扩展维度信息的修改
   * @param europaInfoVo 当前正在被创建的个性化视图所关联的欧罗巴数据视图基本信息
   * @param personalizationTemplateVo 这个是正在被创建的个性化展示信息（修改后的）
   */
  public void onMappingModify(EuropaInfoVo europaInfoVo , PersonalizationTemplateVo personalizationTemplateVo, String key , String value);
  /**
   * 以下种场景会触发该方法：</br>
   * 1、在个性化展示信息被修改，且个性化视图设定信息在修改前包括但在修改后不包括key所代的扩展维度，则该方法被触发
   * @param europaInfoVo 当前正在被创建的个性化视图所关联的欧罗巴数据视图基本信息
   * @param personalizationTemplateVo 这个是正在被创建的个性化展示信息（修改后的）
   * @param key 匹配到的扩展维度名称信息
   * @param oldvValue 匹配到的扩展维度名称的原有值
   */
  public void onMappingDelete(EuropaInfoVo europaInfoVo , PersonalizationTemplateVo personalizationTemplateVo, String key , String oldvValue);
  /**
   * 该方法在数据视图个性化设定功能，需要上层业务模块返回当前给定的用户在给定的模块中，是否有匹配的个性化设定信息时，被触发
   * @param personalizationTemplateVo 
   * @param key 既是这个策略实现中getName方法返回的值
   * @return 如果业务模块没有查询到相关数据视图的个性化设定样式信息，则返回null；其它情况返回value。
   * 至于value所代表的意义，由业务模块本身控制即可。
   */
  public String onRequestByAccount(PersonalizationTemplateVo personalizationTemplateVo , String tenantCode , String account , String key);
  /**
   * 该方法在数据视图个性化设定功能，需要上层业务模块返回当前给定的数据视图、个性化设定信息编号，是不是有设定的维度信息。<p/>
   * 例如 指定数据视图A，个性化样式设定编号X，是不是在角色维度上有设定值。如果有，则返回这个设定值，没有则返回null
   * @param personalizationTemplateVo 
   * @param key 既是这个策略实现中getName方法返回的值
   * @return 如果业务模块没有查询到相关数据视图的个性化设定样式信息，则返回null；其它情况返回value。
   * 至于value所代表的意义，由业务模块本身控制即可。
   */
  public String onRequest(PersonalizationTemplateVo personalizationTemplateVo , String tenantCode , String key);
}
