package com.bizunited.nebula.icon.local.service.internal;

import com.bizunited.nebula.common.service.CodeGeneratorService;
import com.bizunited.nebula.icon.local.entity.IconEntity;
import com.bizunited.nebula.icon.local.repository.IconRepository;
import com.bizunited.nebula.icon.local.service.IconService;
import com.bizunited.nebula.common.service.NebulaToolkitService;
import com.bizunited.nebula.icon.sdk.constant.IconConstant;
import com.bizunited.nebula.icon.sdk.constant.IconRedisKeyConstant;
import com.bizunited.nebula.icon.sdk.dto.IconDto;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.security.core.Authentication;
import org.springframework.security.core.context.SecurityContextHolder;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.CollectionUtils;

import java.util.Date;
import java.util.HashSet;
import java.util.LinkedList;
import java.util.List;
import java.util.Optional;


/**
 * @author hefan
 */
@Service
public class IconServiceImpl implements IconService {

  @Autowired
  @Qualifier("CodeGeneratorService")
  private CodeGeneratorService codeGeneratorService;

  @Autowired
  private IconRepository iconRepository;

  @Autowired
  private NebulaToolkitService nebulaToolkitService;

  @Transactional
  @Override
  public void createByIconDto(IconDto iconDto) {
    if (StringUtils.isEmpty(iconDto.getIconCode())) {
      String code = codeGeneratorService.generate(String.format(IconRedisKeyConstant.ICON_CODE_INDEX), IconConstant.CODE_PREFIX, null, 1, 6);
      iconDto.setIconCode(code);
    } else {
      Integer count = this.iconRepository.countByIconCode(iconDto.getIconCode());
      Validate.isTrue(count == null || count < 1, "图标编码已存在");
    }
    IconEntity mdmIconEntity = this.nebulaToolkitService.copyObjectByBlankList(iconDto, IconEntity.class, HashSet.class, LinkedList.class);
    mdmIconEntity.setCreateAccount(this.findCurrentAccount());
    mdmIconEntity.setCreateTime(new Date());
    mdmIconEntity.setModifyAccount(this.findCurrentAccount());
    this.iconRepository.saveAndFlush(mdmIconEntity);
  }

  @Transactional
  @Override
  public void updateByIconDto(IconDto iconDto) {
    Validate.notBlank(iconDto.getId(), "数据主键不能为空");
    Validate.notBlank(iconDto.getIconCode(), "图标编码不能为空");
    // 根据 ID 统计
    Optional<IconEntity> iconOpt = this.iconRepository.findById(iconDto.getId());
    Validate.isTrue(iconOpt.isPresent(), "图标不存在");
    // 根据iconCode 查询 Id
    List<IconEntity> list = this.iconRepository.findIdByIconCode(iconDto.getIconCode());
    long codeCount = list.stream().filter(x -> !iconDto.getId().equals(x.getId())).count();
    Validate.isTrue(codeCount < 1, "图标编码重复");
    IconEntity entity = this.nebulaToolkitService.copyObjectByBlankList(iconDto, IconEntity.class, HashSet.class, LinkedList.class);
    entity.setModifyTime(new Date());
    entity.setModifyAccount(this.findCurrentAccount());
    IconEntity iconDB = iconOpt.get();
    entity.setCreateAccount(iconDB.getCreateAccount());
    entity.setCreateTime(iconDB.getCreateTime());
    this.iconRepository.saveAndFlush(entity);
    // fixme: 原逻辑：清除页面配置缓存
  }

  @Transactional
  @Override
  public void deleteByIdIn(List<String> ids) {
    Validate.isTrue(!CollectionUtils.isEmpty(ids), "id集合不能为空");
    this.iconRepository.deleteByIdIn(ids);
    // fixme: 原逻辑：清除页面配置缓存

  }

  private String findCurrentAccount() {
    // 可能的操作者信息(如果没有就写入admin)
    Authentication authentication = SecurityContextHolder.getContext().getAuthentication();
    String account = "admin";
    if (authentication != null) {
      account = authentication.getName();
    }
    return account;
  }
}
