package com.bizunited.nebula.mars.sdk.converter;

import java.sql.Connection;
import java.util.Map;
import java.util.Set;
import java.util.Map.Entry;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.TimeUnit;

import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.util.CollectionUtils;

import com.bizunited.nebula.common.util.tenant.TenantUtils;
import com.google.common.cache.CacheBuilder;
import com.google.common.cache.CacheLoader;
import com.google.common.cache.LoadingCache;
import com.google.common.collect.Maps;
import com.google.common.collect.Sets;

public abstract class AbstractMarsAuthoritySqlConverter implements MarsAuthoritySqlConverter {
  
  /**
   * 日志
   */
  private static final Logger LOGGER = LoggerFactory.getLogger(AbstractMarsAuthoritySqlConverter.class);
  
  /**
   * 针对不同顶级租户，都有一组数据层的数据表记录缓存
   */
  private static LoadingCache<String, Set<RepositoryTableNode>> appCodeTableNodes;
  static {
    appCodeTableNodes = CacheBuilder.newBuilder()
    .maximumSize(100000)
    .expireAfterWrite(1, TimeUnit.MINUTES)
    .build(
      new CacheLoader<String, Set<RepositoryTableNode>>() {
        @Override
        public Set<RepositoryTableNode> load(String key) throws Exception {
          return Sets.newConcurrentHashSet();
        }
      }
    );
  }
  
  @Override
  public Map<String, String> analysisTableFileds(Connection connection, String tableName, Map<String, Set<String>> requestMatchedFields) {
    String appCode = TenantUtils.getTenantCode();
    
    /*
     * 处理方式为：
     * 1、试图从缓存中取得数据表
     * 2、如果没有取得则试图从数据层进行查询（并在成功后存放到缓存中）
     * 3、以分组为单位，一个一个的匹配
     * 
     * 注意：虽然调用者要求本数据表tableName检查所有的分组匹配选项，
     * 但只要本数据表tableName检查到匹配任何一个分组的字段要求，则都会返回Map信息
     * */
    // 1、======
    if(connection == null || StringUtils.isBlank(tableName) || CollectionUtils.isEmpty(requestMatchedFields)) {
      return null;
    }
    RepositoryTableNode tableNode = this.findCacheTableNode(appCode, tableName);
    
    // 2、=====
    if(tableNode == null) {
      tableNode = this.findRepositoryTableNode(connection, tableName);
      if(tableNode != null) {
        this.addCacheTableNode(appCode, tableNode);
      } else {
        return null;
      }
    }
    
    // 3、=====
    Map<String , String> matchedDimensionMapping = Maps.newLinkedHashMap();
    Set<String> fieldNames = tableNode.getFields();
    MA: for (String filedName : fieldNames) {
      for (Entry<String , Set<String>> requestMatchedFieldEntry : requestMatchedFields.entrySet()) {
        String dimensionKey = requestMatchedFieldEntry.getKey();
        Set<String> possibleFieldNames = requestMatchedFieldEntry.getValue();
        /*
         * 如果条件成立，说明字段匹配成功，进行返回信息构造
         * 新版本中，由于陪陪字段可以指定特定的数据表名，所以判定过程为：
         * 1、如果当前判定的字段possibleFieldName存在{数据表名}.{字段名}这样的格式，
         * 则tableName.filedName 要匹配possibleFieldName
         * 2、如果当前判定的字段possibleFieldName只存在字段名
         * 则进行filedName的匹配即可
         * */
        for (String possibleFieldName : possibleFieldNames) {
          if(StringUtils.indexOf(possibleFieldName, ".") != -1 
              && StringUtils.equalsIgnoreCase(tableName + "." + filedName, possibleFieldName)) {
            matchedDimensionMapping.put(dimensionKey, possibleFieldName);
            continue MA;
          } else if(StringUtils.indexOf(possibleFieldName, ".") == -1
              && StringUtils.equalsIgnoreCase(filedName, possibleFieldName)) {
            matchedDimensionMapping.put(dimensionKey, possibleFieldName);
            continue MA;
          }
        }
      }
    }
    if(CollectionUtils.isEmpty(matchedDimensionMapping)) {
      return null;
    }
    return matchedDimensionMapping;
  }
  
  /**
   * 该方法负责从缓存中，查询指定的数据表字段结构信息
   * @param appCode 
   * @param tableName 
   * @return 
   */
  private RepositoryTableNode findCacheTableNode(String appCode , String tableName) {
    Set<RepositoryTableNode> repositoryTableNodes = null;
    try {
      repositoryTableNodes = appCodeTableNodes.get(appCode);
    } catch(ExecutionException e) {
      LOGGER.error(e.getMessage() , e);
    }
    
    RepositoryTableNode currentTableNode = null;
    for (RepositoryTableNode repositoryTableNode : repositoryTableNodes) {
      if(StringUtils.equals(repositoryTableNode.getTableName(), tableName)) {
        currentTableNode = repositoryTableNode;
        break;
      }
    }
    return currentTableNode;
  }
  
  /**
   * 该方法负责想缓存中添加一个数据表字段结构信息
   * @param appCode
   * @param tableNode
   */
  private void addCacheTableNode(String appCode , RepositoryTableNode tableNode) {
    Set<RepositoryTableNode> repositoryTableNodes = null;
    try {
      repositoryTableNodes = appCodeTableNodes.get(appCode);
    } catch(ExecutionException e) {
      LOGGER.error(e.getMessage() , e);
    }
    repositoryTableNodes.add(tableNode);
  }
  
  /**
   * 该方法从数据层查询数据表结构
   * @param connection
   * @param tableName
   * @return
   */
  protected abstract RepositoryTableNode findRepositoryTableNode(Connection connection , String tableName);
  
  /**
   * 在mars的数据库字段分析阶段，进行数据表中字段信息记录的类型
   * @author Administrator
   *
   */
  protected static class RepositoryTableNode {
    /**
     * 表名
     */
    private String tableName;
    /**
     * 拥有的字段名（注意，大小写不明感）
     */
    private Set<String> fields = Sets.newHashSet();
    public RepositoryTableNode(String tableName, Set<String> fields) {
      this.tableName = tableName;
      this.fields = fields;
    }
    public String getTableName() {
      return tableName;
    }
    public Set<String> getFields() {
      return fields;
    }
  }
}
