package com.bizunited.nebula.mars.sdk.converter;

import java.util.Arrays;
import java.util.List;

import org.apache.commons.lang3.StringUtils;
import org.springframework.stereotype.Component;

import com.alibaba.druid.sql.ast.SQLExpr;
import com.alibaba.druid.sql.ast.expr.SQLCharExpr;
import com.alibaba.druid.sql.ast.expr.SQLInListExpr;
import com.alibaba.druid.sql.ast.expr.SQLPropertyExpr;
import com.alibaba.fastjson.JSONArray;
import com.bizunited.nebula.common.enums.DatabaseType;
import com.bizunited.nebula.mars.sdk.converter.MarsAuthorityAstConverter;
import com.google.common.collect.Lists;

/**
 * 该方法服务于字符/字符串数组（包括）判定条件构造，构造后的结构为：tt.a in ('xxxx','yyyy','zzzz')
 * @author yinwenjie
 *
 */
@Component
public class ChartArrayMarsAuthorityAstConverter implements MarsAuthorityAstConverter {

  /**
   * 该转换器对应的转换器标识
   */
  public static final String CONVERTER_KEY = "chartArrayMarsAuthorityAstConverter";
  
  @Override
  public String converterKey() {
    return CONVERTER_KEY;
  }
  
  /**
   * 只要是数组，且只要是可串行化的字符类型都支持，例如String、Stringbuffer、StringBuild等等
   */
  @Override
  public boolean support(Class<?> sourceClass, boolean isArrayValue) {
    if(!isArrayValue) {
      return false;
    }
    if(!CharSequence.class.isAssignableFrom(sourceClass)) {
      return false;
    }
    return true;
  }

  /**
   * 转换值的AST-Node的时候，将类型转换为SQLCharExpr的集合
   */
  @Override
  public Object converter(DatabaseType databaseType, Class<?> sourceClass, boolean isArrayValue, Object sourceValue) {
    // 注意，由于HTTP请求的原因，这里可能是一个JSONArray
    String[] sourceValues = null;
    if(sourceValue instanceof JSONArray) {
      sourceValues = ((JSONArray)sourceValue).toArray(new String[] {});
    } else {
      Object[] sourceValueObjects = (Object[])sourceValue;
      sourceValues = Arrays.stream(sourceValueObjects).map(Object::toString).toArray(String[]::new);
    }
    // 开始一个一个转成SQLCharExpr
    List<SQLExpr> sqlCharExprs = Lists.newArrayList();
    for (String sourceValueItem : sourceValues) {
      SQLCharExpr sqlCharExpr = new SQLCharExpr(sourceValueItem);
      sqlCharExprs.add(sqlCharExpr);
    }
    return sqlCharExprs;
  }
  
  @SuppressWarnings("unchecked")
  @Override
  public SQLExpr op(String leftTableName , String leftAlias , String leftFieldName , Object right) {
    /*
     * 使用SQLInListExpr节点进行构造
     * */
    SQLPropertyExpr property = null;
    if(!StringUtils.isBlank(leftAlias)) {
      property = new SQLPropertyExpr(leftAlias, leftFieldName);
    } else {
      property = new SQLPropertyExpr(leftTableName, leftFieldName);
    }
    SQLInListExpr op = new SQLInListExpr(property);
    op.setTargetList((List<SQLExpr>)right);
    return op;
  }
}
