package com.bizunited.nebula.rbac.local.controller;

import com.bizunited.nebula.common.controller.BaseController;
import com.bizunited.nebula.common.controller.model.ResponseModel;
import com.bizunited.nebula.rbac.sdk.service.ButtonVoService;
import com.bizunited.nebula.rbac.sdk.vo.ButtonVo;
import com.google.common.collect.Lists;

import io.swagger.annotations.ApiOperation;
import io.swagger.annotations.ApiParam;

import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.security.core.context.SecurityContext;
import org.springframework.security.core.context.SecurityContextHolder;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.RestController;

import java.util.Set;

/**
 * 按钮的管理类，负责提供相关接口
 * @author: yinwenjie
 * @version 2.0
 * @date: 2020/1/13 14:37
 */
@RestController
@RequestMapping(value="/v1/rbac/buttons")
public class ButtonVoController extends BaseController {

  /** 日志 */
  private static final Logger LOGGER = LoggerFactory.getLogger(ButtonVoController.class);

  @Autowired
  private ButtonVoService buttonService;
  
  /**
   * 根据按钮id查询按钮信息
   * @param buttonId
   * @return
   */
  @ApiOperation(value = "根据按钮id查询按钮信息(包括关联信息)")
  @GetMapping("findById")
  public ResponseModel findById(@RequestParam(name = "buttonId") @ApiParam(name = "buttonId", value = "指定的按钮的编号信息") String buttonId ){
    try {
      ButtonVo buttonVo = this.buttonService.findById(buttonId);
      return this.buildHttpResult(buttonVo);
    } catch(Exception e) {
      LOGGER.error(e.getMessage(), e);
      return this.buildHttpResultForException(e);
    }
  }
  
  /**
   * 按照按钮的业务编号进行查询
   */
  @ApiOperation(value = "按照按钮的业务编号进行查询")
  @GetMapping("findByCode")
  public ResponseModel findByCode(@RequestParam(name = "code") @ApiParam(name = "code", value = "指定的按钮的业务编号") String code ){
    try {
      ButtonVo buttonVo = this.buttonService.findByCode(code);
      return this.buildHttpResult(buttonVo);
    } catch(Exception e) {
      LOGGER.error(e.getMessage(), e);
      return this.buildHttpResultForException(e);
    }
  }

  /**
   * 查询当前登录用户的按钮权限，如果当前登录用户有忽略权限的按钮，那就返回所有的权限按钮
   * @param
   * @return
   */
  @GetMapping("findByCurrentUser")
  @ApiOperation(value = "查询当前登录用户的按钮权限", notes = "查询当前登录用户的按钮权限,如果当前登录用户有忽略权限的按钮，那就返回所有的权限按钮")
  public ResponseModel findByCurrentUser(){
    try {
      SecurityContext securityContext = SecurityContextHolder.getContext();
      String account = null;
      if(securityContext != null && securityContext.getAuthentication() != null 
          && StringUtils.isNotBlank(account = securityContext.getAuthentication().getName())) {
        Set<ButtonVo> buttonVos = this.buttonService.findByAcconut(account);
        return this.buildHttpResult(buttonVos);
      } else {
        return buildHttpResult();
      }
    } catch(Exception e) {
      LOGGER.error(e.getMessage(), e);
      return this.buildHttpResultForException(e);
    }
  }
  
  /**
   * 根据菜单ID查询按钮
   * @param competenceId
   * @return
   */
  @GetMapping("findByTopCompetenceId")
  @ApiOperation(value = "根据菜单ID查询按钮", notes = "根据菜单ID查询按钮")
  public ResponseModel findByTopCompetenceId(@ApiParam(name = "competenceId", value = "按钮ID", required = true) @RequestParam("topCompetenceId") String topCompetenceId){
    try {
      Set<ButtonVo> result = this.buttonService.findByTopCompetenceId(topCompetenceId);
      return buildHttpResult(result);
    } catch (RuntimeException e) {
      LOGGER.error(e.getMessage(), e);
      return buildHttpResultForException(e);
    }
  }
  
  /**
   * 按照按钮从属的角色业务编号和功能技术编号，查询按钮信息
   */
  @GetMapping("findByRoleCodeAndCompetenceId")
  @ApiOperation(value = "按照按钮从属的角色业务编号和功能技术编号，查询按钮信息")
  public ResponseModel findByRoleCodeAndCompetenceId(@ApiParam(name = "roleCode", value = "角色业务编号") @RequestParam("roleCode") String roleCode 
                                                     ,@ApiParam(name = "topCompetenceId", value = "菜单的业务编号") @RequestParam("topCompetenceId")  String topCompetenceId) {
    try {
      Set<ButtonVo> result = this.buttonService.findByRoleCodesAndTopCompetenceId(Lists.newArrayList(roleCode), topCompetenceId);
      return buildHttpResult(result);
    }catch (RuntimeException e){
      LOGGER.error(e.getMessage(), e);
      return buildHttpResultForException(e);
    }
  }
}
