package com.bizunited.nebula.rbac.local.controller;

import com.alibaba.fastjson.JSONArray;
import com.bizunited.nebula.common.controller.BaseController;
import com.bizunited.nebula.common.controller.model.ResponseModel;
import com.bizunited.nebula.common.util.tenant.TenantUtils;
import com.bizunited.nebula.rbac.sdk.config.RbacCustomProperties;
import com.bizunited.nebula.rbac.sdk.service.RoleVoCacheService;
import com.bizunited.nebula.rbac.sdk.vo.RoleVo;
import com.google.common.collect.Sets;
import com.google.common.collect.Sets.SetView;
import io.swagger.annotations.ApiOperation;
import io.swagger.annotations.ApiParam;
import org.apache.commons.lang3.Validate;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.RestController;

import java.security.Principal;
import java.util.Set;

@RestController
@RequestMapping("/v1/rbac/roles")
public class RoleVoController extends BaseController {

  @Autowired
  private RbacCustomProperties rbacCustomProperties;
  @Autowired
  private RoleVoCacheService roleVoCacheService;
  
  /**
   * 日志
   */
  private static final Logger LOGGER = LoggerFactory.getLogger(RoleVoController.class);

  /**
   * 获取指定用户已绑定的角色信息
   */
  @ApiOperation(value = "获取指定用户已拥有的角色信息，包括自己绑定的，所属组织机构的，所属用户组的，甚至第三方系统的，等等",
          notes = "获取指定用户已拥有的角色信息，包括自己绑定的，所属组织机构的，所属用户组的，甚至第三方系统的，等等," +
                  "因为有主副岗位之分，在查询用户的角色用于权限验证时，岗位绑定的角色只会查询主岗位绑定的角色，用户角色维护时，则返回用户所有岗位绑定的角色")
  @RequestMapping(value = "/findByAccount", method = {RequestMethod.GET})
  public ResponseModel findByAccount(@ApiParam(value = "指定的人员数据编号") @RequestParam String account) {
    try {
      String tenantCode = TenantUtils.getTenantCode();
      Set<RoleVo> roles = roleVoCacheService.findByTenantCodeAndUserAccount(tenantCode, account);
      return this.buildHttpResult(roles);
    } catch (RuntimeException e) {
      LOGGER.error(e.getMessage(), e);
      return this.buildHttpResultForException(e);
    }
  }

  /**
   * 根据当前登录用户，查询当前用户具有的角色。这些角色包括了所有来源于上层业务模块的角色绑定
   * @return
   */
  @ApiOperation(value = "根据当前登录用户，查询当前用户具有的角色。这些角色包括了所有来源于上层业务模块的角色绑定")
  @GetMapping("/findByCurrentUser")
  public ResponseModel findByCurrentUser(Principal principal) {
    try {
      String tenantCode = TenantUtils.getTenantCode();
      Set<RoleVo> roles = roleVoCacheService.findByTenantCodeAndUserAccount(tenantCode, principal.getName());
      return this.buildHttpResult(roles);
    } catch (RuntimeException e) {
      LOGGER.error(e.getMessage(), e);
      return this.buildHttpResultForException(e);
    }
  }
  
  /**
   * 根据角色编码与租户编号查询角色信息：注意只查询当前操作者所在租户的范围(请注意该方法存在于rbac模块的意义)
   * @param roleCodes
   * @return
   */
  @ApiOperation(value = "根据角色编码与租户编号查询角色信息：注意只查询当前操作者所在租户的范围")
  @GetMapping("/findByRoleCodes")
  public ResponseModel findByTenantCodeAndRoleCodes(@ApiParam(value = "可以传入的多个角色业务编号") @RequestParam("roleCodes") String[] roleCodes) {
    try {
      String tenantCode = TenantUtils.getTenantCode();
      Set<RoleVo> roles = roleVoCacheService.findByTenantCodeAndRoleCodes(tenantCode, Sets.newHashSet(roleCodes));
      return this.buildHttpResult(roles);
    } catch (RuntimeException e) {
      LOGGER.error(e.getMessage(), e);
      return this.buildHttpResultForException(e);
    }
  }

  @ApiOperation(value = "该方法给定一批角色名，系统会返回这批角色名中，是否有一个或者多个角色拥有管理员性质的角色。而判定具有超级管理员性质角色的用户会被返回",
          notes = "获取指定用户已拥有的角色信息，包括自己绑定的，所属组织机构的，所属用户组的，甚至第三方系统的，等等")
  @GetMapping("/findByIgnoreRoles")
  public ResponseModel findByIgnoreRoles(@ApiParam(value = "当前需要验证的是否具有本地超级管理员") @RequestParam(name = "roleCodes") String[] roleCodes) {
    try {
      Validate.isTrue(roleCodes != null && roleCodes.length > 0, "要进行判定的角色信息必须传入");
      SetView<String> intersections = Sets.intersection(Sets.newHashSet(rbacCustomProperties.getIgnoreMethodCheckRoles()), Sets.newHashSet(roleCodes));
      if (intersections == null || intersections.isEmpty()) {
        return this.buildHttpResult();
      }
      // 组装成json返回
      JSONArray results = new JSONArray();
      for (String item : intersections) {
        results.add(item);
      }
      return this.buildHttpResult(results);
    } catch (RuntimeException e) {
      return this.buildHttpResultForException(e);
    }
  }
  
  /**
   * 角色管理-查询角色树状关系
   * @return
   */
  @ApiOperation(value = "查询角色树状关系（树形的查询暂时不能使用，有需求后再进行完善）")
  @GetMapping(value = "/findRoleTree")
  public ResponseModel findRoleTree() {
    try {
      String tenantCode = TenantUtils.getTenantCode();
      Set<RoleVo> roles = this.roleVoCacheService.findRoleTree(tenantCode);
      return this.buildHttpResult(roles);
    } catch (RuntimeException e) {
      LOGGER.error(e.getMessage(), e);
      return this.buildHttpResultForException(e);
    }
  }
}
