package com.bizunited.nebula.rbac.local.repository;

import org.springframework.data.jpa.repository.JpaRepository;
import org.springframework.data.jpa.repository.JpaSpecificationExecutor;
import org.springframework.data.jpa.repository.Modifying;
import org.springframework.data.jpa.repository.Query;
import org.springframework.data.repository.query.Param;
import org.springframework.stereotype.Repository;

import com.bizunited.nebula.rbac.local.entity.ButtonEntity;

import java.util.List;
import java.util.Set;

/**
 * 为按钮功能的服务提供数据库方法支持
 * @author: yinwenjie
 * @version 2.0
 */
@Repository
public interface ButtonRepository
        extends
        JpaRepository<ButtonEntity, String>,
        JpaSpecificationExecutor<ButtonEntity> {

  /**
   * 建立指定按钮和指定接口的绑定关系
   * @param buttonId 指定的按钮编号
   * @param competenceId 指定的接口编号
   */
  @Modifying
  @Query(value="insert into engine_button_competence_mapping(button_id,competence_id) values (:buttonId,:competenceId)", nativeQuery=true)
  void bindCompetence(@Param("buttonId") String buttonId, @Param("competenceId") String competenceId);

  /**
   * 取消指定按钮和指定接口的绑定关系
   * @param buttonId 指定的按钮编号
   * @param competenceId 指定的接口编号
   */
  @Modifying
  @Query(value="delete from engine_button_competence_mapping where button_id = :buttonId and competence_id = :competenceId", nativeQuery=true)
  void unbindCompetence(@Param("buttonId") String buttonId, @Param("competenceId") String competenceId);

  /**
   * 根据按钮id删除competence绑定关系
   * @param buttonId
   */
  @Modifying
  @Query(value="delete from engine_button_competence_mapping where button_id = :buttonId", nativeQuery=true)
  void unbindAllByButtonId(@Param("buttonId") String buttonId);

  /**
   * 建立指定角色和指定按钮的绑定关系
   * @param roleId 指定的角色编号信息
   * @param buttonId 指定的按钮编号
   */
  @Modifying
  @Query(value="insert into engine_role_button_mapping(role_id,button_id) values (:roleId,:buttonId)", nativeQuery=true)
  void bindRole(@Param("roleId") String roleId, @Param("buttonId") String buttonId);

  /**
   * 取消指定角色和指定按钮的绑定关系
   * @param roleId 指定的角色编号信息
   * @param buttonId 指定的按钮编号
   */
  @Modifying
  @Query(value="delete from engine_role_button_mapping where role_id = :roleId and button_id = :buttonId", nativeQuery=true)
  void unbindRole(@Param("roleId") String roleId, @Param("buttonId") String buttonId);

  /**
   * 根据按钮id删除role的所有绑定关系
   * @param buttonId
   */
  @Modifying
  @Query(value="delete from engine_role_button_mapping where button_id = :buttonId", nativeQuery=true)
  void unbindAllRoleByButtonId(@Param("buttonId") String buttonId);


  /**
   * 查询指定的按钮编号和接口编号的是否有且只有一个绑定关系
   * @param buttonId 指定的按钮编号
   * @param competenceId 指定的接口编号
   * @return
   */
  @Query(value="select count(*) from engine_button_competence_mapping where button_id = :buttonId and competence_id = :competenceId", nativeQuery=true)
  long countByCompetenceIdAndButtonId(@Param("buttonId") String buttonId, @Param("competenceId") String competenceId);

  /**
   * 根据编码统计
   * @param code
   * @return
   */
  @Query("select count(b.id) from ButtonEntity b where b.code = :code")
  long countByCode(@Param("code") String code);

  /**
   * 根据编码统计按钮数量，并排除ID
   * @param code
   * @param id
   * @return
   */
  @Query("select count(b.id) from ButtonEntity b where b.code = :code and b.id <> :id")
  long countByCodeWithoutId(@Param("code") String code,  @Param("id") String id);

  /**
   * 查询指定的角色编号和按钮编号的是否有且只有一个绑定关系
   * @param roleId 指定的角色编号
   * @param buttonId 指定的按钮编号
   * @return
   */
  @Query(value="select count(*) from engine_role_button_mapping where role_id = :roleId and button_id = :buttonId", nativeQuery=true)
  long countByRoleIdAndButtonId(@Param("roleId") String roleId, @Param("buttonId") String buttonId);

  /**
   * 根据按钮id查询按钮信息（包括关联的角色和按钮信息）
   * @param id
   * @return
   */
  @Query(value = "select distinct b from ButtonEntity b "
          + " left join fetch b.topCompetence tc "
          + " left join fetch b.competences c "
          + " left join fetch b.roles r "
          + " where b.id = :id order by b.createTime desc ")
  ButtonEntity findDateilsById(@Param("id") String id);

  /**
   * 根据角色id获取按钮信息
   * @param roleId
   * @return
   */
  @Query(value = "select distinct b from ButtonEntity b left join fetch b.roles r where r.id = :roleId  ")
  Set<ButtonEntity> findByRoleId(@Param("roleId") String roleId);

  /**
   * 根据按钮业务编码查询，包括关联信息
   * @param code
   * @return
   */
  @Query(value = "select distinct b from ButtonEntity b "
          + " left join fetch b.topCompetence tc "
          + " left join fetch b.competences c "
          + " left join fetch b.roles r "
          + " where b.code = :code ")
  ButtonEntity findByCode(@Param("code") String code);

  /**
   * 根据菜单ID查询按钮(注意是菜单)
   * @param topCompetenceId
   * @return
   */
  @Query("select distinct b from ButtonEntity b "
          + " left join fetch b.topCompetence tc "
          + " left join fetch b.competences c "
          + " left join fetch b.roles r "
          + " where tc.id = :topCompetenceId order by b.createTime desc ")
  Set<ButtonEntity> findByTopCompetenceId(@Param("topCompetenceId") String topCompetenceId);

  /**
   * 根据角色code查询关联的按钮
   */
  @Query("select distinct b from ButtonEntity b "
          + " left join fetch b.topCompetence tc "
          + " left join fetch b.competences c "
          + " left join fetch b.roles r "
          + " where r.roleCode in (:roleCodes) and r.tenantCode = :tenantCode order by b.createTime desc")
  Set<ButtonEntity> findByRoleCodesAndTenantCode(@Param("roleCodes") List<String> roleCodes , @Param("tenantCode") String tenantCode);

  /**
   * 根据角色code和功能菜单查询关联的按钮
   */
  @Query("select distinct b from ButtonEntity b "
          + " left join fetch b.topCompetence tc "
          + " left join fetch b.competences c "
          + " left join fetch b.roles r "
          + " where r.roleCode = :roleCode and tc.id = :topCompetenceId and r.tenantCode = :tenantCode "
          + " order by b.createTime desc ")
  Set<ButtonEntity> findByRoleCodeAndTopCompetenceIdAndTenantCode(@Param("roleCode") String roleCode,@Param("topCompetenceId") String topCompetenceId , @Param("tenantCode") String tenantCode);


}
