package com.bizunited.nebula.rbac.local.repository;

import org.springframework.data.jpa.repository.JpaRepository;
import org.springframework.data.jpa.repository.JpaSpecificationExecutor;
import org.springframework.data.jpa.repository.Modifying;
import org.springframework.data.jpa.repository.Query;
import org.springframework.data.repository.query.Param;
import org.springframework.stereotype.Repository;

import com.bizunited.nebula.rbac.local.entity.CompetenceEntity;
import com.bizunited.nebula.rbac.local.repository.internal.CompetenceRepositoryCustom;

import java.util.List;
import java.util.Set;

/**
 * 功能的数据层服务
 * TODO 还要进行清理
 * @author: yinwenjie
 * @version 2.0
 */
@Repository
public interface CompetenceRepository
        extends
        CompetenceRepositoryCustom,
        JpaRepository<CompetenceEntity, String>,
        JpaSpecificationExecutor<CompetenceEntity> {
  /**
   * 建立指定角色和指定功能的绑定关系
   * @param roleId 指定的角色编号信息
   * @param competenceId 指定的功能编号信息
   */
  @Modifying
  @Query(value="insert into engine_role_competence_mapping(role_id,competence_id) values (:roleId,:competenceId)", nativeQuery=true)
  void bindRole(@Param("roleId") String roleId, @Param("competenceId") String competenceId);

  /**
   * 取消指定角色和指定功能的绑定关系
   * @param roleId 指定的角色编号信息
   * @param competenceId 指定的功能编号信息
   */
  @Modifying
  @Query(value="delete from engine_role_competence_mapping where role_id = :roleId and competence_id = :competenceId", nativeQuery=true)
  void unbindRole(@Param("roleId") String roleId, @Param("competenceId") String competenceId);
  /**
   * 根据功能中的viewItem属性， 查询满足条件的所有功能信息（按照sortIndex进行排序）
   * @param viewItem 是否是功能菜单上的功能栏目
   */
  @Query("from CompetenceEntity ce left join fetch ce.parent cep where ce.viewItem = :viewItem order by ce.sortIndex ")
  List<CompetenceEntity> findByViewItem(@Param("viewItem") Integer viewItem);
  /**
   * 根据功能中的viewItem属性、状态属性， 查询满足条件的所有功能信息（按照sortIndex进行排序）
   * @param viewItem 是否是功能菜单上的功能栏目
   * @param tstatusring 状态信息，1：正常；0：禁用
   */
  @Query("from CompetenceEntity ce left join fetch ce.parent cep where ce.viewItem = :viewItem and ce.tstatus = :tstatus order by ce.sortIndex")
  List<CompetenceEntity> findByViewItemAndStatus(@Param("viewItem") Integer viewItem, @Param("tstatus") Integer tstatusring);

  /**
   * 根据功能中的viewItem属性、状态属性， 查询满足条件的顶级功能信息（按照sortIndex进行排序）
   * @param viewItem
   * @param tstatus
   */
  @Query("select c from CompetenceEntity c where c.viewItem = :viewItem and c.tstatus = :tstatus and c.parent is null order by c.sortIndex asc ")
  List<CompetenceEntity> findByViewItemAndTstatusAndParentNull(@Param("viewItem") Integer viewItem, @Param("tstatus") Integer tstatus);

  /**
   * 根据功能中的viewItem属性，查询满足条件的顶级功能信息（按照sortIndex进行排序）
   * @param viewItem
   */
  @Query("select c from CompetenceEntity c where c.viewItem = :viewItem and c.parent is null order by c.sortIndex asc ")
  List<CompetenceEntity> findByViewItemAndParentNull(@Param("viewItem") Integer viewItem);

  /**
   * 查询为当前URL设置的角色信息，注意，这里没有通过method进行过滤
   * @param resource 指定的url源信息
   * @param tstatus 符合的状态信息，1：正常；0：不正常
   */
  List<CompetenceEntity> findByResourceAndTstatus(String resource, Integer tstatus);

  /**
   * 查询为当前URL设置的角色信息，注意，这里没有通过method进行过滤（模糊查询）
   * @param resource 指定的url源信息
   * @param tstatus 符合的状态信息，1：正常；0：不正常
   */
  @Query(value = "from CompetenceEntity ce where ce.resource like concat('%',:resource,'%') and ce.tstatus = :tstatus")
  List<CompetenceEntity> findByResourceAndTstatusLike(@Param("resource") String resource, @Param("tstatus") Integer tstatus);

  /**
   * 根据当前设定的完整地址和多个允许的methods信息，查询指定的功能信息
   * @param resource 指定的url源信息
   */
  List<CompetenceEntity> findByResource(String resource);

  /**
   * 根据名称查询功能信息
   * @param comment 注释
   */
  List<CompetenceEntity> findByComment(String comment);
  /**
   * 该方法按照多个角色信息的name信息，查询能够被这些角色访问的处于指定状态的功能
   * @param viewItem 是否是功能菜单上的功能栏目 1：需要显示在菜单栏上的功能；0: 不需要显示在菜单栏上的功能
   * @param roleName 指定的角色名信息
   * @param status 指定的功能状态
   */
  @Query("select distinct c from CompetenceEntity c left join fetch c.parent p left join fetch c.roles r where r.roleName in (:roleNames) and c.viewItem = :viewItem and c.tstatus = :status order by c.sortIndex")
  List<CompetenceEntity> findByViewItemAndRoleNamesAndStatus(@Param("viewItem") Integer viewItem, @Param("roleNames") String[] roleNames, @Param("status") Integer status);

  /**
   * 该方法按照多个角色信息的name信息，查询能够被这些角色访问的功能
   * @param viewItem 是否是功能菜单上的功能栏目 1：需要显示在菜单栏上的功能；0: 不需要显示在菜单栏上的功能
   * @param roleName 指定的角色名信息
   */
  @Query("select distinct c from CompetenceEntity c left join fetch c.parent p left join fetch c.roles r where r.roleName in (:roleNames) and c.viewItem = :viewItem order by c.sortIndex")
  List<CompetenceEntity> findByViewItemAndRoleNames(@Param("viewItem") Integer viewItem, @Param("roleNames") String[] roleNames);

  /**
   * 按照指定的父级功能编号，查询所有子级功能
   * @param parentId 指定的父级功能编号
   */
  @Query("select distinct c from CompetenceEntity c left join fetch c.buttons buttons where c.parent.id = :parentId order by c.sortIndex")
  List<CompetenceEntity> findByParentId(@Param("parentId") String parentId);

  /**
   * 按照指定的父级功能编号，查询所有子级功能
   * @param parentId 指定的父级功能编号
   */
  @Query("from CompetenceEntity c where c.parent.id = :parentId and c.tstatus = :tstatus order by c.sortIndex")
  List<CompetenceEntity> findByParentIdAndStatus(@Param("parentId") String parentId , @Param("tstatus") Integer tstatus);
  /**
   * 根据说明、显示、父级查询
   * @param comment 注释
   * @param viewItem
   * @param parentId 父级id
   */
  @Query("select c from CompetenceEntity c inner join c.parent cp " +
          "where c.comment = :comment " +
          "and c.viewItem = :viewItem " +
          "and cp.id = :parentId")
  CompetenceEntity findByCommentAndViewItemAndParent(@Param("comment") String comment, @Param("viewItem") int viewItem, @Param("parentId") String parentId);

  /**
   * 根据说明、显示、父级为null查询
   * @param comment 注释
   * @param viewItem
   */
  @Query("select c from CompetenceEntity c " +
          "where c.comment = :comment " +
          "and c.viewItem = :viewItem " +
          "and c.parent.id is null ")
  CompetenceEntity findByCommentAndViewItemAndNullParent(@Param("comment") String comment, @Param("viewItem") int viewItem);

  /**
   * 根据按钮id查询绑定的接口信息
   * @param buttonId 按钮id
   */
  @Query("select c from CompetenceEntity c inner join c.buttons b where b.id = :buttonId ")
  Set<CompetenceEntity> findByButtonId(@Param("buttonId") String buttonId);

  /**
   * 根据路径和请求方法查询接口
   * @param resource 资源路径
   * @param methods 方法
   */
  CompetenceEntity findByResourceAndMethods(String resource, String methods);

  /**
   * 查询功能菜单-功能备注（功能中文名）是否重复
   * @param comment 功能备注
   * @param viewItem 是否是功能菜单上的功能栏目 1：需要显示在菜单栏上的功能；0: 不需要显示在菜单栏上的功能
   */
  @Query(value = "select distinct c from  CompetenceEntity c where c.viewItem = :viewItem and c.comment = :comment")
  List<CompetenceEntity> findByCommentAndViewItem(@Param("comment") String comment, @Param("viewItem") Integer viewItem);

  /**
   * 该方法按照一个角色的id信息，查询能够被这些角色访问的功能
   * @param roleId 指定的角色主键
   */
  @Query("select distinct c from CompetenceEntity c left  join fetch c.parent left join fetch c.roles r where r.id = :roleId order by c.sortIndex")
  List<CompetenceEntity> findByRoleId(@Param("roleId") String roleId);

  /**
   * 根据角色ID查询顶级菜单
   * @param roleId 角色id
   */
  @Query("select distinct c from CompetenceEntity c " +
          " inner join c.roles r where r.id = :roleId and c.parent is null " +
          " order by c.sortIndex asc ")
  List<CompetenceEntity> findByRoleIdAndParentNot(@Param("roleId") String roleId);

  /**
   * 根据是否是菜单统计数据
   */
  @Query("select count(c.id) from CompetenceEntity c where c.viewItem = :viewItem")
  long countByViewItem(@Param("viewItem") int viewItem);

  /**
   * 根据编码查重
   * @param code 功能编码
   */
  @Query("select count(c.id) from CompetenceEntity c where c.code = :code")
  long countByCode(@Param("code") String code);

  /**
   * 查询指定的角色编号和功能编号的是否有且只有一个绑定关系
   * @param roleId 指定的角色编号
   * @param competenceId 指定的功能编号
   */
  @Query(value="select count(*) from engine_role_competence_mapping where role_id = :roleId and competence_id = :competenceId", nativeQuery=true)
  long countByRoleIdAndCompetenceId(@Param("roleId") String roleId, @Param("competenceId") String competenceId);

  /**
   * 统计角色ID关联的顶级菜单
   * @param roleId 角色id
   */
  @Query("select count(c.id) from CompetenceEntity c inner join c.roles r where r.id = :roleId and c.parent is null")
  long countByRoleIdAndParentNot(@Param("roleId") String roleId);

  /**
   * 查询功能菜单-功能备注（功能中文名）是否重复
   * @param comment 功能备注
   * @param viewItem 是否是功能菜单上的功能栏目 1：需要显示在菜单栏上的功能；0: 不需要显示在菜单栏上的功能
   */
  @Query(value = "select count(c.id) from CompetenceEntity c where c.viewItem = :viewItem and c.comment = :comment")
  long countByCommentAndViewItem(@Param("comment") String comment, @Param("viewItem") Integer viewItem);

  /**
   * 更新-查询功能菜单-功能备注（功能中文名）是否重复
   * @param id 编号
   * @param comment 功能备注
   * @param viewItem 是否是功能菜单上的功能栏目 1：需要显示在菜单栏上的功能；0: 不需要显示在菜单栏上的功能
   */
  @Query(value = "select count(c.id) from CompetenceEntity c where c.viewItem = :viewItem and c.comment = :comment and c.id <> :id ")
  long countByCommentAndViewItemAndExceptId(@Param("id") String id, @Param("comment") String comment, @Param("viewItem") Integer viewItem);

}
