package com.bizunited.nebula.rbac.local.service.internal;

import java.util.ArrayList;
import java.util.Date;
import java.util.HashSet;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Optional;
import java.util.Set;

import javax.transaction.Transactional;

import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.security.core.Authentication;
import org.springframework.security.core.context.SecurityContextHolder;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;

import com.bizunited.nebula.common.service.NebulaToolkitService;
import com.bizunited.nebula.common.util.tenant.TenantUtils;
import com.bizunited.nebula.rbac.local.entity.ButtonEntity;
import com.bizunited.nebula.rbac.local.entity.CompetenceEntity;
import com.bizunited.nebula.rbac.local.entity.RoleEntity;
import com.bizunited.nebula.rbac.local.repository.ButtonRepository;
import com.bizunited.nebula.rbac.local.repository.CompetenceRepository;
import com.bizunited.nebula.rbac.local.repository.RoleRepository;
import com.bizunited.nebula.rbac.local.service.ButtonService;
import com.bizunited.nebula.rbac.sdk.event.ButtonEventListener;
import com.bizunited.nebula.rbac.sdk.service.CompetenceVoService;
import com.bizunited.nebula.rbac.sdk.service.RoleVoCacheService;
import com.bizunited.nebula.rbac.sdk.vo.ButtonVo;
import com.google.common.collect.Sets;

/**
 * 按钮服务（不公布成SDK的部分）
 * @author yinwenjie
 */
@Service
public class ButtonServiceImpl implements ButtonService {

  @Autowired
  private ButtonRepository buttonRepository;
  @Autowired
  private RoleRepository roleRepository;
  @Autowired
  private CompetenceRepository competenceRepository;
  @Autowired
  private CompetenceVoService competenceVoService;
  @Autowired
  private RoleVoCacheService roleVoService;
  @Autowired(required = false)
  private List<ButtonEventListener> buttonEventListeners;
  @Autowired
  private NebulaToolkitService nebulaToolkitService;
  
  @Override
  @Transactional
  public ButtonEntity create(ButtonEntity button) {
    Date nowDate = new Date();
    button.setId(null);
    button.setCreateTime(nowDate);
    button.setModifyTime(nowDate);
    button.setSystem(false);
    button.setEffective(true);
    this.createValidation(button);
    this.buttonRepository.save(button);
    
    // 触发事件
    ButtonVo buttonVo = this.nebulaToolkitService.copyObjectByWhiteList(button, ButtonVo.class, LinkedHashSet.class, ArrayList.class);
    if(!CollectionUtils.isEmpty(buttonEventListeners)) {
      for (ButtonEventListener buttonEventListener : buttonEventListeners) {
        buttonEventListener.onCreated(buttonVo);
      }
    }
    return button;
  }

  /**
   * 新增-参数校验
   * @param button
   *
   */
  private void createValidation(ButtonEntity button) {
    Validate.notNull(button, "保存的按钮对象不能为空!");
    Validate.notNull(button.getTopCompetence(), "菜单对象为空，请检查!");
    Validate.notBlank(button.getTopCompetence().getId(), "菜单ID不能为空");
    Validate.notBlank(button.getCode(), "按钮编码不能为空，请检查!");
    Validate.notBlank(button.getName(), "按钮名称不能为空，请检查!");
    Validate.isTrue(button.getId() == null, "创建数据不能有ID");
    //验证是否是菜单
    CompetenceEntity competence = competenceRepository.findById(button.getTopCompetence().getId()).orElse(null);
    Validate.notNull(competence, "未查询到该菜单!");
    //验证按钮编码是否重复
    long countByCode = buttonRepository.countByCode(button.getCode());
    Validate.isTrue(countByCode == 0L, "按钮的编码重复，请检查!");
  }
  
  @Override
  @Transactional
  public void batchCreate(Set<ButtonEntity> buttons) {
    Validate.notEmpty(buttons, "批量创建按钮时，必须至少有一个按钮!!");
    Set<ButtonEntity> buttonSet = new HashSet<>();
    Set<String> codes = new HashSet<>();
    for(ButtonEntity button : buttons){
      //入参校验
      this.createValidation(button);
      //判断是否重复
      Validate.isTrue(!codes.contains(button.getCode()), "传入按钮数据（按钮业务编号）有重复，请检查！");
      codes.add(button.getCode());
      // 可能的操作者信息(如果没有就写入admin)
      Authentication authentication = SecurityContextHolder.getContext().getAuthentication();
      String account = "admin";
      if(authentication != null ) {
        account = authentication.getName();
      }
      button.setCreateTime(new Date());
      button.setCreateAccount(account);
      button.setModifyTime(new Date());
      button.setModifyAccount(account);
      button.setSystem(false);
      buttonSet.add(button);
    }
    this.buttonRepository.saveAll(buttonSet);
    
    // 触发事件
    for (ButtonEntity button : buttonSet) {
      ButtonVo buttonVo = this.nebulaToolkitService.copyObjectByWhiteList(button, ButtonVo.class, LinkedHashSet.class, ArrayList.class);
      if(!CollectionUtils.isEmpty(buttonEventListeners)) {
        for (ButtonEventListener buttonEventListener : buttonEventListeners) {
          buttonEventListener.onCreated(buttonVo);
        }
      }
    }
  }

  @Override
  @Transactional
  public ButtonEntity update(ButtonEntity button) {
    //入参校验
    this.updateValidation(button);
    ButtonEntity updateButton = buttonRepository.findById(button.getId()).orElse(null);
    Validate.notNull(updateButton, "未在数据层找到对应的按钮信息!!");
    Validate.isTrue(!updateButton.getSystem(), "系统按钮不能更新");
    if (button.getButtonDesc() != null){
      updateButton.setButtonDesc(button.getButtonDesc());
    }
    updateButton.setName(button.getName());
    updateButton.setCode(button.getCode());
    updateButton.setModifyTime(new Date());
    // 可能的操作者信息(如果没有就写入admin)
    Authentication authentication = SecurityContextHolder.getContext().getAuthentication();
    String account = "admin";
    if(authentication != null ) {
      account = authentication.getName();
    }
    updateButton.setModifyAccount(account);
    this.buttonRepository.save(updateButton);
    
    // 提醒缓存清理
    Set<RoleEntity> roles = updateButton.getRoles();
    Set<String> tenantCodes = Sets.newHashSet();
    if(!CollectionUtils.isEmpty(roles)) {
      roles.forEach(item -> tenantCodes.add(item.getTenantCode()));
    }
    this.notifyCacheClear(tenantCodes, updateButton);
    return updateButton;
  }

  /**
   * 修改-参数校验
   * @param button
   */
  private void updateValidation(ButtonEntity button) {
    Validate.notBlank(button.getCode(), "按钮编码不能为空，请检查!");
    Validate.notBlank(button.getName(), "按钮名称不能为空，请检查!");
    Validate.notBlank(button.getId(), "修改数据必须要有ID");
    //验证按钮编码是否重复
    long countByCode = buttonRepository.countByCodeWithoutId(button.getCode(), button.getId());
    Validate.isTrue(countByCode == 0L, "按钮的编码重复，请检查!");
  }
  
  /**
   * 由于按钮的信息变化，需要通知功能菜单缓存和角色信息缓存，所以都通过这个方法进行通知
   * @param tenantCodes
   * @param button
   */
  private void notifyCacheClear(Set<String> tenantCodes , ButtonEntity button) {
    String appCode = TenantUtils.getAppCode();
    this.competenceVoService.notifyCacheRefresh(appCode);
    if(!CollectionUtils.isEmpty(tenantCodes)) {
      for (String tenantCode : tenantCodes) {
        this.roleVoService.notifyCacheRefresh(tenantCode);
      }
    }
  }
  
  @Override
  @Transactional
  public void deleteById(String id) {
    Validate.notNull(id, "删除必须要给定主键信息!");
    ButtonEntity button = buttonRepository.findById(id).orElse(null);
    Validate.notNull(button , "删除时，未找到指定的按钮信息，请检查!!");
    Validate.isTrue(!button.getSystem(), "系统按钮不能删除");
    Set<RoleEntity> roles = button.getRoles();
    Set<String> tenantCodes = Sets.newHashSet();
    if(!CollectionUtils.isEmpty(roles)) {
      roles.forEach(item -> tenantCodes.add(item.getTenantCode()));
    }
    // 由于按钮时RBAC中最底层节点，且不可能在挂接任何下层节点，所以即使有角色和功能的绑定，也可以进行删除
    this.buttonRepository.unbindAllByButtonId(id);
    this.buttonRepository.unbindAllRoleByButtonId(id);
    // 然后才进行按钮删除
    this.buttonRepository.delete(button);
    // 删除通知
    if(!CollectionUtils.isEmpty(this.buttonEventListeners)) {
      ButtonVo buttonVo = this.nebulaToolkitService.copyObjectByWhiteList(button, ButtonVo.class, LinkedHashSet.class, ArrayList.class);
      for (ButtonEventListener buttonEventListener : buttonEventListeners) {
        buttonEventListener.onDeleted(buttonVo);
      }
    }
    
    // 最后缓存清理的通知
    this.notifyCacheClear(tenantCodes, button);
  }
  
  @Override
  @Transactional
  public void batchDelete(String[] ids) {
    Validate.isTrue(ids != null && ids.length > 0 , "进行按钮删除时，必须传入要删除的按钮技术编号!!");
    for (String id : ids) {
      this.deleteById(id);
    }
  }

  @Override
  @Transactional
  public void bindRole(String roleId, String[] buttonIds) {
    Validate.notBlank(roleId , "进行按钮和角色绑定时，必须指定当前角色编号!!");
    Optional<RoleEntity> op = this.roleRepository.findById(roleId);
    Validate.isTrue(op.isPresent() , "没有发现指定的角色信息");
    Validate.isTrue(buttonIds != null && buttonIds.length > 0 , "进行按钮和角色绑定时，必须至少指定一个功能编号!!");
    for (String buttonId : buttonIds) {
      Optional<ButtonEntity> opButton = this.buttonRepository.findById(buttonId);
      Validate.isTrue(opButton.isPresent() , "没有发现指定的按钮信息[%s]，请检查!!" , buttonId);
      
      // 该判断逻辑保证不会进行重复绑定
      long count = this.buttonRepository.countByRoleIdAndButtonId(roleId, buttonId);
      if(count == 0) {
        this.buttonRepository.bindRole(roleId, buttonId);
      }
    }  
    // 进行通知
    RoleEntity currentRole = op.get();
    this.roleVoService.notifyCacheRefresh(currentRole.getTenantCode());
  }

  @Override
  @Transactional
  public void rebindRole(String roleId, String[] buttonIds) {
    /*
     * 所谓重新绑定，就是删除之前和这个roleId有关的按钮绑定
     * 然后重新进行绑定
     * */
    Validate.notBlank(roleId,"角色主键不能为空");
    RoleEntity currentRole = this.roleRepository.findById(roleId).orElse(null);
    Validate.notNull(currentRole , "没有发现指定的角色信息");
    Set<ButtonEntity> buttons = this.buttonRepository.findByRoleId(roleId);
    if(!CollectionUtils.isEmpty(buttons)){
      String[] exsitButtonIds = buttons.stream().map(ButtonEntity::getId).toArray(String[]::new);
      if(exsitButtonIds != null) {
        for (String exsitButtonId : exsitButtonIds) {
          this.buttonRepository.unbindRole(roleId, exsitButtonId);
        }
      }
    }
    
    //然后进行重新绑定
    if(buttonIds != null) {
      for (String buttonId : buttonIds) {
        Optional<ButtonEntity> opButton = this.buttonRepository.findById(buttonId);
        Validate.isTrue(opButton.isPresent() , "没有发现指定的按钮信息[%s]，请检查!!" , buttonId);
        this.buttonRepository.bindRole(roleId, buttonId);
      }
    }
    
    // 进行角色缓存更新的通知
    this.roleVoService.notifyCacheRefresh(currentRole.getTenantCode());
  }

  @Override
  @Transactional
  public void unbindRole(String roleId, String[] buttonIds) {
    Validate.notBlank(roleId , "进行按钮和角色解绑时，必须指定当前角色编号!!");
    Optional<RoleEntity> op = this.roleRepository.findById(roleId);
    Validate.isTrue(op.isPresent() , "没有发现指定的角色信息");
    Validate.isTrue(buttonIds != null && buttonIds.length > 0 , "进行按钮和角色解绑时，必须至少指定一个功能编号!!");

    for (String buttonId : buttonIds) {
      Optional<ButtonEntity> opButton = this.buttonRepository.findById(buttonId);
      Validate.isTrue(opButton.isPresent() , "没有发现指定的按钮信息[%s]，请检查!!" , buttonId);
      this.competenceRepository.unbindRole(roleId, buttonId);
    }
    
    // 进行通知
    RoleEntity currentRole = op.get();
    this.roleVoService.notifyCacheRefresh(currentRole.getTenantCode());
  }

  @Override
  @Transactional
  public void unbindAllRoles(String roleId) {
    Validate.notBlank(roleId,"角色主键不能为空");
    Optional<RoleEntity> op = this.roleRepository.findById(roleId);
    Validate.isTrue(op.isPresent() , "没有发现指定的角色信息");
    //解绑角色与菜单之间的关联;
    Set<ButtonEntity> buttons = this.buttonRepository.findByRoleId(roleId);
    if(CollectionUtils.isEmpty(buttons)){
      return;
    }
    String[] buttonIds = buttons.stream().map(ButtonEntity::getId).toArray(String[]::new);
    this.unbindRole(roleId , buttonIds);
    
    // 进行通知
    RoleEntity currentRole = op.get();
    this.roleVoService.notifyCacheRefresh(currentRole.getTenantCode());
  }
}