package com.bizunited.nebula.rbac.local.service.internal;

import java.util.ArrayList;
import java.util.Date;
import java.util.HashSet;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Optional;
import java.util.Set;

import javax.transaction.Transactional;

import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.PageImpl;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Pageable;
import org.springframework.security.access.AccessDeniedException;
import org.springframework.security.core.Authentication;
import org.springframework.security.core.context.SecurityContextHolder;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;

import com.bizunited.nebula.common.configuration.SimpleTenantProperties;
import com.bizunited.nebula.common.service.CodeGeneratorService;
import com.bizunited.nebula.common.service.NebulaToolkitService;
import com.bizunited.nebula.common.util.tenant.TenantUtils;
import com.bizunited.nebula.rbac.local.dto.RoleConditionDto;
import com.bizunited.nebula.rbac.local.entity.ButtonEntity;
import com.bizunited.nebula.rbac.local.entity.CompetenceEntity;
import com.bizunited.nebula.rbac.local.entity.RoleEntity;
import com.bizunited.nebula.rbac.local.repository.RoleRepository;
import com.bizunited.nebula.rbac.local.service.ButtonService;
import com.bizunited.nebula.rbac.local.service.CompetenceService;
import com.bizunited.nebula.rbac.local.service.RoleService;
import com.bizunited.nebula.rbac.sdk.config.RbacCustomProperties;
import com.bizunited.nebula.rbac.sdk.constant.Constants;
import com.bizunited.nebula.rbac.sdk.constant.RedisKeyConstants;
import com.bizunited.nebula.rbac.sdk.event.RoleEventListener;
import com.bizunited.nebula.rbac.sdk.service.RoleVoCacheService;
import com.bizunited.nebula.rbac.sdk.vo.RoleVo;
import com.google.common.collect.Lists;
import com.google.common.collect.Sets;

/**
 * 请注意角色业务编号（code），除了系统为指定租户初始化的BASEROLE和ADMIN两个角色以外，其它角色的Code全部都是系统自动生成。
 * 而且是租户唯一。
 * @author yinwenjie paul
 */
@Service
public class RoleServiceImpl implements RoleService {

  @Autowired
  private RoleRepository roleRepository;
  @Autowired
  private RoleVoCacheService roleVoService;
  @Autowired(required = false)
  private List<RoleEventListener> roleEventListeners;
  @Autowired
  private CodeGeneratorService codeGeneratorService;
  @Autowired
  private CompetenceService competenceService;
  @Autowired
  private ButtonService buttonService;
  @Autowired
  private NebulaToolkitService nebulaToolkitService;
  @Autowired
  private RbacCustomProperties rbacCustomProperties;
  @Autowired
  private SimpleTenantProperties simpleTenantProperties;
  
  @Transactional
  @Override
  public RoleEntity create(RoleEntity role) {
    /*
     * 创建角色时，操作步骤为：
     * 1、首先在验证角色基本信息的正确性以后，先创建角色的基本信息(包括上级信息)
     * 2、如果创建角色时，携带了关联的菜单/功能信息，则进行角色和菜单/功能的绑定
     * 3、如果创建角色时，携带了关联的按钮信息，则进行角色和按钮的绑定
     * 4、进行角色创建后的事件通知
     * */
    
    // 1、=======
    String tenantCode = TenantUtils.getTenantCode();
    // 如果没有传入编码，则重新生成一个
    if(StringUtils.isBlank(role.getRoleCode())) {
      String code = codeGeneratorService.generate(String.format(RedisKeyConstants.ROLE_CODE_INDEX, tenantCode), Constants.ROLE_CODE_PREFIX, null, 1, 6);
      role.setRoleCode(code);
    }
    role.setTenantCode(tenantCode);
    // 进入传入信息的检查
    this.validateRoleBeforeCreate(role);
    this.roleRepository.saveAndFlush(role);
    
    // 2、======
    Set<CompetenceEntity> competences = role.getCompetences();
    if(!CollectionUtils.isEmpty(competences)) {
      String[] competenceIds = competences.stream().map(CompetenceEntity::getId).toArray(String[]::new);
      this.competenceService.bindRole(role.getId(), competenceIds);
    }
    
    // 3、=====
    // 注意，buttonService由于没有缓存，所以buttonService内部不会刷新缓存
    Set<ButtonEntity> buttons = role.getButtons();
    if(!CollectionUtils.isEmpty(buttons)) {
      String[] buttonIds = buttons.stream().map(ButtonEntity::getId).toArray(String[]::new);
      this.buttonService.bindRole(role.getId(), buttonIds);
    }
    
    // 4、=====
    if(CollectionUtils.isEmpty(this.roleEventListeners)) {
      RoleVo roleVo = this.nebulaToolkitService.copyObjectByWhiteList(role, RoleVo.class, LinkedHashSet.class, ArrayList.class);
      for (RoleEventListener roleEventListener : roleEventListeners) {
        roleEventListener.onCreated(roleVo);
      }
    }
    this.roleVoService.notifyCacheRefresh(tenantCode);
    return role;
  }
  
  /**
   * 该私有方法在新增一个role信息前，检查传入信息的正确性
   */
  private void validateRoleBeforeCreate(RoleEntity role) {
    Validate.notNull(role, "角色信息不能为空!");
    Validate.notBlank(role.getRoleName(), "角色名称不能为空!");
    Validate.notBlank(role.getTenantCode() , "租户信息/系统信息 不能为空!");
    String roleCode = role.getRoleCode();
    Validate.notBlank(roleCode , "添加角色时，角色业务编号必须填写!");
    Validate.isTrue(StringUtils.indexOf(roleCode, "_") == -1 , "添加角色时，角色业务编号不允许使用“_”等特殊字符!");
    // code必须验证
    RoleEntity oldRole = this.roleRepository.findByTenantCodeAndRoleCode(role.getTenantCode(), roleCode);
    Validate.isTrue(oldRole == null, "当前设定的角色编号（role code[%s]）已经被使用，请更换!" , roleCode);
    // 当前的创建时间和修改时间要一起写入(如果没有登录人信息，统一写为admin)
    Date currentTime = new Date();
    role.setCreateTime(currentTime);
    role.setModifyTime(currentTime);
    if(StringUtils.isBlank(role.getCreateAccount())){
      Authentication authentication = SecurityContextHolder.getContext().getAuthentication();
      if(authentication != null ) {
        role.setCreateAccount(authentication.getName());
        role.setModifyAccount(authentication.getName());
      }
    }

    Validate.isTrue(StringUtils.isNotBlank(role.getCreateAccount()) && StringUtils.isNotBlank(role.getModifyAccount()),"角色创建人和修改人信息不能为空");
    // 说明性信息
    if (StringUtils.isBlank(role.getComment())) {
      role.setComment("[未填写]");
    }
    // 当前角色必须是状态正常的
    Validate.isTrue(role.getTstatus() == null || role.getTstatus() == 1, "当前角色必须是状态正常的");
    role.setTstatus(1);
    if (role.getParent() != null) {
      Validate.notBlank(role.getParent().getId(), "未找到该角色的父级角色的id，请检查！");
      RoleEntity parentRole = roleRepository.findById(role.getParent().getId()).orElse(null);
      Validate.notNull(parentRole, "未找到角色的父级角色");
    }
  }

  @Transactional
  @Override
  public RoleEntity update(RoleEntity role) {
    /*
     * 修改操作只能修改指定角色的基本信息
     * 注意：
     * 1、管理员角色不能进行修改
     * 2：由于修改信息只是修改了角色的基本信息，根本不涉及关联信息核心数据的修改，所以不需要向上层业务模块进行事件通知
     * */
    Validate.notNull(role, "角色信息不能为空");
    String roleId = role.getId();
    String updateComment = role.getComment();
    Validate.notEmpty(roleId, "角色不存在，请检查!");
    Optional<RoleEntity> op = this.roleRepository.findById(roleId);
    Validate.isTrue(op.isPresent(), "未找到该角色");
    RoleEntity currentRole = op.get();
    String roleCode = role.getRoleCode();
    Validate.isTrue(!StringUtils.isAnyBlank(roleCode, currentRole.getRoleCode()), "角色编码不能为空！");
    Validate.isTrue(StringUtils.indexOf(roleCode, "_") == -1 , "添加角色时，角色业务编号不允许使用“_”等特殊字符!");
    String dbRoleCode = currentRole.getRoleCode();
    if (!StringUtils.equals(dbRoleCode, roleCode)) {
      RoleEntity roleEntity = roleRepository.findByTenantCodeAndRoleCode(TenantUtils.getTenantCode(), roleCode);
      Validate.isTrue(roleEntity == null, "角色编码【%s】已经存在", roleCode);
    }
    String currentName = currentRole.getRoleName();
    // 如果条件成立，说明这个角色信息不能被修改
    if(this.rbacCustomProperties.getDeleteDenys() != null) {
      for (String deleteDeny : this.rbacCustomProperties.getDeleteDenys()) {
        if (StringUtils.equals(currentName, deleteDeny)) {
          throw new AccessDeniedException("这个角色为系统特定默认角色，不允许修改！");
        }
      }
    }
    
    // ========== 以下步骤开始进行基本信息的修改（包括父级节点信息）
    if (!StringUtils.isBlank(updateComment)) {
      currentRole.setComment(updateComment);
    }
    if (role.getParent() != null) {
      Validate.notBlank(role.getParent().getId(), "未找到该角色的父级角色的id，请检查！");
      RoleEntity parentRole = roleRepository.findById(role.getParent().getId()).orElse(null);
      Validate.notNull(parentRole, "未找到角色的父级角色");
      Validate.isTrue(!role.getId().equals(parentRole.getId()), "禁止将该角色本身设置为上级角色");
      currentRole.setParent(parentRole);
      Set<String> roleStack = new HashSet<>();
      roleStack.add(role.getParent().getId());
      this.handleCircular(currentRole, roleStack);
    } else {
      currentRole.setParent(null);
    }
    currentRole.setRoleName(role.getRoleName());
    currentRole.setRoleType(role.getRoleType());
    currentRole.setIsDeny(role.getIsDeny());
    //修改扩展字段
    currentRole.setExtend1(role.getExtend1());
    currentRole.setExtend2(role.getExtend2());
    currentRole.setExtend3(role.getExtend3());
    currentRole.setExtend4(role.getExtend4());
    currentRole.setExtend5(role.getExtend5());
    currentRole.setExtend6(role.getExtend6());
    currentRole.setExtend7(role.getExtend7());
    currentRole.setExtend8(role.getExtend8());
    currentRole.setExtend9(role.getExtend9());
    currentRole.setExtend10(role.getExtend10());
    currentRole.setTenantCode(TenantUtils.getTenantCode());
    this.roleRepository.saveAndFlush(currentRole);
    
    // 进行缓存清理通知
    String tenantCode = role.getTenantCode();
    this.roleVoService.notifyCacheRefresh(tenantCode);
    return role;
  }
  
  /**
   * 判断父级角色是否形成循环依赖
   * @param parent
   * @param roleStack
   */
  private void handleCircular(RoleEntity parent, Set<String> roleStack) {
    if (CollectionUtils.isEmpty(parent.getChildren())) {
      return;
    }
    for (RoleEntity roleEntity : parent.getChildren()) {
      Validate.isTrue(!roleStack.contains(roleEntity.getId()), "形成循环依赖，更新失败，请检查！");
      roleStack.add(roleEntity.getId());
      this.handleCircular(roleEntity, roleStack);
    }
  }
  
  @Transactional
  @Override
  public void enable(String[] roleIds) {
    Validate.notEmpty(roleIds, "角色id不能为空!");
    Set<String> tenantCodes = Sets.newHashSet();
    for (String roleId : roleIds) {
      RoleEntity currentRole = this.roleRepository.findById(roleId).orElse(null);
      Validate.notNull(currentRole, "未找到指定的角色信息，请检查!!");
      Validate.isTrue(!currentRole.getIsDeny(),"角色【%s】系统角色不允许启用禁用",currentRole.getRoleName());
      // 启用角色
      currentRole.setTstatus(1);
      this.roleRepository.save(currentRole);
      tenantCodes.add(currentRole.getTenantCode());
    }
    
    // 根据roleIds进行tenantCode分组，以便通知role缓存更新
    for (String tenantCode : tenantCodes) {
      this.roleVoService.notifyCacheRefresh(tenantCode);
    }
  }

  @Transactional
  @Override
  public void disable(String[] roleIds) {
    /*
     * 注意：在配置文件中已经设定的那些不能操作的角色信息，是不允许禁用的
     */
    Validate.notEmpty(roleIds, "角色id不能为空!");
    Set<String> tenantCodes = Sets.newHashSet();
    for (String roleId : roleIds) {
      RoleEntity currentRole = this.roleRepository.findById(roleId).orElse(null);
      Validate.notNull(currentRole, "未找到指定的角色信息，请检查!!");
      Validate.isTrue(!currentRole.getIsDeny(),"角色【%s】系统角色不允许启用禁用",currentRole.getRoleName());
      // 如果条件成立，说明这个角色信息不能被删除（或者说作废）
      if(this.rbacCustomProperties.getDeleteDenys() != null && 
          StringUtils.equalsAny(currentRole.getRoleCode() , this.rbacCustomProperties.getDeleteDenys())) {
        throw new AccessDeniedException("至少有一个角色被不能禁用！");
      }
      tenantCodes.add(currentRole.getTenantCode());
      
      // 禁用角色
      currentRole.setTstatus(0);
      this.roleRepository.save(currentRole);
    }
    
    // 根据roleIds进行tenantCode分组，以便通知role缓存更新
    for (String tenantCode : tenantCodes) {
      this.roleVoService.notifyCacheRefresh(tenantCode);
    }
  }
  
  @Transactional
  @Override
  public void deleteByIds(String[] roleIds) {
    Validate.notEmpty(roleIds, "删除时，至少有一个角色编码");
    Set<String> tenantCodes = Sets.newHashSet();
    for (String roleId : roleIds) {
      RoleEntity roleEntity = this.roleRepository.findById(roleId).orElse(null);
      Validate.notNull(roleEntity, "至少有一个角色不存在，无法删除");
      // 发出事件通知
      if(!CollectionUtils.isEmpty(this.roleEventListeners)) {
        RoleVo roleVo = this.nebulaToolkitService.copyObjectByWhiteList(roleEntity, RoleVo.class, LinkedHashSet.class, ArrayList.class);
        for (RoleEventListener roleEventListener : this.roleEventListeners) {
          roleEventListener.onDeleted(roleVo);
        }
      }
      tenantCodes.add(roleEntity.getTenantCode());
      
      // 判定角色和按钮的的绑定情况
      Set<CompetenceEntity> competences = roleEntity.getCompetences();
      Validate.isTrue(!CollectionUtils.isEmpty(competences) , "当前角色已经绑定了菜单/功能，不能进行删除");
      // 判定角色和功能的绑定情况
      Set<ButtonEntity> buttons = roleEntity.getButtons();
      Validate.isTrue(!CollectionUtils.isEmpty(buttons) , "当前角色已经绑定了按钮，不能进行删除");
      // 最终再删除角色基本信息
      this.roleRepository.deleteById(roleId);
    }
    
    // 根据roleIds进行tenantCode分组，以便通知role缓存更新
    for (String tenantCode : tenantCodes) {
      this.roleVoService.notifyCacheRefresh(tenantCode);
    }
  }
  
  // ============ 注意，以下查询方式，都是从数据库进行查询

  @Override
  public RoleEntity findDetailsById(String id) {
    if(StringUtils.isBlank(id)) {
      return null;
    }
    RoleEntity role = roleRepository.findDetailsById(id);
    if(role == null) {
      return null;
    }
    return this.buildCopyRole(role);
  }

  @Override
  public Set<RoleEntity> findDetailsByIds(List<String> ids) {
    if(CollectionUtils.isEmpty(ids)) {
      return null;
    }
    Set<RoleEntity> resutls = this.roleRepository.findByIds(ids);
    if(CollectionUtils.isEmpty(resutls)) {
      return null;
    }
    
    // 进行中拷
    Set<RoleEntity> targetResutls = Sets.newLinkedHashSet();
    for (RoleEntity role : resutls) {
      targetResutls.add(this.buildCopyRole(role));
    }
    return targetResutls;
  }

  @Override
  public RoleEntity findDetailsByCode(String tenantCode, String roleCode) {
    if(StringUtils.isBlank(tenantCode)) {
      tenantCode = this.simpleTenantProperties.getDefaultTenantCode();
    }
    RoleEntity roleEntity = this.roleRepository.findByTenantCodeAndRoleCode(tenantCode, roleCode);
    return this.buildCopyRole(roleEntity);
  }

  @Override
  public Set<RoleEntity> findDetailsByCodes(String tenantCode, List<String> roleCodes) {
    if(CollectionUtils.isEmpty(roleCodes)) {
      return null;
    }
    if(StringUtils.isBlank(tenantCode)) {
      tenantCode = this.simpleTenantProperties.getDefaultTenantCode();
    }
    
    Set<RoleEntity> resutls = this.roleRepository.findByTenantCodeAndRoleCodes(tenantCode, Sets.newLinkedHashSet(roleCodes));
    // 进行中拷
    Set<RoleEntity> targetResutls = Sets.newLinkedHashSet();
    for (RoleEntity role : resutls) {
      targetResutls.add(this.buildCopyRole(role));
    }
    return targetResutls;
  }

  @Override
  public Page<RoleEntity> findByConditions(RoleConditionDto conditions, Pageable pageable) {
    if(pageable == null) {
      pageable = PageRequest.of(0, 20);
    }
    if(conditions == null) {
      conditions = new RoleConditionDto();
    }
    // 设定租户
    conditions.setTenantCode(TenantUtils.getTenantCode());
    Page<RoleEntity> results = this.roleRepository.findByConditions(conditions, pageable);
    if(results == null || CollectionUtils.isEmpty(results.getContent())) {
      return null;
    }
    
    // 进行内容信息中考
    List<RoleEntity> contents = results.getContent();
    List<RoleEntity> targetContents = Lists.newArrayList();
    for (RoleEntity content : contents) {
      RoleEntity targetContent = this.buildCopyRole(content);
      targetContents.add(targetContent);
    }
    return new PageImpl<>(targetContents, pageable, results.getTotalElements());
  }
  
  /**
   * 对角色信息进行中考，形成一味结构。注意其中的parent属性不能形成循环依赖</br>
   * 并且不包括子级角色信息
   * @param roleEntity
   * @return
   */
  private RoleEntity buildCopyRole(RoleEntity roleEntity) {
    RoleEntity targetRole = this.nebulaToolkitService.copyObjectByWhiteList(roleEntity, RoleEntity.class, LinkedHashSet.class, ArrayList.class);
    RoleEntity parentEntity = roleEntity.getParent();
    if(parentEntity != null) {
      RoleEntity targetParentEntity = this.nebulaToolkitService.copyObjectByWhiteList(parentEntity, RoleEntity.class, LinkedHashSet.class, ArrayList.class);
      targetRole.setParent(targetParentEntity);
    }
    return targetRole;
  }
}