package com.bizunited.nebula.rbac.local.service.internal;

import java.util.ArrayList;
import java.util.Collection;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.locks.ReentrantReadWriteLock;
import java.util.concurrent.locks.ReentrantReadWriteLock.ReadLock;
import java.util.concurrent.locks.ReentrantReadWriteLock.WriteLock;

import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.redisson.Redisson;
import org.redisson.api.RTopic;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.util.CollectionUtils;

import com.bizunited.nebula.common.service.NebulaToolkitService;
import com.bizunited.nebula.rbac.local.entity.RoleEntity;
import com.bizunited.nebula.rbac.local.repository.RoleRepository;
import com.bizunited.nebula.rbac.sdk.config.RbacCustomProperties;
import com.bizunited.nebula.rbac.sdk.event.RoleEventListener;
import com.bizunited.nebula.rbac.sdk.service.RoleVoCacheService;
import com.bizunited.nebula.rbac.sdk.vo.RoleVo;
import com.google.common.collect.Maps;
import com.google.common.collect.Sets;

/**
 * 角色信息缓存不会记录角色和菜单/功能以及按钮间的关联信息</br>
 * 而主要是记录角色和用户的关系</p>
 * @author yinwenjie
 */
public class RoleVoCacheServiceImpl implements RoleVoCacheService {
  @Autowired
  private RoleRepository roleRepository;
  @Autowired
  private NebulaToolkitService nebulaToolkitService;
  @Autowired
  private Redisson redisson;
  @Autowired(required = false)
  private List<RoleEventListener> roleEventListeners;
  @Autowired
  private RbacCustomProperties rbacConfig;
  /**
   * 这是按照租户为单位进行的当前有效的角色缓存信息
   * K0：租户信息</br>
   * K1：角色的业务编号</br>
   * V0：角色详细信息（带所有关联信息）</p>
   * 
   * 这里请注意，指定tenantCode下的RoleVo信息，在完成缓存后是一个一维表结构
   * 如果调用者需要返回一个树形结构，可以通过诸如findByRoleCodeForTree这样的方式进行查询
   */
  private static Map<String, Map<String , RoleVo>> roleCacheMapping = Maps.newConcurrentMap();
  /**
   * 保证在缓存刷新过程中，需要读取缓存的线程进行阻塞(按照租户进行区分)</br>
   * K0：租户信息</br>
   * V0：可以使用的读写分离锁</p>
   */
  private static Map<String, ReentrantReadWriteLock> roleLockMapping = Maps.newConcurrentMap();
  
  @Override
  public void clearCache(String tenantCode) {
    Validate.notBlank(tenantCode , "清理RBAC缓存时，二级租户信息必须传入!!");
    WriteLock writeLock = this.getWriteLockByTenantCode(tenantCode);
    try {
      writeLock.lock();
      if(roleCacheMapping != null) {
        roleCacheMapping.remove(tenantCode);
      }
    } finally {
      writeLock.unlock();
    }
  }
  
  @Override
  public void clearCache(String tenantCode, String roleCode) {
    WriteLock writeLock = this.getWriteLockByTenantCode(tenantCode);
    try {
      writeLock.lock();
      if(roleCacheMapping != null) {
        Map<String, RoleVo> cacheRoleMapping = roleCacheMapping.get(tenantCode);
        RoleVo cacheRole = null;
        if(cacheRoleMapping != null && (cacheRole = cacheRoleMapping.get(roleCode)) != null) {
          cacheRole.setCacheInvalid(true);
        }
      }
    } finally {
      writeLock.unlock();
    }
  }

  /**
   * 通过指定的经销商业务编号，获取经销商对缓存操作的写锁
   * @param tenantCode
   * @return 
   */
  private WriteLock getWriteLockByTenantCode(String tenantCode) {
    while(roleLockMapping.get(tenantCode) == null) {
      synchronized (roleLockMapping) {
        if(roleLockMapping.get(tenantCode) == null) {
          roleLockMapping.put(tenantCode, new ReentrantReadWriteLock());
        }
      }
    }
    ReentrantReadWriteLock lock = roleLockMapping.get(tenantCode);
    return lock.writeLock(); 
  }
  
  /**
   * 通过指定的经销商业务编号，获取经销商对缓存读取的读锁
   * @param tenantCode
   * @return
   */
  private ReadLock getReadLockByTenantCode(String tenantCode) {
    while(roleLockMapping.get(tenantCode) == null) {
      synchronized (roleLockMapping) {
        if(roleLockMapping.get(tenantCode) == null) {
          roleLockMapping.put(tenantCode, new ReentrantReadWriteLock());
        }
      }
    }
    ReentrantReadWriteLock lock = roleLockMapping.get(tenantCode);
    return lock.readLock();
  }

  @Override
  public void notifyCacheRefresh(String tenantCode) {
    if(StringUtils.isBlank(tenantCode)) {
      return;
    }
    RTopic topic = redisson.getTopic(RoleVoCacheService.TENANT_ROLE_NOTIFY);
    topic.publish(tenantCode);
  }
  
  @Override
  public void notifyCacheRefresh(String tenantCode, String roleCode) {
    if(StringUtils.isAnyBlank(tenantCode , roleCode)) {
      return;
    }
    RTopic topic = redisson.getTopic(RoleVoCacheService.TENANT_ROLE_IDENTITY_NOTIFY);
    topic.publish(StringUtils.join(tenantCode , "|" , roleCode));
  }

  @Override
  public String findBaseRoleCode() {
    return this.rbacConfig.getBaseRoleCode();
  }

  /**
   * 该方法从数据层，基于指定的tenantCode进行全角色缓存的构建
   * 使用该方法时，无论指定tenantCode下的某个角色节点是否失效，都会进行重新缓存
   * @return
   */
  private void findDetailsFromRepository(String tenantCode) {
    if(StringUtils.isEmpty(tenantCode)) {
      return;
    }
    roleCacheMapping.remove(tenantCode);
    Set<RoleEntity> rootRoles = this.roleRepository.findByTenantCodeAndParentIsNull(tenantCode);
    if(CollectionUtils.isEmpty(rootRoles)) {
      return;
    }
    
    /*
     * 处理过程为：
     * 1、首先从读锁改为写锁，并考虑高并发场景
     * 2、查询该tenantCode下所有的状态正常的角色信息，然后调用findDetailsFromRepository依次进行缓存
     * 注意，这里是递归，并且是从父级role开始遍历，且数据库中的属性结构，在缓存后会将为为一维表结构
     * 3、结束后，释放写锁，重新改为读锁
     * */
    // 1、=======
    ReadLock readLock = this.getReadLockByTenantCode(tenantCode);
    readLock.unlock();
    WriteLock writeLock = this.getWriteLockByTenantCode(tenantCode);
    // 如果没有抢占到写锁，则自旋后退出
    if(!writeLock.tryLock()) {
      Thread.yield();
      // 如果条件成立，说明之前抢占到写操作权的线程，还没有工作完成
      while(!writeLock.tryLock()) {
        Thread.yield();
      }
      writeLock.unlock();
      readLock.lock();
      // 退出
      return;
    }
    
    Map<String , RoleVo> cacheRolesMap = Maps.newLinkedHashMap();
    try {
      // 2、========
      for (RoleEntity rootRoleEntity : rootRoles) {
        this.recursiveLoadRole(rootRoleEntity, cacheRolesMap);
      }
      roleCacheMapping.put(tenantCode, cacheRolesMap);
    } finally {
      writeLock.unlock();
      readLock.lock();
    }
  }
  
  /**
   * 递归进行角色树的缓存，缓存的结果将降维成一维结构
   * @param roleEntity
   */
  private void recursiveLoadRole(RoleEntity roleEntity , Map<String , RoleVo> cacheRolesMap) {
    this.findDetailsFromRepository(roleEntity.getTenantCode(), roleEntity, cacheRolesMap, false);
    Set<RoleEntity> children = this.roleRepository.findByTenantCodeAndParent(roleEntity.getTenantCode(), roleEntity.getId());
    // 递归向下级role进行缓存，最终缓存为一个一维结构在cacheRolesMap中
    if(!CollectionUtils.isEmpty(children)) {
      for (RoleEntity childRoleEntity : children) {
        this.recursiveLoadRole(childRoleEntity, cacheRolesMap);
      }
    }
  }
  
  /**
   * 该方法从数据层，基于指定的tenantCode和roleid进行指定角色缓存的构建
   * 并将缓存信息存储到指定的Map信息中
   * @param roleId 
   * @param cachehMap 工作的前提是，该map不能为null。K-角色业务编号；V，缓存的角色和其关联信息
   * @param releaseLock 是否需要释放锁，例如当findDetailsFromRepository(String tenantCode)调用该方法时，就不需要进行锁的释放，因为后者的代码块中已经释放了
   */
  private RoleVo findDetailsFromRepository(String tenantCode , RoleEntity roleEntity , Map<String , RoleVo> cacheRolesMap , boolean releaseLock) {
    if(roleEntity == null) {
      return null;
    }
    ReadLock readLock = null;
    WriteLock writeLock = null;
    if(releaseLock) {
      readLock = this.getReadLockByTenantCode(tenantCode);
      readLock.unlock();
      writeLock = this.getWriteLockByTenantCode(tenantCode);
      // 如果没有抢占到写锁，则自旋后退出
      if(!writeLock.tryLock()) {
        Thread.yield();
        // 如果条件成立，说明之前抢占到写操作权的线程，还没有工作完成
        while(!writeLock.tryLock()) {
          Thread.yield();
        }
        writeLock.unlock();
        readLock.lock();
        // 退出
        return cacheRolesMap.get(roleEntity.getRoleCode());
      }
    }
    
    /*
     * 处理过程为：
     * 1、如果指定的角色存在，那么首先进行基本信息的中拷
     * 2、通知进行角色关联用户，或间接关联用户的查询
     * 3、进行可能的父级角色基本信息的查询（注意，只有基本信息）
     * 4、进行可能的子级角色基本信息的查询（注意，只有基本信息）
     * */
    RoleVo currentRole = null;
    try {
      // 1、======
      currentRole = this.nebulaToolkitService.copyObjectByWhiteList(roleEntity, RoleVo.class, LinkedHashSet.class, ArrayList.class);
      // 注意单词是invalid，那么true就表示已失效
      currentRole.setCacheInvalid(false);
      
      // 2、======
      Map<String , Set<String>> userAccountMaps = Maps.newLinkedHashMap();
      if(!CollectionUtils.isEmpty(this.roleEventListeners)) {
        for (RoleEventListener roleEventListener : roleEventListeners) {
          List<String> eachUserAccounts = null;
          // 如果特定的业务模块存在角色和用户的关联记录（哪怕是间接关联）
          // 那么就需要记录到缓存中——通过userAccountMaps属性
          if(!CollectionUtils.isEmpty(eachUserAccounts = roleEventListener.onRequestUserAccount(currentRole.getRoleCode(), tenantCode))) {
            String mouduleName = roleEventListener.moduleName();
            Set<String> userAccounts = null;
            if(CollectionUtils.isEmpty(userAccounts = userAccountMaps.get(mouduleName))) {
              userAccounts = Sets.newLinkedHashSet(eachUserAccounts);
              userAccountMaps.put(mouduleName, userAccounts);
            } else {
              userAccounts.addAll(eachUserAccounts);
            }
          }
        }
      }
      // 注意，有可能某个角色没有关联任何用户账户信息，这种情况也是正常的
      if(!CollectionUtils.isEmpty(userAccountMaps)) {
        currentRole.setUserAccountMaps(userAccountMaps);
      }
      
      // 3、======
      if(roleEntity.getParent() != null) {
        RoleEntity parentRoleEntity = roleEntity.getParent();
        RoleVo parentRoleVo = this.nebulaToolkitService.copyObjectByWhiteList(parentRoleEntity, RoleVo.class, LinkedHashSet.class, ArrayList.class);
        currentRole.setParent(parentRoleVo);
      }
      
      // 4、======
      Set<RoleEntity> children = this.roleRepository.findByTenantCodeAndParent(tenantCode, roleEntity.getId());
      if(!CollectionUtils.isEmpty(children)) {
        Collection<RoleVo> childrenVos = this.nebulaToolkitService.copyCollectionByBlankList(children, RoleEntity.class, RoleVo.class, LinkedHashSet.class, ArrayList.class);
        currentRole.setChildren(Sets.newHashSet(childrenVos));
      }
      cacheRolesMap.put(currentRole.getRoleCode(), currentRole);
    } finally {
      if(releaseLock) {
        writeLock.unlock();
        readLock.lock();
      }
    }
    return currentRole;
  }

  // =========== 以下都是从缓存中进行查询（呈现一维结构的查询）
  
  /**
   * 这是真正的查询方法，这个查询方法不负责管理读锁
   * @param tenantCode
   * @param roleCode
   * @return
   */
  private RoleVo realFindByRoleCode(String tenantCode, String roleCode) {
    RoleVo currentCacheRole = null;
    Map<String, RoleVo> cacheRolesMapping; ;
    // 如果条件成立，说明需要加载指定租户的缓存信息
    if((cacheRolesMapping = roleCacheMapping.get(tenantCode)) == null) {
      this.findDetailsFromRepository(tenantCode);
      cacheRolesMapping = roleCacheMapping.get(tenantCode);
    }
    if(cacheRolesMapping == null|| cacheRolesMapping.isEmpty()) {
      return null;
    }
    
    // 下面开始做查询
    currentCacheRole = cacheRolesMapping.get(roleCode);
    // 如果条件成立，说明只是当条RoleVo缓存失效，那么还需要再次加载指定租户下、指定角色下的角色缓存信息
    if(currentCacheRole != null && currentCacheRole.isCacheInvalid()) {
      RoleEntity roleEntity = this.roleRepository.findByTenantCodeAndRoleCode(tenantCode, roleCode);
      currentCacheRole = this.findDetailsFromRepository(tenantCode, roleEntity, cacheRolesMapping, true);
    }
    return currentCacheRole;
  }
  
  @Override
  public RoleVo findByTenantCodeAndRoleCode(String tenantCode, String roleCode) {
    if(StringUtils.isAnyBlank(tenantCode , roleCode)) {
      return null;
    }
    ReadLock readLock = this.getReadLockByTenantCode(tenantCode);
    readLock.lock();
    
    try {
      return this.realFindByRoleCode(tenantCode, roleCode);
    } finally {
      readLock.unlock();
    }
  }

  @Override
  public Set<RoleVo> findByTenantCodeAndRoleCodes(String tenantCode, Set<String> roleCodes) {
    if(StringUtils.isBlank(tenantCode) || CollectionUtils.isEmpty(roleCodes)) {
      return null;
    }
    ReadLock readLock = this.getReadLockByTenantCode(tenantCode);
    readLock.lock();
    
    Set<RoleVo> results = Sets.newLinkedHashSet();
    try {
      for (String roleCode : roleCodes) {
        RoleVo roleVo = this.realFindByRoleCode(tenantCode, roleCode);
        if(roleVo != null) {
          results.add(roleVo);
        }
      }
      return results;
    } finally {
      readLock.unlock();
    }
  }

  @Override
  public Set<RoleVo> findByTenantCodeAndUserAccount(String tenantCode, String userAccount) {
    if(StringUtils.isAnyBlank(tenantCode , userAccount)) {
      return null;
    }
    ReadLock readLock = this.getReadLockByTenantCode(tenantCode);
    readLock.lock();
    
    // 结果存储到currentCacheRoles这里
    Set<RoleVo> currentCacheRoles = Sets.newLinkedHashSet();
    try {
      Map<String, RoleVo> cacheRolesMapping = roleCacheMapping.get(tenantCode);
      if(cacheRolesMapping == null) {
        this.findDetailsFromRepository(tenantCode);
        cacheRolesMapping = roleCacheMapping.get(tenantCode);
      } 
      if(cacheRolesMapping == null) {
        return null;
      }
      
      // 开始遍历其中所有role信息，找到服务条件的role信息
      Collection<RoleVo> roles = cacheRolesMapping.values();
      if(CollectionUtils.isEmpty(roles)) {
        return null;
      }
      for (RoleVo roleVo : roles) {
        // 如果条件成立，说明这个角色虽然被缓存，但已经由于各种原因失效了，所以要重新进行加载
        if(roleVo.isCacheInvalid()) {
          RoleEntity roleEntity = this.roleRepository.findByTenantCodeAndRoleCode(tenantCode, roleVo.getRoleCode());
          roleVo = this.findDetailsFromRepository(tenantCode, roleEntity, cacheRolesMapping, true);
        }
        Map<String, Set<String>> userAccountMaps = roleVo.getUserAccountMaps();
        // TODO 如果角色超过10个，（每10个）采用线程池进行查询
        if(userAccountMaps != null && userAccountMaps.size() != 0) {
          Collection<Set<String>> userAccountValues = userAccountMaps.values();
          for (Set<String> userAccountValue : userAccountValues) {
            if(userAccountValue.contains(userAccount)) {
              currentCacheRoles.add(roleVo);
              break;
            }
          }
        }
      }
    } finally {
      readLock.unlock();
    }
    
    return currentCacheRoles;
  }
  
  // ======= 以下都是从缓存中进行查询（呈现树结构的查询）
  // TODO 由于目前业务需求没有相关要求，可以暂缓实现
  
  @Override
  public Set<RoleVo> findRoleTree(String tenantCode) {
    // TODO Auto-generated method stub
    return null;
  }

  @Override
  public Set<RoleVo> findRoleTree(String tenantCode, String roleCode) {
    // TODO Auto-generated method stub
    return null;
  }  
}
