package com.bizunited.nebula.rbac.local.service.notifier;

import java.util.List;
import java.util.Set;

import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.web.servlet.mvc.method.RequestMappingInfo;

import com.bizunited.nebula.rbac.sdk.config.RbacCustomProperties;
import com.bizunited.nebula.rbac.sdk.service.CompetenceVoService;
import com.bizunited.nebula.rbac.sdk.vo.CompetenceVo;
import com.bizunited.nebula.security.sdk.constant.Constants;
import com.bizunited.nebula.security.sdk.event.AuthenticationRbacEventListener;
import com.google.common.collect.Lists;
import com.google.common.collect.Sets;

public class AuthenticationRbacEventListenerImpl implements AuthenticationRbacEventListener {
  @Autowired
  private RbacCustomProperties rbacCustomProperties;
  @Autowired
  private CompetenceVoService competenceVoService;

  /**
   * 从rbac模块的配置信息中获取“管理员角色”
   */
  @Override
  public Set<String> onRequestIgnoreMethodCheckRoles() {
    String[] ignoreMethodCheckRoles = this.rbacCustomProperties.getIgnoreMethodCheckRoles();
    if(ignoreMethodCheckRoles == null || ignoreMethodCheckRoles.length == 0) {
      return null;
    }
    return Sets.newHashSet(ignoreMethodCheckRoles);
  }

  /**
   * 该事件订阅基于当前请求URL功能在RBAC模块的配置中寻找这个请求对应的角色（多个）信息
   */
  @Override
  public Set<String> onRequestRoleCodes(List<RequestMappingInfo> matchedRequestMappings, String tenantCode, String currentMethod) {
    /*
     * 这个方法的核心含义是，一个指定的URL地址，需要什么角色（可以是多个）才能访问。
     * 如果没有设定，那么按照“不限制即同意”的原则，就至少包括匿名角色能够访问。确定功能路径绑定的角色过程如下：
     * 
     * 1、如果存在至少一个匹配路径，那么从数据库中找出这些对应的资源信息（可能不止一个）
     * 2、这里进行资源访问角色的过滤，将能够访问这些资源的角色写入结果集合
     * */
    Set<String> resultRoleCodes = Sets.newHashSet();

    // 如果没有找到任何处理器，说明这个URL地址并没有加入到权限控制体系，则不做权限过滤
    // 原则就是“未明确拒绝则是允许”
    // 1、=======
    List<CompetenceVo> currentCompetences = Lists.newArrayList();
    for (RequestMappingInfo requestMappingInfo : matchedRequestMappings) {
      Set<String> resources = requestMappingInfo.getPatternsCondition().getPatterns();
      if(resources == null || resources.isEmpty()) {
        continue;
      }
      for (String resource : resources) {
        Set<CompetenceVo> cacheCompetences = this.competenceVoService.findByResource(resource , 1);
        if(cacheCompetences != null && !cacheCompetences.isEmpty()) {
          currentCompetences.addAll(cacheCompetences);
        }
      }
    }
    // 如果没有匹配任何存储功能信息，则不再做权限过滤（但正常情况下，不会出现这样的问题）
    // 原则也是“未明确拒绝则是允许”
    if(currentCompetences.isEmpty()) {
      resultRoleCodes.add(Constants.ANONYMOUS);
      return resultRoleCodes;
    }
    
    // 2、===================    
    String matchedFix = tenantCode + "|";
    for (CompetenceVo competence : currentCompetences) {
      String methods = competence.getMethods();
      // 注意，方法类型也必须满足
      if(methods.indexOf(currentMethod) != -1) {
        // 取得当前功能和角色的绑定关系(注意这里每一条信息都是tenantCode_roleCode)，且使用“,”进行了分割
        String tenantAndRoleCodes = competence.getRoles();
        if(StringUtils.isBlank(tenantAndRoleCodes)) {
          continue;
        }
        String[] tenantAndRoleCodeArray = StringUtils.split(tenantAndRoleCodes, ",");
        if(tenantAndRoleCodeArray == null || tenantAndRoleCodeArray.length == 0) {
          continue;
        }
        // 将角色添加到返回值中，证明当前系统中已设定当前角色可以访该功能
        // 每一条记录都是由tenantCode_roleCode构成
        for (String tenantAndRoleCode: tenantAndRoleCodeArray) {
          if(StringUtils.indexOf(tenantAndRoleCode, matchedFix) != 0) {
            continue;
          }
          String currentRoleCode = StringUtils.substringAfterLast(tenantAndRoleCode, matchedFix);
          resultRoleCodes.add(currentRoleCode);
        }
      }
    }
    
    return resultRoleCodes;
  }
}
