package com.bizunited.nebula.rbac.local.service.strategy;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Set;
import java.util.stream.Collectors;

import org.apache.commons.lang3.StringUtils;
import org.springframework.util.CollectionUtils;

import com.bizunited.nebula.rbac.sdk.vo.ButtonVo;
import com.bizunited.nebula.rbac.sdk.vo.CompetenceVo;
import com.google.common.collect.Lists;
import com.google.common.collect.Sets;

/**
 * 可以指定功能路径，功能方式，功能状态和绑定角色，作为查询条件，进行全功能树遍历查询的处理过程
 * @author yinwenjie
 */
// TODO 注意，区分扁平化遍历，和树形遍历
public class QueryByResourcesAndMethodAndRoleCodes extends AbstractCompetenceQueryStrategy implements CompetenceQueryStrategy {

  /**
   * 必须进行条件限制的一个或者多个功能路径信息
   */
  private String[] resources;
  /**
   * 是否进行模糊匹配
   */
  private boolean like;
  /**
   * 可能传入的方法类型比较（如：POST|GET等等）
   */
  private String method;
  /**
   * 可能作为查询条件的功能、按钮状态信息；如果不传入说明任何状态都行
   */
  private Integer status;
  /**
   * 可能作为查询条件的租户和角色编号组
   */
  private String tenantCode;
  /**
   * 可能作为查询条件的
   */
  private String[] roleCodes; 
  /**
   * 指定的是否作为管理员角色进行过滤判定
   */
  private boolean isAdmin;
  /**
   * 可忽略方法级判定的角色名
   */
  private String[] ignoreMethodCheckRoles; 
  
  /**
   * @param resources {@link #resources}
   * @param like {@link #like}
   * @param method {@link #method}
   * @param status {@link #status}
   * @param tenantCode {@link #tenantCode}
   * @param roleCodes {@link #roleCodes}
   * @param ignoreMethodCheckRoles {@link #ignoreMethodCheckRoles}
   */
  public QueryByResourcesAndMethodAndRoleCodes(String[] resources, boolean like,
      String method, Integer status, String tenantCode, String[] roleCodes, boolean isAdmin, String[] ignoreMethodCheckRoles) {
    this.resources = resources;
    if(resources == null || resources.length ==0) {
      throw new IllegalArgumentException("进行菜单/功能匹配时，必须传入基于比较的功能信息");
    }
    this.like = like;
    this.method = method;
    this.status = status;
    this.tenantCode = tenantCode;
    this.roleCodes = roleCodes;
    this.isAdmin = isAdmin;
    this.ignoreMethodCheckRoles = ignoreMethodCheckRoles;
  }

  @Override
  public CompetenceVo filterCompetence(CompetenceVo competence) {
    String targetResource = competence.getResource();
    String targetMethods = competence.getMethods();
    Integer targetStatus = competence.getTstatus();
    // 注意缓存中roles的格式 tenantcode_roleCode，并以","分割
    String targetRoles = competence.getRoles();
    
    /*
     * 过滤过程为：
     * 1、然后根据可能的方法类型、状态匹配进行过滤
     * 2、首先根据必须的resources信息进行过滤，注意依据like的不同判断方式还不同
     * 3、最后依据可能的tenantCode和RoleCodes进行过滤，注意，根据是否是管理员角色，判断方式还不同
     * */
    
    // 1、========
    boolean canContinue = false;
    if(StringUtils.isAnyBlank(targetResource , targetMethods) || targetStatus == null) {
      return null;
    } 
    // 比较可能的方法类型
    if(!StringUtils.isBlank(this.method) && StringUtils.indexOf(targetMethods, this.method) == -1) {
      return null;
    }
    // 可能的状态过来条件
    if(this.status != null && this.status.intValue() != targetStatus.intValue()) {
      return null;
    }
    
    // 2、========
    CompetenceVo copyCompetence = this.copyCompetence(competence);
    String[] matchedRolesReuslt = new String[] {};
    for (String resource : this.resources) {
      // 如果条件成立那么使用精确匹配，否则就是模糊匹配
      // TODO 不知道为什么还有模糊查询。。。。
      if(!like && StringUtils.equals(resource, targetResource)) {
        canContinue = true;
        break;
      } else if(like && StringUtils.indexOf(resource, targetResource) != -1) {
        canContinue = true;
        break;
      }
    }
    if(!canContinue) {
      return null;
    }
    
    // 3、=======
    // 注意：运行到这里，canContinue一定为true
    // 如果条件成立，则说明需要按照tenantCode和roleCodes进行过滤
    if(!this.isAdmin && StringUtils.isNotBlank(tenantCode) && roleCodes != null && roleCodes.length > 0) {
      if(!StringUtils.isEmpty(targetRoles)) {
        List<String> matchTargetRoles = Lists.newArrayList();
        for (String roleCode : roleCodes) {
          matchTargetRoles.add(StringUtils.join(tenantCode , "|" , roleCode));
        }
        String[] matchTargetRoleArray = matchTargetRoles.toArray(new String[] {});
        String[] matchedRoles = Arrays.stream(StringUtils.split(targetRoles, ",")).filter(item -> StringUtils.equalsAny(item, matchTargetRoleArray)).toArray(String[]::new);
        // 如果条件成立，说明并没有匹配到任何角色,则不进行返回
        if(matchedRoles == null || matchedRoles.length == 0) {
          canContinue = false;
        } else {
          matchedRolesReuslt = Arrays.stream(matchedRoles).map(item -> StringUtils.split(item , "|")[1]).toArray(String[]::new);
        }
      } else {
        canContinue = false;
      }
    } 
    // 如果认定是通过管理员权限进行的过滤，则返回的CompetenceVo绑定ignoreMethodCheckRoles
    else if(this.isAdmin) {
      matchedRolesReuslt = this.ignoreMethodCheckRoles;
    } 
    if(!canContinue) {
      return null;
    }
    
    // 开始进行返回
    if(matchedRolesReuslt != null && matchedRolesReuslt.length > 0) {
      copyCompetence.setRoles(StringUtils.join(matchedRolesReuslt , ","));
    }
    return copyCompetence;
  }

  @Override
  public ButtonVo filterButton(ButtonVo button) {
    /*
     * 过滤过程和QueryStrategyByResourcesAndMethodAndRoleCodes.filterCompetence(CompetenceVo competence)类似
     * */
    boolean effective = button.getEffective();
    // 注意缓存中roles的格式 tenantcode_roleCode，并以","分割
    Set<String> targetRoles = button.getRoles();
    // 可能的状态过来条件
    if(this.status != null && this.status.intValue() != (effective?1:0)) {
      return null;
    }
    ButtonVo copyButton = this.nebulaToolkitService.copyObjectByWhiteList(button, ButtonVo.class, LinkedHashSet.class, ArrayList.class); 
    
    // 以下是匹配可能额角色信息
    Set<String> matchedRolesReuslt = Sets.newLinkedHashSet();
    if(!this.isAdmin && StringUtils.isNotBlank(tenantCode) && roleCodes != null && roleCodes.length > 0) {
      if(!CollectionUtils.isEmpty(targetRoles)) {
        List<String> matchTargetRoles = Lists.newArrayList();
        for (String roleCode : roleCodes) {
          matchTargetRoles.add(StringUtils.join(tenantCode , "|" , roleCode));
        }
        String[] matchTargetRoleArray = matchTargetRoles.toArray(new String[] {});
        String[] matchedRoles = targetRoles.stream().filter(item -> StringUtils.equalsAny(item, matchTargetRoleArray)).toArray(String[]::new);
        // 如果条件成立，说明并没有匹配到任何角色,则不进行返回
        if(matchedRoles == null || matchedRoles.length == 0) {
          return null;
        } else {
          matchedRolesReuslt = Arrays.stream(matchedRoles).map(item -> StringUtils.split(item , "|")[1]).collect(Collectors.toSet());
        }
      } else {
        return null;
      }
    } // 如果认定是通过管理员权限进行的过滤，则返回的ButtonVo绑定ignoreMethodCheckRoles
    else if(this.isAdmin) {
      matchedRolesReuslt = Sets.newHashSet(this.ignoreMethodCheckRoles);
    } 
    // 其它情况，说明没有使用角色进行过滤（或者角色过滤条件不起作用）
    else {
      matchedRolesReuslt = targetRoles;
    }
    copyButton.setRoles(matchedRolesReuslt);
    return copyButton;
  }
}
