package com.bizunited.nebula.rbac.local.service.strategy;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Set;
import java.util.stream.Collectors;

import org.apache.commons.lang3.StringUtils;
import org.springframework.util.CollectionUtils;

import com.bizunited.nebula.rbac.sdk.vo.ButtonVo;
import com.bizunited.nebula.rbac.sdk.vo.CompetenceVo;
import com.google.common.collect.Lists;
import com.google.common.collect.Sets;

/**
 * 可以指定菜单/功能的分类，并制定角色、节点状态，作为查询条件
 * @author yinwenjie
 */
public class QueryByViewItemAndRoleCodes extends AbstractCompetenceQueryStrategy implements CompetenceQueryStrategy {

  private Boolean viewItem;
  
  private boolean isAdmin;
  
  private String tenantCode;
  
  private String[] roleCodes;
  
  private Integer status;
  
  /**
   * 可忽略方法级判定的角色名
   */
  private String[] ignoreMethodCheckRoles; 
  
  /**
   * @param viewItem 菜单/功能部分的条件过滤
   * @param isAdmin 当前角色是否拥有管理员角色的过滤
   * @param tenantCode 当前租户的过滤(必须传入)
   * @param roleCodes 当前角色业务编号的过滤（注意大小写，必须传入）
   * @param status 节点状态的过滤
   * @param ignoreMethodCheckRoles 既是在系统环境设定的管理员的角色业务编号（可以是多个）
   */
  public QueryByViewItemAndRoleCodes(Boolean viewItem, boolean isAdmin, String tenantCode, String[] roleCodes, Integer status , String[] ignoreMethodCheckRoles) {
    this.viewItem = viewItem;
    this.isAdmin = isAdmin;
    this.tenantCode = tenantCode;
    this.roleCodes = roleCodes;
    this.status = status;
    this.ignoreMethodCheckRoles = ignoreMethodCheckRoles;
  }

  @Override
  public CompetenceVo filterCompetence(CompetenceVo competence) {
    /*
     * 1、首先进行节点自身可用性、性质的过滤，如果不满足，则不返回
     * 2、然后进行节点所属角色的过滤，如果角色过滤不满足，则不返回
     * */
    // 1、=======
    Integer competenceStatus = competence.getTstatus();
    Integer competenceViewItem = competence.getViewItem();
    boolean canContinue = true;
    // 如果仅要求进行菜单部分的查询时
    if(this.viewItem != null && this.viewItem == true && competenceViewItem != 1) {
      canContinue = false;
    }
    // 如果仅要求进行状态正常的菜单/功能/按钮查询时
    if(this.status != null && this.status == 1 && competenceStatus != 1) {
      canContinue = false;
    }
    if(!canContinue) {
      return null;
    }
    
    // 2、======
    CompetenceVo copyCompetence = this.copyCompetence(competence);
    String[] matchedRolesReuslt = new String[] {};
    // 如果不是管理员，则还需要进行菜单/功能角色匹配部分的过滤
    if(!isAdmin) {
      String competenceRoles = competence.getRoles();
      // 根据tenantCode和roleCodes构建所有可能的角色匹配条件
      if(!StringUtils.isEmpty(competenceRoles) && StringUtils.isNotBlank(tenantCode) && roleCodes != null && roleCodes.length > 0) {
        List<String> matchTargetRoles = Lists.newArrayList();
        for (String roleCode : roleCodes) {
          matchTargetRoles.add(StringUtils.join(tenantCode , "|" , roleCode));
        }
        String[] matchTargetRoleArray = matchTargetRoles.toArray(new String[] {});
        String[] matchedRoles = Arrays.stream(StringUtils.split(competenceRoles, ",")).filter(item -> StringUtils.equalsAny(item, matchTargetRoleArray)).toArray(String[]::new);
        // 如果条件成立，说明并没有匹配到任何角色
        if(matchedRoles == null || matchedRoles.length == 0) {
          canContinue = false;
        } else {
          matchedRolesReuslt = Arrays.stream(matchedRoles).map(item -> StringUtils.split(item , "|")[1]).toArray(String[]::new);
        }
      }
      // 否则，说明入参（tenantCode和roleCodes）存在问题
      else {
        canContinue = false;
      }
    } 
    // 否则就是管理员场景，那么匹配的角色直接写为ignoreMethodCheckRoles部分的角色
    else {
      matchedRolesReuslt = this.ignoreMethodCheckRoles;
    }
    if(!canContinue) {
      return null;
    }
    copyCompetence.setRoles(StringUtils.join(matchedRolesReuslt , ","));
    return copyCompetence;
  }

  @Override
  public ButtonVo filterButton(ButtonVo button) {
    boolean effective = button.getEffective();
    // 如果仅要求进行状态正常的菜单/功能/按钮查询时
    boolean canContinue = true;
    if(status != null && status == 1 && !effective) {
      canContinue = false;
    }
    if(!canContinue) {
      return null;
    }
    ButtonVo copyButton = this.nebulaToolkitService.copyObjectByWhiteList(button, ButtonVo.class, LinkedHashSet.class, ArrayList.class); 
    // 下面进行角色匹配
    Set<String> matchedRolesReuslt = Sets.newLinkedHashSet();
    // 如果不是管理员，则还需要进行菜单/功能角色匹配部分的过滤
    if(!isAdmin) {
      Set<String> buttonRoles = button.getRoles();
      // 根据tenantCode和roleCodes构建所有可能的角色匹配条件
      if(!CollectionUtils.isEmpty(buttonRoles) && StringUtils.isNotBlank(tenantCode) && roleCodes != null && roleCodes.length > 0) {
        List<String> matchTargetRoles = Lists.newArrayList();
        for (String roleCode : roleCodes) {
          matchTargetRoles.add(StringUtils.join(tenantCode , "|" , roleCode));
        }
        String[] matchTargetRoleArray = matchTargetRoles.toArray(new String[] {});
        String[] matchedRoles = buttonRoles.stream().filter(item -> StringUtils.equalsAny(item, matchTargetRoleArray)).toArray(String[]::new);
        // 如果条件成立，说明并没有匹配到任何角色
        if(matchedRoles == null || matchedRoles.length == 0) {
          canContinue = false;
        } else {
          matchedRolesReuslt = Arrays.stream(matchedRoles).map(item -> StringUtils.split(item , "|")[1]).collect(Collectors.toSet());
        }
      }
      // 否则，说明入参（tenantCode和roleCodes）存在问题
      else {
        canContinue = false;
      }
    } else {
      matchedRolesReuslt = Sets.newHashSet(this.ignoreMethodCheckRoles);
    }
    copyButton.setRoles(matchedRolesReuslt);
    return copyButton;
  }
}