package com.bizunited.nebula.security.local.config;

import com.bizunited.nebula.common.filters.AppFilter;
import com.bizunited.nebula.common.filters.TenantFilter;
import com.bizunited.nebula.security.local.notifier.DefaultAuthenticationRbacEventListener;
import com.bizunited.nebula.security.local.notifier.DefaultAuthenticationUserEventListener;
import com.bizunited.nebula.security.local.service.CustomAccessDecisionManager;
import com.bizunited.nebula.security.local.service.CustomFilterInvocationSecurityMetadataSource;
import com.bizunited.nebula.security.local.service.CustomFilterSecurityInterceptor;
import com.bizunited.nebula.security.local.service.SimpleAuthenticationDetailsSource;
import com.bizunited.nebula.security.local.service.SimpleAuthenticationProvider;
import com.bizunited.nebula.security.local.service.handle.SimpleAccessDeniedHandler;
import com.bizunited.nebula.security.local.service.handle.SimpleAuthenticationFailureHandler;
import com.bizunited.nebula.security.local.service.handle.SimpleAuthenticationSuccessHandler;
import com.bizunited.nebula.security.sdk.config.NebulaWebSecurityConfigurerAdapter;
import com.bizunited.nebula.security.sdk.config.SimpleSecurityProperties;
import com.bizunited.nebula.security.sdk.event.AuthenticationRbacEventListener;
import com.bizunited.nebula.security.sdk.event.AuthenticationUserEventListener;
import com.bizunited.nebula.security.sdk.password.Aes2PasswordEncoder;
import com.google.common.collect.Lists;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.boot.autoconfigure.condition.ConditionalOnMissingBean;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.ComponentScan;
import org.springframework.context.annotation.Configuration;
import org.springframework.security.access.AccessDecisionManager;
import org.springframework.security.authentication.AuthenticationDetailsSource;
import org.springframework.security.config.annotation.authentication.builders.AuthenticationManagerBuilder;
import org.springframework.security.config.annotation.web.builders.HttpSecurity;
import org.springframework.security.config.annotation.web.configuration.EnableWebSecurity;
import org.springframework.security.config.annotation.web.configuration.WebSecurityConfigurerAdapter;
import org.springframework.security.crypto.password.PasswordEncoder;
import org.springframework.security.crypto.password.Pbkdf2PasswordEncoder;
import org.springframework.security.web.access.intercept.FilterInvocationSecurityMetadataSource;
import org.springframework.security.web.access.intercept.FilterSecurityInterceptor;
import org.springframework.security.web.authentication.AuthenticationFailureHandler;
import org.springframework.security.web.authentication.AuthenticationSuccessHandler;
import org.springframework.security.web.authentication.UsernamePasswordAuthenticationFilter;
import org.springframework.security.web.authentication.WebAuthenticationDetails;
import org.springframework.security.web.authentication.logout.LogoutSuccessHandler;
import org.springframework.web.cors.CorsConfiguration;
import org.springframework.web.cors.CorsConfigurationSource;
import org.springframework.web.cors.CorsUtils;
import org.springframework.web.cors.UrlBasedCorsConfigurationSource;

import javax.servlet.http.HttpServletRequest;
import java.util.ArrayList;
import java.util.Arrays;
/**
 * 和经典权限验证有关的配置信息在这里
 * @author yinwenjie
 */
@Configuration
@EnableWebSecurity
@ComponentScan(basePackages = {"com.bizunited.nebula.security"})
public class SecurityConfigAutoConfiguration extends WebSecurityConfigurerAdapter {
  /**
   * 日志
   */
  private static final Logger LOGGER = LoggerFactory.getLogger(SecurityConfigAutoConfiguration.class);
  @Autowired
  private SimpleSecurityProperties simpleSecurityProperties;
  @Autowired
  private AuthenticationSuccessHandler authenticationSuccessHandler;
  @Autowired 
  private AuthenticationFailureHandler authenticationFailureHandler;
  @Autowired 
  private LogoutSuccessHandler logoutSuccessHandler;
  @Autowired
  private AuthenticationDetailsSource<HttpServletRequest, WebAuthenticationDetails> authenticationDetailsSource;
  @Autowired
  private AccessDecisionManager accessDecisionManager;
  @Autowired
  private SimpleAuthenticationProvider authenticationProvider;
  @Autowired
  private FilterInvocationSecurityMetadataSource securityMetadataSource;
  @Autowired(required = false)
  private NebulaWebSecurityConfigurerAdapter nebulaWebSecurityConfigurerAdapter;
  @Autowired
  private AppFilter appFilter;
  @Autowired
  private TenantFilter tenantFilter;
  
  @Override
  protected void configure(HttpSecurity http) throws Exception {
    /*
     * 自定义的过滤器，这个过滤器的意思是：
     * 1、确定访问的资源具备什么样的角色访问权限(securityMetadataSource) -> 
     * 2、根据当前用户具备的哪些角色进行比对，最终确定是否能够访问指定的资源(accessDecisionManager) ->
     * 3、authenticationManager是投票策略，spring security支持多种投票策略，有多数派、有全部通过，那么这里默认使用的投票器是“只做一次投票”
     * 4、关于ignoreUrls的信息，还要增加系统默认就不加入功能权限的几个路径，例如：验证码和校验码的路径
     * */
    CustomFilterSecurityInterceptor filterSecurityInterceptor = new CustomFilterSecurityInterceptor(securityMetadataSource, accessDecisionManager, super.authenticationManager());
    CorsConfigurationSource configurationSource = this.corsConfigurationSource();
    // 无访问权限时的处理器
    SimpleAccessDeniedHandler simpleAccessDeniedHandler = new SimpleAccessDeniedHandler(simpleSecurityProperties);
    String[] ignoreUrls = this.simpleSecurityProperties.getIgnoreUrls();
    String loginUrl = this.simpleSecurityProperties.getLoginUrl();
    String logoutUrl = this.simpleSecurityProperties.getLogoutUrl();
    String loginPageUrl = this.simpleSecurityProperties.getLoginPageUrl();
    String logoutSuccessRedirect = this.simpleSecurityProperties.getLogoutSuccessRedirect();
    // 在忽略权限判断的设置中，增加几个系统默认的路径
    ArrayList<String> currentIgnoreUrls = new ArrayList<>();
    if(ignoreUrls != null && ignoreUrls.length > 0) {
      currentIgnoreUrls.addAll(Lists.newArrayList(ignoreUrls));
    }
    currentIgnoreUrls.addAll(Lists.newArrayList(SimpleSecurityProperties.DEFAULT_IGNOREURLS));
    if(StringUtils.isNotBlank(logoutSuccessRedirect)) {
      currentIgnoreUrls.add(logoutSuccessRedirect);
    }
    if(StringUtils.isNotBlank(loginPageUrl)) {
      currentIgnoreUrls.add(loginPageUrl);
    }
    http
      .addFilterAt(filterSecurityInterceptor,FilterSecurityInterceptor.class)
      // 允许登录操作时跨域
      .cors().configurationSource(configurationSource).and()
      // 允许iframe嵌入
      .headers().frameOptions().disable().and()
      // 设定显示jsession设定信息
      .sessionManagement()
        .enableSessionUrlRewriting(true).and()
      .authorizeRequests()
        //对preflight放行
        .requestMatchers(CorsUtils::isPreFlightRequest).permitAll()
        // 系统中的“登录页面”在被访问时，不进行权限控制
        .antMatchers(currentIgnoreUrls.toArray(new String[] {})).permitAll()
        // 其它访问都要验证权限
        .anyRequest().authenticated().and()
      // =================== 无权限访问资源或者用户登录信息过期时，会出发这个异常处理器
      .exceptionHandling()
        .authenticationEntryPoint(simpleAccessDeniedHandler)
        .accessDeniedHandler(simpleAccessDeniedHandler).and()
      // ==================== 设定登录页的url地址，它不进行权限控制
      .formLogin()
        // 由于后端提供的都是restful接口，并没有直接跳转的页面
        // 所以只要访问的url没有通过权限认证，就跳到这个请求上，并直接排除权限异常
        .loginPage("/v1/rbac/loginFail")
        // 登录请求点
        .loginProcessingUrl(loginUrl)
        // 一旦权限验证成功，则执行这个处理器
        .successHandler(this.authenticationSuccessHandler)
        // 一旦权限验证过程出现错误，则执行这个处理器
        .failureHandler(this.authenticationFailureHandler)
        // 自定义的登录表单信息（默认的登录表单结构只有两个字段，用户名和输入的密码）
        .authenticationDetailsSource(this.authenticationDetailsSource)
        // 由于使用前后端分离时，设定了failureHandler，所以failureForwardUrl就不需要设定了
        // 由于使用前后端分离时，设定了successHandler，所以successForwardUrl就不需要设定了
        .permitAll().and()
        // ===================== 设定登出后的url地址
      .logout()
        // 登出页面
        .logoutUrl(logoutUrl)
        .logoutSuccessHandler(logoutSuccessHandler).permitAll().and()
        // 由于使用前后端分离时，设定了logoutSuccessHandler，所以logoutSuccessUrl就不需要设定了
        // 登录成功后
        // ===================== 关闭csrf
      .csrf()
        .disable() 
      // 由于security的本质是filter，所以为了保证基础设施中的多应用首先起作用，即首先让AppFilter工作起来
      // 注意：TenantFilter不能先于security工作，因为系统登录前，因为系统登录前很可能没有确认二级多租户信息，要登录成功后，才能确认
      .addFilterBefore(this.appFilter, UsernamePasswordAuthenticationFilter.class)
      .addFilterAfter(this.tenantFilter, UsernamePasswordAuthenticationFilter.class);
    if(nebulaWebSecurityConfigurerAdapter != null) {
      nebulaWebSecurityConfigurerAdapter.configure(http);
    }
  }
  
  @Override
  protected void configure(AuthenticationManagerBuilder auth) throws Exception {
    auth.authenticationProvider(authenticationProvider);
  }
  
  @Bean("simpleAuthenticationSuccessHandler")
  @ConditionalOnMissingBean
  public AuthenticationSuccessHandler getAuthenticationSuccessHandler() {
    return new SimpleAuthenticationSuccessHandler();
  }
  
  @Bean("simpleAuthenticationFailureHandler")
  @ConditionalOnMissingBean
  public AuthenticationFailureHandler getAuthenticationFailureHandler() {
    return new SimpleAuthenticationFailureHandler();
  }
  
  /**
   * 用户密码的默认加密方式为PBKDF2加密
   * @return
   */
  @Bean(name="passwordEncoder")
  @ConditionalOnMissingBean
  public PasswordEncoder passwordEncoder() {
    return new Pbkdf2PasswordEncoder();
  }

  /**
   * 用户密码采用AES-CBC方式加密
   * @return
   */
  @Bean(name = "aes2PasswordEncoder")
  @ConditionalOnMissingBean
  public Aes2PasswordEncoder aes2PasswordEncoder() {
    return new Aes2PasswordEncoder();
  }

  @Bean(name="customAccessDecisionManager")
  @ConditionalOnMissingBean
  public AccessDecisionManager getCustomAccessDecisionManager() {
    return new CustomAccessDecisionManager();
  }
  @Bean(name="customFilterInvocationSecurityMetadataSource")
  @ConditionalOnMissingBean
  public FilterInvocationSecurityMetadataSource getCustomFilterInvocationSecurityMetadataSource() {
    return new CustomFilterInvocationSecurityMetadataSource();
  }
  /**
   * 该接口用来定义登录请求的数据获取方式
   * @return
   */
  @Bean(name="authenticationDetailsSource")
  @ConditionalOnMissingBean
  public AuthenticationDetailsSource<HttpServletRequest, WebAuthenticationDetails> getAuthenticationDetailsSource() {
    return new SimpleAuthenticationDetailsSource();
  }

  /**
   * 配置权限的跨域操作
   * @return
   */
  private CorsConfigurationSource corsConfigurationSource() {
    CorsConfiguration configuration = new CorsConfiguration();
    configuration.setAllowedOrigins(Arrays.asList("*"));
    configuration.setAllowedMethods(Arrays.asList("*"));
    configuration.setAllowedHeaders(Arrays.asList("*"));
    configuration.setAllowCredentials(true);
    UrlBasedCorsConfigurationSource source = new UrlBasedCorsConfigurationSource();
    source.registerCorsConfiguration("/**", configuration);
    return source;
  }
  
  // =========== 以下是两个默认实现
  @Bean
  @ConditionalOnMissingBean
  public AuthenticationRbacEventListener getDefaultAuthenticationRbacEventListener() {
    LOGGER.warn(" ==== 当前应用程序使用了默认的AuthenticationRbacEventListener监听接口实现com.bizunited.nebula.security.local.notifier.DefaultAuthenticationRbacEventListener，该实现默认所有的方法都可以访问，如果需要限制方法权限，请自行实现AuthenticationRbacEventListener监听接口（若已使用@Primary注解，则可以忽略该警告）!! ");
    return new DefaultAuthenticationRbacEventListener();
  }
  
  @Bean
  @ConditionalOnMissingBean
  public AuthenticationUserEventListener getDefaultAuthenticationUserEventListener() {
    LOGGER.warn(" ==== 当前应用程序使用了默认的AuthenticationUserEventListener监听接口实现com.bizunited.nebula.security.local.notifier.DefaultAuthenticationUserEventListener，该实现默认所有的操作者都拥有ADMIN角色，如需要限制用户的角色，请自行实现AuthenticationUserEventListener监听接口(若已使用@Primary注解，则可以忽略该警告)!! ");
    return new DefaultAuthenticationUserEventListener();
  }
}