package com.bizunited.nebula.security.local.notifier;

import java.util.ArrayList;
import java.util.List;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

import org.apache.commons.lang3.ArrayUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.security.core.Authentication;
import org.springframework.security.core.context.SecurityContext;
import org.springframework.security.core.context.SecurityContextHolder;
import org.springframework.security.web.util.matcher.AntPathRequestMatcher;
import org.springframework.security.web.util.matcher.OrRequestMatcher;
import org.springframework.security.web.util.matcher.RequestMatcher;
import org.springframework.util.CollectionUtils;

import com.bizunited.nebula.common.event.RequestTenantInfoEventListener;
import com.bizunited.nebula.common.util.tenant.TenantContextHolder;
import com.bizunited.nebula.common.vo.AbstractTenantInfo;
import com.bizunited.nebula.security.sdk.config.SimpleSecurityProperties;
import com.bizunited.nebula.security.sdk.vo.LoginDetails;
import com.bizunited.nebula.security.sdk.vo.SimpleTenantInfo;

/**
 * 按照下层基础设施-nebula——tenant组件的要求，
 * 本经典登录模块需要实现TenantRequestEventListener事件监听，以便告知nebula——tenant组件如何找到系统的tenantCode信息
 * 以及如何组装tencodeInfo信息
 * @author yinwenjie
 */
public class TenantRequestEventListenerForSecurity implements RequestTenantInfoEventListener { 

  @Autowired
  private SimpleSecurityProperties simpleSecurityProperties;
  
  @Override
  public AbstractTenantInfo onBuildTenantInfo(HttpServletRequest request, HttpServletResponse response) { 
    /*
     * 在默认的经典登录模块，获取当前登录用户的tenantCode信息，过程如下：
     * 试图取得当前登录者的基本信息（loginDetails），如果没有tenat
     * */
    SecurityContext securityContext = SecurityContextHolder.getContext();
    if(securityContext == null) {
      return null;
    }
    Authentication authentication = securityContext.getAuthentication();
    if(authentication == null) {
      return null;
    }
    LoginDetails loginDetails = (LoginDetails)authentication.getDetails();
    if(loginDetails == null) {
      return null;
    }
    TenantContextHolder.setTenant(loginDetails.getTenantCode());
    
    // 为什么这里默认的的SimpleTenantInfo不需要设置租户信息，可参见SimpleTenantInfo类的源代码
    SimpleTenantInfo simpleTenantInfo = new SimpleTenantInfo();
    return simpleTenantInfo;
  } 
  
  @Override 
  public boolean shouldNotTenantInfoFilter(HttpServletRequest request) { 
    // 凡是存在于IgnoreUrls中的url就不进行tenantCode过滤。
    // 但是有例外，就是存在于IgnoreUrls配置中的，但是又在ignoreUrlsButFilterTenantCode配置中要求进行tenantFilter过滤的
    String[] ignoreUrls = simpleSecurityProperties.getIgnoreUrls();
    String[] ignoreUrlsButFilterTenantCode = simpleSecurityProperties.getIgnoreUrlsButFilterTenantCode();
    if (ArrayUtils.isEmpty(ignoreUrls)) {
      return false;
    }
    List<RequestMatcher> ignoreUrlmatchers = new ArrayList<>();
    List<RequestMatcher> ignoreUrlsButFilterTenantCodeMatchers = new ArrayList<>();
    for (String pattern : ignoreUrls) {
      ignoreUrlmatchers.add(new AntPathRequestMatcher(pattern, null));
    }
    if(!ArrayUtils.isEmpty(ignoreUrlsButFilterTenantCode)) {
      for (String pattern : ignoreUrlsButFilterTenantCode) {
        ignoreUrlsButFilterTenantCodeMatchers.add(new AntPathRequestMatcher(pattern, null));
      }
    }
    
    // ====== 开始进行排查
    OrRequestMatcher matcher = new OrRequestMatcher(ignoreUrlmatchers);
    OrRequestMatcher butFilterTenantCodeMatcher = null;
    if(!CollectionUtils.isEmpty(ignoreUrlsButFilterTenantCodeMatchers)) {
      butFilterTenantCodeMatcher = new OrRequestMatcher(ignoreUrlsButFilterTenantCodeMatchers);
    }
    if (matcher.matches(request)) {
      // 还要进行ignoreUrlsButFilterTenantCode配置项的排除
      if(butFilterTenantCodeMatcher != null) {
        return butFilterTenantCodeMatcher.matches(request);
      }
    }
    return false;
  }

  @Override
  public boolean shouldNotAppFilter(HttpServletRequest request) {
    // 凡是存在于IgnoreUrls中的url就不进行appCode过滤
    // 但是有例外，就是存在于IgnoreUrls配置中的，但是又在ignoreUrlsButFilterAppCode配置中要求进行tenantFilter过滤的
    String[] ignoreUrls = simpleSecurityProperties.getIgnoreUrls();
    String[] ignoreUrlsButFilterAppCode = simpleSecurityProperties.getIgnoreUrlsButFilterAppCode();
    if (ArrayUtils.isEmpty(ignoreUrls)) {
      return false;
    }
    List<RequestMatcher> ignoreUrlmatchers = new ArrayList<>();
    List<RequestMatcher> ignoreUrlsButFilterAppCodeMatchers = new ArrayList<>();
    for (String pattern : ignoreUrls) {
      ignoreUrlmatchers.add(new AntPathRequestMatcher(pattern, null));
    }
    if(!ArrayUtils.isEmpty(ignoreUrlsButFilterAppCode)) {
      for (String pattern : ignoreUrlsButFilterAppCode) {
        ignoreUrlsButFilterAppCodeMatchers.add(new AntPathRequestMatcher(pattern, null));
      }
    }
    
    // ====== 开始进行排查
    OrRequestMatcher matcher = new OrRequestMatcher(ignoreUrlmatchers);
    OrRequestMatcher butFilterAppCodeMatcher = null;
    if(!CollectionUtils.isEmpty(ignoreUrlsButFilterAppCodeMatchers)) {
      butFilterAppCodeMatcher = new OrRequestMatcher(ignoreUrlsButFilterAppCodeMatchers);
    }
    if (matcher.matches(request)) {
      // 还要进行ignoreUrlsButFilterAppCode配置项的排除
      if(butFilterAppCodeMatcher != null) {
        return butFilterAppCodeMatcher.matches(request);
      }
    }
    return false;
  }
}