package com.bizunited.nebula.security.local.service.handle;

import com.alibaba.fastjson.JSONObject;
import com.bizunited.nebula.common.controller.model.ResponseCode;
import com.bizunited.nebula.common.controller.model.ResponseModel;
import com.bizunited.nebula.common.util.tenant.TenantContextHolder;
import com.bizunited.nebula.security.sdk.event.AuthenticatedEventListener;
import com.bizunited.nebula.security.sdk.vo.LoginDetails;

import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.security.core.Authentication;
import org.springframework.security.core.GrantedAuthority;
import org.springframework.security.web.authentication.SavedRequestAwareAuthenticationSuccessHandler;
import org.springframework.util.CollectionUtils;

import javax.servlet.ServletException;
import javax.servlet.http.Cookie;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import javax.servlet.http.HttpSession;
import javax.transaction.Transactional;
import java.io.IOException;
import java.util.Collection;

/**
 * 鉴权成功（无论哪种鉴权模式），将执行这个处理器：</p>
 * 当登陆成功后，默认跳转到这个URL，并且返回登录成功后的用户基本信息一旦登录成功，服务端将会向客户端返回两个重要属性：</br>
 * 1、SESSION属性：该属性可保证从最后一次服务器请求开始的30分钟内登录信息有效(前提是服务节点没有重启)</br>
 * 2、persistence属性：该属性可保证从最后一次登录操作开始的100天内登录信息有效(前提是服务节点没有重启)</br>
 * 客户端应该至少保证在进行HTTP请求时，向服务器传递persistence属性。但为了保证服务端业务处理性能，应该两个属性都进行传递。</br></br>
 * 请注意：正常情况下SESSION属性可能发生变化，一旦变化客户端应该向服务端传递最新的SESSION属性</br>
 * 一旦登录成功或者通过persistence自动重登录成功后，服务端将通过以下两种方式向客户端返回新的SESSION属性和persistence属性：</br>
 * 1、http response的header信息中，将携带。类似于：persistence =YWRtaW46MTUxNDUxNzA4MjYzNjplYzI0OTFlYWEyNDhkZmIyZWIyNjNjODc3YzM2M2Q0MA 和 SESSION =54fd02c7-4067-43c9-94f8-5f6e474cd858</br>
 * 2、http response的cookies信息中，将携带。（此种方式是推荐使用的方式）</br></br>
 * 注意：以上描述只限于登录成功后的返回信息，并不是说每一次业务请求操作，服务端都会向客户端这样返回SESSION属性和persistence属性</br>
 * 为了保证服务端能够正确识别到客户端已登录的用户权限信息，在正常的前端请求过程中，每一次客户端的请求都需要向服务端发送SESSION属性（非必要，但推荐）和persistence属性</br>
 * 客户端可以使用以下方式，向服务端发送SESSION属性和persistence属性：</br>
 * 1、直接使用http request的cookies发送。（此种方式是推荐使用的方式）</br></br>
 * @author yinwenjie
 */
public class SimpleAuthenticationSuccessHandler extends SavedRequestAwareAuthenticationSuccessHandler implements HandleOutPut {
  
  /**
   * 事件监听
   */
  @Autowired(required = false)
  private AuthenticatedEventListener authenticatedEventListener;
  
  @Override
  @Transactional
  public void onAuthenticationSuccess(HttpServletRequest request, HttpServletResponse response, Authentication authentication) throws IOException, ServletException {
    /*
     * 一旦登录操作鉴权成功，需要完成以下工作：
     * 1、激活可能已经注册的authenticationEventListener监听事件
     * 2、首选按照要求，为登录操作的返回信息写入特定的head属性
     * 
     * */

    // 1、======
    LoginDetails loginDetails = (LoginDetails)authentication.getDetails();
    String tenantCode = loginDetails.getTenantCode();
    TenantContextHolder.setTenant(tenantCode);
    if(this.authenticatedEventListener != null) {
      this.authenticatedEventListener.onAuthenticationSuccess(loginDetails, authentication);
    }
    
    // 2、======
    // 由于前端模块的要求，在用户登录成功，或者通过persistence自动重登录成功后，
    // 后端服务都需要将cookies中重新设定的persistence和JSESSIONID以自定义属性的形式写入到head中。
    Collection<String> setCookies = response.getHeaders("Set-Cookie");
    String jsession = null;
    if(setCookies != null) {
      for (String setCookie : setCookies) {
        if(StringUtils.indexOf(setCookie, "persistence=") != -1) {
          int indexd = setCookie.indexOf('=');
          String value = setCookie.substring(indexd+1);
          response.setHeader("persistence", value);
        } else if(StringUtils.indexOf(setCookie, "JSESSIONID=") != -1) {
          int indexd = setCookie.indexOf('=');
          String value = setCookie.substring(indexd+1);
          jsession = value;
        }
      }
      response.setHeader("Access-Control-Expose-Headers", "JSESSIONID,persistence,Cookie");
    }
    // 从spring session取出相关信息——如果Security没有取到
    if(StringUtils.isBlank(jsession)) {
      HttpSession session = request.getSession();
      jsession = session.getId();
      Cookie cookie = new Cookie("JSESSIONID", jsession);
      cookie.setSecure(true);
      response.addCookie(cookie);
      session.setAttribute("tenantCode", tenantCode);
    }
    response.setHeader("JSESSIONID", jsession);
    // 返回的信息包括了当前用户的租户信息、账户信息和角色信息
    ResponseModel result = new ResponseModel(System.currentTimeMillis(), null, ResponseCode.E0, null);
    JSONObject resultJson = new JSONObject();
    String[] roles = null;
    Collection<? extends GrantedAuthority> authoritys = authentication.getAuthorities();
    if(!CollectionUtils.isEmpty(authoritys)) {
      roles = authoritys.stream().map(GrantedAuthority::getAuthority).toArray(String[]::new);
    }
    // 构建json
    resultJson.put("account", loginDetails.getAccount());
    resultJson.put("phone", loginDetails.getPhone());
    resultJson.put("tenantCode", loginDetails.getTenantCode());
    resultJson.put("username", loginDetails.getUsername());
    resultJson.put("roles", roles);
    result.setData(resultJson);
    this.writeResponse(response, result);
  }
}
