package com.bizunited.nebula.task.local.configuration;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.boot.autoconfigure.domain.EntityScan;
import org.springframework.context.ApplicationContext;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.ComponentScan;
import org.springframework.context.annotation.Configuration;
import org.springframework.data.jpa.repository.config.EnableJpaRepositories;
import org.springframework.scheduling.concurrent.ThreadPoolTaskScheduler;
import org.springframework.stereotype.Component;

import java.util.concurrent.LinkedBlockingQueue;
import java.util.concurrent.ThreadFactory;
import java.util.concurrent.ThreadPoolExecutor;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicInteger;

/**
 * 动态任务管理器的配置，动态任务管理器所管理的任务全部通过groovy进行驱动，并且可以实现动态启停、错误日志记录等操作
 * @author yinwenjie
 */
@Configuration
@EntityScan(basePackages = "com.bizunited.nebula.task.local.entity")
@EnableJpaRepositories(basePackages = "com.bizunited.nebula.task.local.repository")
@ComponentScan(basePackages = {"com.bizunited.nebula.task"})
@Component("_schedulerConfig")
public class SchedulerAutoConfiguration {
  @Autowired
  private ApplicationContext applicationContext;
  
  /**
   * Master节点状态保持者线程池
   * @return
   */
  @Bean("platform_dTaskMasterKeeperThreadExecutor")
  public ThreadPoolExecutor getDynamicTaskMasterKeeperThreadExecutor() {
    ThreadPoolExecutor dynamicTaskMasterKeeperThreadExecutor = new ThreadPoolExecutor(1, 1, 10, TimeUnit.SECONDS, new LinkedBlockingQueue<>(), new ThreadFactory() {
      private AtomicInteger count = new AtomicInteger(0);
      @Override
      public Thread newThread(Runnable r) {
        Thread thread = new Thread(r, "platform_DynamicTask_MasterKeeperThread_" + count.incrementAndGet());
        ClassLoader classLoader = applicationContext.getClassLoader();
        thread.setContextClassLoader(classLoader);
        thread.setDaemon(false);
        return thread;
      }
    });
    return dynamicTaskMasterKeeperThreadExecutor;
  }
  
  /**
   * 周期性调度线程池，其中的任务负责周期性刷新动态任务运行状态
   * @return 
   */
  @Bean("platform_dynamicTaskLoadingThreadScheduler")
  public ThreadPoolTaskScheduler getInitProcessThreadPoolExecutor() {
    ThreadPoolTaskScheduler scheduler = new ThreadPoolTaskScheduler();
    // 配置参数暂时不开放给开发人员
    scheduler.setBeanName("platform_dynamicTaskMSThreadScheduler");
    scheduler.setDaemon(false);
    scheduler.setPoolSize(50);
    scheduler.setThreadFactory(new ThreadFactory() {
      private AtomicInteger count = new AtomicInteger(1);
      @Override
      public Thread newThread(Runnable r) {
        Thread currentThread = new Thread(r);
        ClassLoader classLoader = applicationContext.getClassLoader();
        currentThread.setDaemon(false);
        currentThread.setName("platform_DynamicTask_LoadingThread_" + count.getAndIncrement());
        currentThread.setContextClassLoader(classLoader);
        return currentThread;
      }
    });
    scheduler.setThreadNamePrefix("platform_dynamicTaskMSThreadScheduler_");
    return scheduler;
  }
  
  /**
   * 实际任务运行调度线程池
   * @return
   */
  @Bean("platform_dynamicTaskScheduler")
  public ThreadPoolTaskScheduler getThreadPoolTaskScheduler() {
    ThreadPoolTaskScheduler scheduler = new ThreadPoolTaskScheduler();
    // 配置参数暂时不开放给开发人员
    scheduler.setBeanName("platform_dynamicTaskScheduler");
    scheduler.setDaemon(false);
    scheduler.setPoolSize(50);
    scheduler.setThreadFactory(new ThreadFactory() {
      private AtomicInteger count = new AtomicInteger(1);
      @Override
      public Thread newThread(Runnable r) {
        Thread currentThread = new Thread(r);
        ClassLoader classLoader = applicationContext.getClassLoader();
        currentThread.setDaemon(false);
        currentThread.setName("platform_DynamicTask_" + count.getAndIncrement());
        currentThread.setContextClassLoader(classLoader);
        return currentThread;
      }
    });
    scheduler.setThreadNamePrefix("platform_");
    return scheduler;
  }
}