package com.bizunited.nebula.task.local.repository;

import com.bizunited.nebula.task.local.entity.DynamicTaskSchedulerEntity;
import com.bizunited.nebula.task.local.repository.internal.DynamicTaskSchedulerRepositoryCustom;

import org.springframework.data.jpa.repository.JpaRepository;
import org.springframework.data.jpa.repository.JpaSpecificationExecutor;
import org.springframework.data.jpa.repository.Modifying;
import org.springframework.data.jpa.repository.Query;
import org.springframework.data.repository.query.Param;
import org.springframework.stereotype.Repository;

import java.util.Date;
import java.util.List;
import java.util.Set;

/**
 * 动态任务持久化数据层的处理服务
 * @author yinwenjie
 */
@Repository("DynamicTaskSchedulerRepository")
public interface DynamicTaskSchedulerRepository
        extends
        JpaRepository<DynamicTaskSchedulerEntity, String> ,
        JpaSpecificationExecutor<DynamicTaskSchedulerEntity> ,
        DynamicTaskSchedulerRepositoryCustom {

  /**
   * 按照指定的任务状态和工作状态查询符合条件的动态任务信息，并且按照任务code进行排序
   * @param tstatus 指定的动态任务状态
   * @param workingStatuses 指定的动态任务工作状态（可以是多个）
   * @param applicationName 对应的应用程序名称——一般通过spring boot进行配置
   * @param appCode 对应的appCode顶级租户信息
   * @return
   */
  @Query("from DynamicTaskSchedulerEntity dts where dts.tstatus = :tstatus AND dts.appCode = :appCode AND dts.applicationName = :applicationName AND dts.workingStatus in (:workingStatuses) order by dts.taskCode ")
  public Set<DynamicTaskSchedulerEntity> findByTstatusAndWorkingStatusAndApplicationNameAndAppCode(@Param("tstatus") int tstatus, @Param("workingStatuses") int[] workingStatuses , @Param("applicationName") String applicationName, @Param("appCode") String appCode);
  /**
   * 按照指定的定时器任务内容的执行方式和当前应用服务名，对任务进行分类查找：1：groovy脚本执行器；2：基于注解的Java method
   * @param invokeType 1：groovy脚本执行器；2：基于注解的Java method
   * @param appCode 当前应用服务名
   * @return
   */
  public Set<DynamicTaskSchedulerEntity> findByInvokeTypeAndApplicationNameAndAppCode(@Param("invokeType") int invokeType, @Param("applicationName") String applicationName, @Param("appCode") String appCode);
  /**
   * 按照指定的任务状态查询符合要求的动态任务信息，并按照任务code进行排序
   * @param tstatus 指定的动态任务状态
   * @param appCode 当前应用服务名
   * @return
   */
  @Query("from DynamicTaskSchedulerEntity dts where dts.tstatus = :tstatus AND dts.applicationName = :applicationName AND dts.appCode = :appCode order by dts.taskCode ")
  public Set<DynamicTaskSchedulerEntity> findByTstatusAndApplicationNameAndAppCode(@Param("tstatus") int tstatus, @Param("applicationName") String applicationName, @Param("appCode") String appCode);
  /**
   * 按照指定的任务运行状态查询符合要求的动态任务信息，并按照任务code进行排序
   * @param workingStatuses 指定的动态任务工作状态（可以是多个）
   * @param appCode 当前应用服务名
   * @return
   */
  @Query("from DynamicTaskSchedulerEntity dts where dts.applicationName = :applicationName AND dts.appCode = :appCode AND dts.workingStatus in (:workingStatuses) order by dts.taskCode ")
  public Set<DynamicTaskSchedulerEntity> findByWorkingStatusAndApplicationNameAndAppCode(@Param("workingStatuses") int[] workingStatuses, @Param("applicationName") String applicationName, @Param("appCode") String appCode);
  /**
   * 查询任务编号对应的动态任务信息，也包括可能的创建者、修改者信息
   * @param taskCode 指定的任务编号
   * @return
   */
  @Query("from DynamicTaskSchedulerEntity dts where dts.taskCode = :taskCode AND dts.applicationName = :applicationName AND  dts.appCode = :appCode ")
  public DynamicTaskSchedulerEntity findByTaskCodeAndApplicationNameAndAppCode(@Param("taskCode") String taskCode, @Param("applicationName") String applicationName, @Param("appCode") String appCode);
  /**
   * 查询任务编号对应的动态任务信息(多个)，也包括可能的创建者、修改者信息
   * @param taskCode 指定的任务编号
   * @return
   */
  @Query("from DynamicTaskSchedulerEntity dts where dts.taskCode in :taskCodes AND dts.applicationName = :applicationName AND  dts.appCode = :appCode ")
  public Set<DynamicTaskSchedulerEntity> findByTaskCodesAndApplicationNameAndAppCode(@Param("taskCodes") List<String> taskCodes, @Param("applicationName") String applicationName, @Param("appCode") String appCode);
  /**
   * 按照任务有效期（已超时）的关键信息，查询那些已过期的任务
   * @param validityTime 在指定时间之前的动态任务，都会被查询出来
   * @return
   */
  @Query("from DynamicTaskSchedulerEntity dts where dts.validityTime <= :validityTime AND dts.applicationName = :applicationName AND  dts.appCode = :appCode ")
  public Set<DynamicTaskSchedulerEntity> findByValidityTimeAndApplicationNameAndAppCode(@Param("validityTime") Date validityTime, @Param("applicationName") String applicationName, @Param("appCode") String appCode);
  /**
   * 将指定的一个或者多个动态任务的状态变更为“失效”
   * @param taskCodes
   */
  @Modifying
  @Query(value = "update engine_dynamic_task set working_status = 3 , t_status = 0 where application_name = :applicationName and app_code = :appCode and task_code in (:taskCodes)" , nativeQuery = true)
  public void updateInvalidByTaskCodesAndApplicationNameAndAppCode(@Param("taskCodes") String[] taskCodes, @Param("applicationName") String applicationName, @Param("appCode") String appCode);
  /**
   * 将指定的一个或者多个动态任务的状态变更为“有效”，且执行状态为“已停止”
   * @param taskCodes
   */
  @Modifying
  @Query(value = "update engine_dynamic_task set working_status = 3 , t_status = 1 where application_name = :applicationName and app_code = :appCode and task_code in (:taskCodes)" , nativeQuery = true)
  public void updateEffectiveByTaskCodesAndApplicationNameAndAppCode(@Param("taskCodes") String[] taskCodes, @Param("applicationName") String applicationName, @Param("appCode") String appCode);
  /**
   * 将指定的一个或者多个动态任务的状态变更为“要求启动”状态
   * @param taskCodes
   */
  @Modifying
  @Query(value = "update engine_dynamic_task set working_status = 0 , t_status = 1 where application_name = :applicationName and app_code = :appCode and task_code in (:taskCodes)" , nativeQuery = true)
  public void updateReadyRunningByTaskCodesAndApplicationNameAndAppCode(@Param("taskCodes") String[] taskCodes, @Param("applicationName") String applicationName, @Param("appCode") String appCode);
  /**
   * 将指定的一个或者多个动态任务的状态变更为“要求停止”状态
   * @param taskCodes
   */
  @Modifying
  @Query(value = "update engine_dynamic_task set working_status = 1 , t_status = 1 where application_name = :applicationName and app_code = :appCode and task_code in (:taskCodes)" , nativeQuery = true)
  public void updateReadyStopByTaskCodesAndApplicationNameAndAppCode(@Param("taskCodes") String[] taskCodes, @Param("applicationName") String applicationName, @Param("appCode") String appCode);
  /**
   * 将指定的一个或者多个动态任务的状态变更为“要求重启”状态
   * @param taskCodes
   */
  @Modifying
  @Query(value = "update engine_dynamic_task set working_status = 4 where application_name = :applicationName and app_code = :appCode and task_code in (:taskCodes)" , nativeQuery = true)
  public void updateRestartByTaskCodesAndApplicationNameAndAppCode(@Param("taskCodes") String[] taskCodes, @Param("applicationName") String applicationName, @Param("appCode") String appCode);
  /**
   * 将指定的一个或者多个动态任务的状态变更为“运行中”
   * @param taskCodes
   */
  @Modifying
  @Query(value = "update engine_dynamic_task set working_status = 2 , t_status = 1 where application_name = :applicationName and app_code = :appCode and task_code in (:taskCodes)" , nativeQuery = true)
  public void updateRunningByTaskCodesAndApplicationNameAndAppCode(@Param("taskCodes") String[] taskCodes, @Param("applicationName") String applicationName, @Param("appCode") String appCode);
  /**
   * 将指定的一个或者多个动态任务的状态变更为“已停止”
   * @param taskCodes
   */
  @Modifying
  @Query(value = "update engine_dynamic_task set working_status = 3 where application_name = :applicationName and app_code = :appCode and task_code in (:taskCodes)" , nativeQuery = true)
  public void updateStopedByTaskCodesAndApplicationNameAndAppCode(@Param("taskCodes") String[] taskCodes, @Param("applicationName") String applicationName, @Param("appCode") String appCode);
}