package com.bizunited.nebula.task.local.service.init;

import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.Arrays;
import java.util.Date;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.concurrent.ArrayBlockingQueue;
import java.util.concurrent.BlockingQueue;
import java.util.concurrent.ThreadPoolExecutor;

import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.apache.commons.lang3.time.DateUtils;
import org.apache.commons.lang3.tuple.Pair;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.boot.CommandLineRunner;
import org.springframework.context.ApplicationContext;
import org.springframework.scheduling.support.CronTrigger;
import org.springframework.stereotype.Component;
import org.springframework.util.CollectionUtils;

import com.bizunited.nebula.init.service.InitProcessEnvironmentService;
import com.bizunited.nebula.script.context.InvokeProxyContext;
import com.bizunited.nebula.task.annotations.DynamicTaskService;
import com.bizunited.nebula.task.local.configuration.DynamicTaskProperties;
import com.bizunited.nebula.task.local.service.scheduler.DynamicMasterKeeperTask;
import com.bizunited.nebula.task.service.DynamicTaskSchedulerVoService;
import com.bizunited.nebula.task.vo.DynamicTaskSchedulerVo;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;

import javassist.ClassPool;
import javassist.CtClass;
import javassist.CtMethod;
import javassist.NotFoundException;

/**
 * 动态任务模块初始化动作在这里完成。包括：</br>
 * 1、根据nebula初始化框架，完成每一个appname-appCode的锁获取任务的创建和执行</br>
 * 2、扫描整个应用程序中所有使用DynamicTaskService注解的方法，并进行匹配的数据持久层幂等性同步
 * @author Administrator
 */
@Component("_SchedulerTaskInitProcess")
public class SchedulerTaskInitProcess implements CommandLineRunner , InitProcessEnvironmentService {
  @Autowired
  private ApplicationContext applicationContext;
  /**
   * 动态任务——master节点独占操作权管理线程池
   */
  @Autowired
  @Qualifier("platform_dTaskMasterKeeperThreadExecutor")
  private ThreadPoolExecutor dynamicTaskMasterKeeperThreadExecutor;
  @Autowired
  private DynamicTaskSchedulerVoService dynamicTaskSchedulerVoService;
  @Autowired
  private DynamicTaskProperties dynamicTaskProperties;
  /**
   * 队列中的每一个节点携带两个数据，左侧：applicationName；右侧：appCode
   */
  private BlockingQueue<Pair<String, String>> taskQueue = new ArrayBlockingQueue<>(100);
  /**
   * 这里记录了应用程序中所有使用了DynamicTaskService注解的，被IOC容器管理的方法
   */
  private static Map<String, List<CtMethod>> nebulaServiceMethodMapping = Maps.newLinkedHashMap();
  private ClassPool pool = ClassPool.getDefault();
  /**
   * 日志
   */
  private static final Logger LOGGER = LoggerFactory.getLogger(SchedulerTaskInitProcess.class);
  
  @Override
  public void run(String... args) throws Exception {
    /*
     * 应用程序启动时，为了保证所有appcode级别的动态任务都可以被初始化，
     * 所以在这里要负责对应用程序中所有使用了DynamicTaskService注解的位置进行扫描，并记录下来:
     * 
     * 1、检查当前所有已在数据库中持久化存储的所有Groovy执行性质的，拥有corn表达式的所有定时任务，其属性信息是满足合规性的
     * 2、分析当前系统中当前基于DynamicTaskService注解的java method执行任务，并同步更新相关数据库中记录的信息（已被后续initForAppCode方法使用）
     * 3、启动master节点独占操作权管理任务，并保证通过BlockingQueue进行数据协同
     * */
    // 1、TODO 第一步骤有时间再补上
    
    // 2、=======
    String[] definitionNames = applicationContext.getBeanDefinitionNames();
    for (String definitionName : definitionNames) {
      List<CtMethod> methods = Lists.newLinkedList();
      Class<?> beanClass = applicationContext.getType(definitionName);
      String beanClassName = Objects.requireNonNull(beanClass).getName();
      // 如果这个类被Spring代理，或者是一个内隐类，则判定其主类
      if(beanClassName.contains("$")) {
        beanClassName = beanClassName.substring(0, beanClassName.indexOf("$"));
      }
      if(StringUtils.isBlank(beanClassName)) {
        continue;
      }
      CtClass beanCtClass = null;
      try {
        beanCtClass = pool.get(beanClassName);
      } catch(NotFoundException e) {
        continue;
      }
      // 试图获取这个class中带有DynamicTaskService注解的方法
      Arrays.stream(beanCtClass.getDeclaredMethods()).filter(item -> item.hasAnnotation(DynamicTaskService.class)).forEach(methods::add);
      if(!methods.isEmpty()) {
        nebulaServiceMethodMapping.put(definitionName, methods);
      }
    }
    
    // 3、=======(只有一个管理线程池)
    DynamicMasterKeeperTask dynamicMasterKeeperTask = applicationContext.getBean(DynamicMasterKeeperTask.class, taskQueue);
    dynamicTaskMasterKeeperThreadExecutor.submit(dynamicMasterKeeperTask);
  }
  
  @Override
  public boolean doInitForAppCode(String appCode) {
    // 每次应用程序装载都要进行appCode（应用程序/品牌商）级别的初始化
    return true;
  }

  @Override
  public void initForAppCode(String appCode) {
    /* 
     * 涉及到的每一组application_appCode，都需要让DynamicMasterKeeperTask
     * 建立一个独占权保持，然后才能启动扫描。处理步骤如下：
     * 
     * 1、首先确认这些基于DynamicTaskService注解的动态任务方法已经被同步到数据持久层了
     * 如果没有同步，则进行同步
     * 2、然后发消息给在dynamicTaskMasterKeeperThreadExecutor线程池运行的DynamicMasterKeeperTask任务
     * */
    if(CollectionUtils.isEmpty(nebulaServiceMethodMapping)) {
      return;
    }
    String applicationName = dynamicTaskProperties.getApplicationName();
    
    // 1、========
    // 开始对这些method进行验证-并进行数据库同步(以每一个definitionName为一组)
    List<DynamicTaskSchedulerVo> currentScanTasks = Lists.newArrayList();
    for (Map.Entry<String,List<CtMethod>> entry : nebulaServiceMethodMapping.entrySet()) {
      try {
        currentScanTasks.addAll(this.analysisMethods(entry.getValue() , entry.getKey() , applicationName , appCode));
      } catch(ClassNotFoundException | NotFoundException e) {
        // 报错了，输出警告后，后续的定时器方法还要接着分析
        LOGGER.error(e.getMessage() , e);
      }
    }
    this.dynamicTaskSchedulerVoService.saveIgnorePrefix(currentScanTasks, 2, applicationName, appCode);
    
    // 2、======
    Pair<String, String> pair = Pair.of(applicationName, appCode);
    this.taskQueue.add(pair);
  }
  
  /**
   * 该私有方法，分析指定spring实现类中的所有被标注了DynamicTaskService注解的服务，并形成对应的DynamicTaskSchedulerVo集合
   * 注意：利用DynamicTaskService注解描述的动态任务，需要区分appcode进行写入，而这要归功于nebula初始化组件提供的相关方法
   * @param nebulaServiceMethods 指定的java method信息
   * @param definitionName 所属的spring接口的定义名
   * @return 
   * @throws ClassNotFoundException
   * @throws NotFoundException
   */
  private List<DynamicTaskSchedulerVo> analysisMethods(List<CtMethod> nebulaServiceMethods , String definitionName , String applicationName , String appCode) throws ClassNotFoundException , NotFoundException {
    // currentTasks变量记录当前初始化扫描的所有基于DynamicTaskService注解的任务信息
    List<DynamicTaskSchedulerVo> currentScanTasks = Lists.newArrayList();
    SimpleDateFormat dateFormat = new SimpleDateFormat("yyyy-MM-dd HH:mm:ss");
    for (CtMethod ctMethod : nebulaServiceMethods) {
      DynamicTaskService dynamicTaskService  = (DynamicTaskService)ctMethod.getAnnotation(DynamicTaskService.class);
      if(dynamicTaskService == null) {
        continue;
      }
      String cornExpression = dynamicTaskService.cornExpression();
      String executePoint = dynamicTaskService.executePoint();
      String taskDesc = dynamicTaskService.taskDesc();
      String validityPoint = dynamicTaskService.validityPoint();
      boolean onlyOnce = dynamicTaskService.onlyOnce();
      String methodName = ctMethod.getName();
      // 将会当成task code使用
      String methodLongName = ctMethod.getLongName();
      CtClass[] parameterTypes = ctMethod.getParameterTypes();
      
      // 开始进行有效性验证
      DynamicTaskSchedulerVo currentTask = new DynamicTaskSchedulerVo();
      Validate.isTrue(parameterTypes == null || parameterTypes.length == 0 || (parameterTypes.length == 1 && parameterTypes[0] == pool.getCtClass(InvokeProxyContext.class.getName()))
          , "使用DynamicTaskService注解的方法，必须托管给Spring，并且目前只允许没有入参或者入参为com.bizunited.platform.core.service.invoke.InvokeProxyContext的方法，请检查%s" , methodLongName);
      Validate.notBlank(taskDesc , "基于DynamicTaskService注解的定时器任务并没有说明其业务意义（expressionDesc），请检查%s!", methodLongName);
      
      // 如果条件成立，就是一次性任务，否则就是周期性任务
      if(onlyOnce) {
        // 判定并设置执行时间信息（一次性执行任务有两种设定方式）
        if(!StringUtils.isBlank(executePoint)) {
          Date executePointTime = null;
          try {
            executePointTime = dateFormat.parse(executePoint);
          } catch(ParseException e) {
            throw new IllegalArgumentException(String.format("一次性定时任务%s设定的日期格式存在问题，请检查", methodLongName));
          }
          currentTask.setExecutePoint(executePointTime);
          // 默认设定当前一次性任务会在执行时间点（executePoint）之后的1小时内自动到期
          Date validityTime = DateUtils.addMinutes(executePointTime, 60);
          currentTask.setValidityTime(validityTime);
        } else if(!StringUtils.isBlank(cornExpression)) {
          try {
            new CronTrigger(cornExpression);
          } catch(RuntimeException e) {
            throw new IllegalArgumentException(String.format("一次性任务%s设定的corn表达是存在格式问题，请检查", methodLongName));
          }
          currentTask.setExecuteExpression(cornExpression);
          currentTask.setExpressionDesc(cornExpression);
        } else {
          throw new IllegalArgumentException(String.format("当任务是一次性执行任务时，要么设定corn表达式（cornExpression），要么设定确定的执行时间点（executePoint），请检查任务%s", methodLongName));
        }
        currentTask.setTaskType(1);
      } else {
        if(!StringUtils.isBlank(cornExpression)) {
          try {
            new CronTrigger(cornExpression);
          } catch(RuntimeException e) {
            throw new IllegalArgumentException(String.format("周期性执行任务%s设定的corn表达是存在格式问题，请检查", methodLongName));
          }
          currentTask.setExecuteExpression(cornExpression);
          currentTask.setExpressionDesc(cornExpression);
        } else {
          throw new IllegalArgumentException(String.format("当任务是周期性执行任务时，要设定corn表达式（cornExpression），请检查任务%s", methodLongName));
        }
        currentTask.setTaskType(2);
      }
      
      // 如果用户自行设定了失效时间
      if(!StringUtils.isBlank(validityPoint)) {
        Date validityPointTime = null;
        try {
          validityPointTime = dateFormat.parse(validityPoint);
        } catch(ParseException e) {
          throw new IllegalArgumentException(String.format("定时任务%s设定的失效时间日期格式存在问题，请检查", methodLongName));
        }
        currentTask.setValidityTime(validityPointTime);
      }
      // 当前定时器的业务意义
      currentTask.setTaskDesc(taskDesc);
      currentTask.setInvokeType(2);
      // 将完整的方法名作为任务code
      currentTask.setTaskCode(methodLongName);
      currentTask.setScriptName("");
      // 其它所需要的反射信息记录
      currentTask.setInvokeBeanName(definitionName);
      currentTask.setInvokeMethod(methodName);
      // 当前应用程序名和appCode（多应用业务编号）
      currentTask.setAppCode(appCode);
      currentTask.setApplicationName(applicationName);
      currentScanTasks.add(currentTask);
    }
    return currentScanTasks;
  }
}
