package com.bizunited.nebula.task.local.service.internal;

import com.bizunited.nebula.common.service.NebulaToolkitService;
import com.bizunited.nebula.common.util.tenant.TenantUtils;
import com.bizunited.nebula.task.dto.DynamicTaskSchedulerLogDto;
import com.bizunited.nebula.task.local.configuration.DynamicTaskProperties;
import com.bizunited.nebula.task.local.entity.DynamicTaskSchedulerEntity;
import com.bizunited.nebula.task.local.entity.DynamicTaskSchedulerLogEntity;
import com.bizunited.nebula.task.local.repository.DynamicTaskSchedulerLogRepository;
import com.bizunited.nebula.task.service.DynamicTaskSchedulerLogVoService;
import com.bizunited.nebula.task.vo.DynamicTaskSchedulerLogVo;
import com.google.common.collect.Lists;
import com.google.common.collect.Sets;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.PageImpl;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Pageable;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;

import javax.transaction.Transactional;

import java.util.ArrayList;
import java.util.Collection;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.stream.Collectors;

/**
 * TODO 注释未写
 * @author saturn
 */
@Service("DynamicTaskSchedulerLogServiceImpl")
public class DynamicTaskSchedulerLogVoServiceImpl implements DynamicTaskSchedulerLogVoService {
  @Autowired
  private DynamicTaskSchedulerLogRepository dynamicTaskSchedulerLogRepository;
  @Autowired
  private DynamicTaskProperties dynamicTaskProperties;
  @Autowired
  private NebulaToolkitService nebulaToolkitService;

  @Transactional
  @Override
  public DynamicTaskSchedulerLogVo create(DynamicTaskSchedulerLogVo log) {
    return this.createForm(log);
    //==================================================== 
    //    这里可以处理第三方系统调用（或特殊处理过程）
    //====================================================
  }
  
  private DynamicTaskSchedulerLogVo createForm(DynamicTaskSchedulerLogVo log) { 
    this.createValidation(log);
    
    // 转换后保存
    DynamicTaskSchedulerLogEntity logEntity = this.nebulaToolkitService.copyObjectByWhiteList(log, DynamicTaskSchedulerLogEntity.class, LinkedHashSet.class, ArrayList.class);
    DynamicTaskSchedulerEntity dynamicTask = new DynamicTaskSchedulerEntity();
    dynamicTask.setId(log.getDynamicTaskId());
    logEntity.setDynamicTaskScheduler(dynamicTask);
    this.dynamicTaskSchedulerLogRepository.save(logEntity);
    
    // 返回最终处理的结果，里面带有详细的关联信息
    log.setId(logEntity.getId());
    return log;
  }
  /**
   * 在创建一个新的DynamicTaskSchedulerLogEntity模型对象之前，检查对象各属性的正确性，其主键属性必须没有值
   */
  private void createValidation(DynamicTaskSchedulerLogVo log) { 
    Validate.notNull(log , "进行当前操作时，信息对象必须传入!!");
    // 判定那些不能为null的输入值：条件为 caninsert = true，且nullable = false
    Validate.isTrue(StringUtils.isBlank(log.getId()), "添加信息时，当期信息的数据编号（主键）不能有值！");
    log.setId(null);
    Validate.notBlank(log.getDynamicTaskId() , "执行涉及的具体任务技术编号必须传入! ");
    Validate.notBlank(log.getTaskCode() , "执行涉及的具体任务业务编号必须传入! ");
    if(StringUtils.isBlank(log.getApplicationName())) {
      log.setApplicationName(dynamicTaskProperties.getApplicationName());
    }
    if(StringUtils.isBlank(log.getAppCode())) {
      log.setAppCode(TenantUtils.getAppCode());
    }
    Validate.notNull(log.getStartTime(), "执行开始时间不能为空！");
    Validate.notNull(log.getEndTime(), "执行结束时间不能为空！"); 
    Validate.notNull(log.getSuccess(), "执行是否成功不能为空！");
    // 验证长度，被验证的这些字段符合特征: 字段类型为String，且不为PK （注意连续空字符串的情况） 
    Validate.isTrue(log.getErrorContents() == null || log.getErrorContents().length() < 512 , "可能的错误日志,填入值超过了限定长度(512)，请检查!");
  }

  @Override
  @Transactional
  public void deleteById(String id) {
    // 只有存在才进行删除
    Validate.notBlank(id , "进行删除时，必须给定主键信息!!");
    Optional<DynamicTaskSchedulerLogEntity> current = this.dynamicTaskSchedulerLogRepository.findById(id);
    if(current.isPresent()) { 
      this.dynamicTaskSchedulerLogRepository.delete(current.get());
    } 
  }

  @Override
  @Transactional
  public void deleteByDynamicTaskId(String dynamicTaskId) {
    Validate.notBlank(dynamicTaskId , "进行删除时，必须给定动态任务编号信息!!");
    this.dynamicTaskSchedulerLogRepository.deleteByDynamicTaskId(dynamicTaskId);
  }

  @Override
  public DynamicTaskSchedulerLogVo findById(String id) { 
    if(StringUtils.isBlank(id)) { 
      return null;
    }
    
    DynamicTaskSchedulerLogEntity currentDynamicTask = dynamicTaskSchedulerLogRepository.findDetailsById(id);
    if(currentDynamicTask == null) {
      return null;
    }
    
    // 转换后返回
    DynamicTaskSchedulerLogVo result = this.nebulaToolkitService.copyObjectByWhiteList(currentDynamicTask, DynamicTaskSchedulerLogVo.class, LinkedHashSet.class, ArrayList.class);
    DynamicTaskSchedulerEntity dynamicTaskScheduler = currentDynamicTask.getDynamicTaskScheduler();
    result.setDynamicTaskId(dynamicTaskScheduler.getId());
    result.setTaskCode(dynamicTaskScheduler.getTaskCode());
    result.setApplicationName(dynamicTaskScheduler.getApplicationName());
    result.setAppCode(dynamicTaskScheduler.getAppCode());
    return result;
  }
  
  @Override
  public Set<DynamicTaskSchedulerLogVo> findDetailsByDynamicTask(String dynamicTaskId) {
    if(StringUtils.isBlank(dynamicTaskId)) {
      return null;
    }
    
    Set<DynamicTaskSchedulerLogEntity> dynamicTaskSchedulerLogs = this.dynamicTaskSchedulerLogRepository.findDetailsByDynamicTask(dynamicTaskId);
    if(CollectionUtils.isEmpty(dynamicTaskSchedulerLogs)) {
      return null;
    }
    Map<String , DynamicTaskSchedulerLogEntity> taskMap = dynamicTaskSchedulerLogs.stream().collect(Collectors.toMap(DynamicTaskSchedulerLogEntity::getId, item -> item));
    
    // 转换后返回
    Collection<DynamicTaskSchedulerLogVo> dynamicTaskSchedulerLogVos = this.nebulaToolkitService.copyCollectionByWhiteList(dynamicTaskSchedulerLogs, DynamicTaskSchedulerLogEntity.class, DynamicTaskSchedulerLogVo.class, LinkedHashSet.class, ArrayList.class);
    for (DynamicTaskSchedulerLogVo dynamicTaskSchedulerLogItem : dynamicTaskSchedulerLogVos) {
      DynamicTaskSchedulerLogEntity taskLogEntity = taskMap.get(dynamicTaskSchedulerLogItem.getId());
      DynamicTaskSchedulerEntity taskEntity = taskLogEntity.getDynamicTaskScheduler();
      dynamicTaskSchedulerLogItem.setDynamicTaskId(taskEntity.getId());
      dynamicTaskSchedulerLogItem.setTaskCode(taskEntity.getTaskCode());
      dynamicTaskSchedulerLogItem.setApplicationName(taskEntity.getApplicationName());
      dynamicTaskSchedulerLogItem.setAppCode(taskEntity.getAppCode());
    }
    return Sets.newLinkedHashSet(dynamicTaskSchedulerLogVos);
  }

  @Override
  public Page<DynamicTaskSchedulerLogVo> findByConditions(Pageable pageable, DynamicTaskSchedulerLogDto conditions) {
    if(pageable ==null){
      pageable = PageRequest.of(0,50);
    }
    if(conditions == null) {
      conditions = new DynamicTaskSchedulerLogDto();
    }
    // 一些条件必须传入
    conditions.setAppCode(TenantUtils.getAppCode());
    String applicationName = dynamicTaskProperties.getApplicationName();
    conditions.setApplicationName(applicationName);
    Page<DynamicTaskSchedulerLogEntity> page = this.dynamicTaskSchedulerLogRepository.findByConditions(pageable, conditions);
    if(page == null || !page.hasContent()) {
      return null;
    }
    
    // 转换后返回(注意转换后对dynamicTaskId的获取)
    List<DynamicTaskSchedulerLogEntity> entityContents = page.getContent();
    Map<String , DynamicTaskSchedulerLogEntity> pageMap = entityContents.stream().collect(Collectors.toMap(DynamicTaskSchedulerLogEntity::getId, item -> item));
    Collection<DynamicTaskSchedulerLogVo> dynamicTaskSchedulerLogVos = this.nebulaToolkitService.copyCollectionByWhiteList(entityContents, DynamicTaskSchedulerLogEntity.class, DynamicTaskSchedulerLogVo.class, LinkedHashSet.class, ArrayList.class);
    for (DynamicTaskSchedulerLogVo dynamicTaskSchedulerLogVo : dynamicTaskSchedulerLogVos) {
      DynamicTaskSchedulerLogEntity taskLogEntity = pageMap.get(dynamicTaskSchedulerLogVo.getId());
      DynamicTaskSchedulerEntity taskEntity = taskLogEntity.getDynamicTaskScheduler();
      dynamicTaskSchedulerLogVo.setDynamicTaskId(taskEntity.getId());
      dynamicTaskSchedulerLogVo.setTaskCode(taskEntity.getTaskCode());
      dynamicTaskSchedulerLogVo.setApplicationName(taskEntity.getApplicationName());
      dynamicTaskSchedulerLogVo.setAppCode(taskEntity.getAppCode());
    }
    return new PageImpl<>(Lists.newArrayList(dynamicTaskSchedulerLogVos), pageable, page.getTotalElements());
  }
} 