package com.bizunited.nebula.task.local.service.internal;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Date;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Map.Entry;
import java.util.Set;
import java.util.UUID;
import java.util.concurrent.ScheduledFuture;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;

import javax.transaction.Transactional;

import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.redisson.executor.CronExpression;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.context.ApplicationContext;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.PageImpl;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Pageable;
import org.springframework.scheduling.concurrent.ThreadPoolTaskScheduler;
import org.springframework.scheduling.support.CronTrigger;
import org.springframework.security.core.context.SecurityContext;
import org.springframework.security.core.context.SecurityContextHolder;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;

import com.bizunited.nebula.common.service.NebulaToolkitService;
import com.bizunited.nebula.common.service.redis.RedisMutexService;
import com.bizunited.nebula.common.util.tenant.TenantUtils;
import com.bizunited.nebula.script.service.ScriptVoService;
import com.bizunited.nebula.script.vo.ScriptVo;
import com.bizunited.nebula.task.dto.DynamicTaskSchedulerDto;
import com.bizunited.nebula.task.local.configuration.DynamicTaskProperties;
import com.bizunited.nebula.task.local.entity.DynamicTaskParamEntity;
import com.bizunited.nebula.task.local.entity.DynamicTaskSchedulerEntity;
import com.bizunited.nebula.task.local.repository.DynamicTaskParamRepository;
import com.bizunited.nebula.task.local.repository.DynamicTaskSchedulerRepository;
import com.bizunited.nebula.task.local.service.scheduler.DynamicTask;
import com.bizunited.nebula.task.service.DynamicTaskSchedulerLogVoService;
import com.bizunited.nebula.task.service.DynamicTaskSchedulerVoService;
import com.bizunited.nebula.task.service.strategy.DynamicTaskParamValueConverter;
import com.bizunited.nebula.task.vo.DynamicTaskParamVo;
import com.bizunited.nebula.task.vo.DynamicTaskSchedulerVo;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.google.common.collect.Sets;

/**
 * 动态任务的服务层具体实现
 * @author yinwenjie
 */
@Service("dynamicTaskSchedulerServiceImpl")
public class DynamicTaskSchedulerServiceImpl implements DynamicTaskSchedulerVoService {
  @Autowired
  @Qualifier("platform_dynamicTaskScheduler")
  private ThreadPoolTaskScheduler threadPoolTaskScheduler;
  @Autowired
  private ApplicationContext applicationContext;
  @Autowired
  private NebulaToolkitService nebulaToolkitService;
  @Autowired
  private ScriptVoService scriptService;
  @Autowired
  private RedisMutexService redisMutexService;
  @Autowired
  private DynamicTaskProperties dynamicTaskProperties;
  @Autowired
  private DynamicTaskSchedulerLogVoService dynamicTaskSchedulerLogService;
  @Autowired
  private DynamicTaskSchedulerRepository dynamicTaskSchedulerRepository;
  @Autowired
  private DynamicTaskParamRepository dynamicTaskParamRepository;
  /**
   * 该映射关系用来记录当前的定时器任务运行情况
   * key：任务的唯一编号
   */
  private Map<String, ScheduledFuture<?>> taskMapping = Maps.newConcurrentMap();
  /**
   * 动态任务预生成的redis-map结构的名字
   */
  private static final String _TASK_GEN_PREFIX = "_NEBULA_TASK_REPEAT";
  /**
   * 可以使用的任务方法参数转换器
   */
  @Autowired(required = false)
  private List<DynamicTaskParamValueConverter> dynamicTaskParamValueConverters;
  private static final Logger LOGGER = LoggerFactory.getLogger(DynamicTaskSchedulerServiceImpl.class);
  
  @Override
  @Transactional
  public void loading(String applicationName, String appCode) {
    /*
     * 带有幂等性的处理过程为：
     * 
     * 1、首先同步验证“正在运行”的任务，保证数据库中记录的“正在运行”的任务和实际的“正在运行”的任务是一致的。其中就包括：
     * 1.1、数据库中个“正在运行”的任务必须确认是在实际运行
     * 1.2、未在数据库中标识为“正在运行”的实际运行任务，要进行停止
     * 
     * 2、在数据库中“要求运行”的任务，要进行运行。
     * 
     * 3、在数据库中“要求停止”的任务、标记为“无效”的任务和“超过有效期”的任务，要确认是否在运行，如果在运行则需要进行停止，且标记为“已停止状态”。
     * 
     * 4、在数据库中“要求重启”的任务，需要进行重启，并修改运行状态为“已运行”，
     * 
     * 注意：该方法不需要考虑操作权的问题，操作权的问题由外部调用者（例如DynamicMasterNotifyTask，进行考虑）
     * */
    
    // 1、========(处理方式，主要是取两个集合的差集)
    int[] workingStatuses = new int[] {2};
    Date now = new Date();
    // 取得数据库中“正在运行”任务的信息(注意进行有效期判定)
    Set<DynamicTaskSchedulerEntity> dbDynamicTasks = this.dynamicTaskSchedulerRepository.findByTstatusAndWorkingStatusAndApplicationNameAndAppCode(1, workingStatuses, applicationName, appCode);
    if(!CollectionUtils.isEmpty(dbDynamicTasks)) {
      dbDynamicTasks.stream().filter(item -> item.getValidityTime() == null || item.getValidityTime().getTime() >= now.getTime()).collect(Collectors.toSet());
    }
    Set<String> dbDynamicTaskCodes = dbDynamicTasks.stream().map(item -> StringUtils.join(applicationName , "_" , appCode , "_" , item.getTaskCode())).collect(Collectors.toSet());
    // 取得当前引用系统中记录的“正在运行”任务的信息
    Set<String> currentDynamicTaskCodes = Sets.newHashSet();
    if(!CollectionUtils.isEmpty(taskMapping)) {
      String matchKey = StringUtils.join(applicationName , "_" , appCode , "_");
      currentDynamicTaskCodes = taskMapping.keySet().stream().filter(item -> StringUtils.indexOf(item, matchKey) == 0).collect(Collectors.toSet());
    }
    
    // 1.1、=======
    // 条件成立所涉及的集合，就是数据库为“正在运行”，但实际没有运行的任务
    final Set<String> mustStartTaskCodes;
    if(!CollectionUtils.isEmpty(mustStartTaskCodes = Sets.difference(dbDynamicTaskCodes, currentDynamicTaskCodes))) {
      Set<DynamicTaskSchedulerEntity> mustStartTasks = dbDynamicTasks.stream().filter(item -> mustStartTaskCodes.contains(StringUtils.join(item.getApplicationName() , "_" , item.getAppCode() , "_" , item.getTaskCode()))).collect(Collectors.toSet());
      this.startupTasks(applicationName, appCode, mustStartTasks);
    }
    
    // 1.2、=======
    // 条件成立涉及的集合，就是实际正在运行，但数据库记录状态，要求不运行的
    final Set<String> mustStopTaskCodes;
    if(!CollectionUtils.isEmpty(mustStopTaskCodes = Sets.difference(currentDynamicTaskCodes , dbDynamicTaskCodes))) {
      Set<DynamicTaskSchedulerEntity> mustStopTasks = dbDynamicTasks.stream().filter(item -> mustStopTaskCodes.contains(StringUtils.join(item.getApplicationName() , "_" , item.getAppCode() , "_" , item.getTaskCode()))).collect(Collectors.toSet());
      this.stopTasks(applicationName, appCode, mustStopTasks);
    }
    
    // 2、========
    workingStatuses = new int[] {0};
    Set<DynamicTaskSchedulerEntity> requestStartupDynamicTasks = this.dynamicTaskSchedulerRepository.findByTstatusAndWorkingStatusAndApplicationNameAndAppCode(1, workingStatuses, applicationName, appCode);
    if(!CollectionUtils.isEmpty(requestStartupDynamicTasks)) {
      this.startupTasks(applicationName, appCode, requestStartupDynamicTasks);
    }
    
    // 3、========
    workingStatuses = new int[] {1};
    Set<DynamicTaskSchedulerEntity> requestStopDynamicTasks = Sets.newHashSet();
    // 要求停止的任务
    Set<DynamicTaskSchedulerEntity> requestStopEffectiveDynamicTasks = this.dynamicTaskSchedulerRepository.findByTstatusAndWorkingStatusAndApplicationNameAndAppCode(1, workingStatuses, applicationName, appCode);
    if(!CollectionUtils.isEmpty(requestStopEffectiveDynamicTasks)) {
      requestStopDynamicTasks.addAll(requestStopEffectiveDynamicTasks);
    }
    // 已失效的任务
    Set<DynamicTaskSchedulerEntity> requestStopInvalidDynamicTasks =  this.dynamicTaskSchedulerRepository.findByTstatusAndApplicationNameAndAppCode(0, applicationName, appCode);
    if(!CollectionUtils.isEmpty(requestStopInvalidDynamicTasks)) {
      requestStopDynamicTasks.addAll(requestStopInvalidDynamicTasks);
    }
    // 已过期的任务
    Set<DynamicTaskSchedulerEntity> validityTimeDynamicTasks =  this.dynamicTaskSchedulerRepository.findByValidityTimeAndApplicationNameAndAppCode(now, applicationName, appCode);
    if(!CollectionUtils.isEmpty(validityTimeDynamicTasks)) {
      requestStopDynamicTasks.addAll(validityTimeDynamicTasks);
    }
    if(!CollectionUtils.isEmpty(requestStopDynamicTasks)) {
      this.stopTasks(applicationName, appCode, requestStopDynamicTasks);
    }
    
    // 4、======
    workingStatuses = new int[] {4};
    Set<DynamicTaskSchedulerEntity> requestRestartupDynamicTasks = this.dynamicTaskSchedulerRepository.findByTstatusAndWorkingStatusAndApplicationNameAndAppCode(1, workingStatuses, applicationName, appCode);
    if(!CollectionUtils.isEmpty(requestRestartupDynamicTasks)) {
      this.restartupTasks(applicationName, appCode, requestRestartupDynamicTasks);
    }
  }
  
  /**
   * 该私有方法用于停止指定的动态任务
   * @param applicationName 指定的应用程序名 
   * @param appCode 指定的顶级租户业务编号
   * @param mustStopTaskCodes
   */
  private void stopTasks(final String applicationName , final String appCode , Set<DynamicTaskSchedulerEntity> mustStopTasks) {
    /*
     * 停止过程包括：
     * 1、首先停止当前系统中正在运行的动态任务
     * 2、然后修正数据库中这些任务的记录（虽然可能数据库中的数据状态是正常的）
     * */
    // 给单纯的taskCode增加applicationName_appCode_taskCode，然后就可以和taskMapping中的key信息进行比对了
    Set<String> mustStopFullTaskCodes = mustStopTasks.stream().map(item -> StringUtils.join(applicationName , "_" , appCode , "_" , item.getTaskCode())).collect(Collectors.toSet());
    Set<String> mustStopScheduledFutureKeys = taskMapping.keySet().stream().filter(item -> mustStopFullTaskCodes.contains(item)).collect(Collectors.toSet());
    // 注意，只有从taskMapping筛选出来的任务，才是真正要停止的任务——realTasks
    Set<DynamicTaskSchedulerEntity> realStopTasks = mustStopTasks.stream().filter(item -> mustStopScheduledFutureKeys.contains(StringUtils.join(applicationName , "_" , appCode , "_" , item.getTaskCode()))).collect(Collectors.toSet());
    Set<String> realStopTaskCodes = Sets.newHashSet();
    // 1、========
    if(!CollectionUtils.isEmpty(realStopTasks)) {
      for (DynamicTaskSchedulerEntity realStopTask : realStopTasks) {
        String fullTaskCode = StringUtils.join(applicationName , "_" , appCode , "_" , realStopTask.getTaskCode());
        ScheduledFuture<?> scheduledFuture = taskMapping.get(fullTaskCode);
        scheduledFuture.cancel(true);
        taskMapping.remove(fullTaskCode);
        LOGGER.info("===== 动态任务：{}:{}:{}[{}] 停止  ===== " , applicationName , appCode  , realStopTask.getTaskCode() , realStopTask.getTaskDesc());
        realStopTaskCodes.add(realStopTask.getTaskCode());
      }
    }
    
    // 2、=======
    if(!CollectionUtils.isEmpty(realStopTaskCodes)) {
      this.dynamicTaskSchedulerRepository.updateStopedByTaskCodesAndApplicationNameAndAppCode(realStopTaskCodes.toArray(new String[] {}) , applicationName , appCode);
    }
  }
  
  /**
   * 该私有方法用于启动指定的动态任务
   * @param applicationName 指定的应用程序名 
   * @param appCode 指定的顶级租户业务编号
   * @param mustStartTasks
   */
  private void startupTasks(String applicationName, String appCode , Set<DynamicTaskSchedulerEntity> mustStartTasks) {
    /*
     * 处理是依次进行循环处理，每一个任务描述的处理过程如下：
     * 1、如果当前任务的执行类型为1：就是一次性执行任务，那么由此处进行处理
     * 2、如果当前任务的执行类型为2：就是周期性执行任务
     * 3、在处理过程，会有一些要求运行的任务信息不完整，导致不能启动，这些信息要记录在invalidTaskCodes中，并进行取消
     * 成功启动的任务，也要更新数据库中的任务状态
     * */
    List<String> invalidTaskCodes = new ArrayList<>(100);
    List<String> runningTaskCodes = new ArrayList<>(100);
    for (DynamicTaskSchedulerEntity mustStartTask : mustStartTasks) {
      DynamicTask dynamicTask = this.applicationContext.getBean(DynamicTask.class, mustStartTask);
      String key = StringUtils.join(mustStartTask.getApplicationName() , "_" , mustStartTask.getAppCode() , "_" , mustStartTask.getTaskCode());
      // 1、======== 
      if(mustStartTask.getTaskType() == 1) {
        //判断该一次性任务是否执行过，如果执行过，则需要把该一次性任务设置为无效任务
        boolean hasRunning = this.checkDynamicOneTimeTask(mustStartTask);
        if(hasRunning){
          invalidTaskCodes.add(mustStartTask.getTaskCode());
          continue;
        }
        Date executePoint = mustStartTask.getExecutePoint();
        String cornExpression = mustStartTask.getExecuteExpression();
        Validate.isTrue(!(executePoint == null && StringUtils.isBlank(cornExpression)) , "指定任务[%s]是一个一次性任务，但是却没有指定任务执行时间，请检查" , mustStartTask.getTaskCode());
        ScheduledFuture<?> scheduledFuture = null;
        if(executePoint != null) {
          scheduledFuture = this.threadPoolTaskScheduler.schedule(dynamicTask, executePoint);
        } else {
          CronTrigger cronTrigger = new CronTrigger(cornExpression);
          scheduledFuture = this.threadPoolTaskScheduler.schedule(dynamicTask, cronTrigger);
        }
        LOGGER.info("===== 一次性任务：{}:{}:{}[{}] 启动 ===== " , applicationName , appCode , mustStartTask.getTaskCode() , mustStartTask.getTaskDesc());
        this.taskMapping.put(key, scheduledFuture);
        runningTaskCodes.add(mustStartTask.getTaskCode());
      }
      // 2、=======
      else if(mustStartTask.getTaskType() == 2) {
        String cornExpression = mustStartTask.getExecuteExpression();
        Validate.isTrue(StringUtils.isNotBlank(cornExpression) , "指定动态任务[%s]是一个周期性任务，但是却没有指定有效的corn表达式，请检查" , mustStartTask.getTaskCode());
        CronTrigger cronTrigger = new CronTrigger(cornExpression);
        ScheduledFuture<?> scheduledFuture = this.threadPoolTaskScheduler.schedule(dynamicTask, cronTrigger);
        LOGGER.info("===== 周期性任务：{}:{}:{}[{}] 启动 ===== " , applicationName , appCode , mustStartTask.getTaskCode() , mustStartTask.getTaskDesc() );
        this.taskMapping.put(key, scheduledFuture);
        runningTaskCodes.add(mustStartTask.getTaskCode());
      }
      // 其它情况下，直接抛出异常
      else {
        throw new IllegalArgumentException("错误的动态任务类型[" + mustStartTask.getTaskCode() + "]，请检查");
      }
    }
    
    // 3、=====
    if(!invalidTaskCodes.isEmpty()) {
      this.dynamicTaskSchedulerRepository.updateInvalidByTaskCodesAndApplicationNameAndAppCode(invalidTaskCodes.toArray(new String[] {}) , applicationName , appCode);
    }
    if(!runningTaskCodes.isEmpty()) {
      this.dynamicTaskSchedulerRepository.updateRunningByTaskCodesAndApplicationNameAndAppCode(runningTaskCodes.toArray(new String[] {}) , applicationName , appCode);
    }
  }
  
  /**
   * 该私有方法用于重启那些，标识为“要求重启”的任务。注意，只有状态为“有效”的任务，才能在停止后被启动；</br>
   * 如果停止后的任务为“无效任务”或者重启过程失败，任务状态都会被表示为“已停止”
   * @param applicationName
   * @param appCode
   * @param mustReStartTasks 注意，这些任务并不是真正需要重启的任务，而是在数据层面要求进行重启的任务
   */
  private void restartupTasks(String applicationName, String appCode , Set<DynamicTaskSchedulerEntity> mustRestartTasks) {
    /*
     * 处理过程分为两个大的步骤：
     * 1、在要求重启的任务中，停止那些真正在运行的任务，并将所有任务标识为“要求启动”
     * 2、按照要求启动的任务逻辑进行处理，详见startupTasks方法
     * */
    if(CollectionUtils.isEmpty(mustRestartTasks)) {
      return;
    }
    for (DynamicTaskSchedulerEntity mustRestartTask : mustRestartTasks) {
      Validate.isTrue(mustRestartTask.getTstatus() == 1 , "在进行任务重启时，发现任务%s:%s:%s状态错误，请检查!!" , applicationName , appCode , mustRestartTask.getTaskCode());
    }
    
    // 1、========= 
    Set<String> mustRestartFullTaskCodes = mustRestartTasks.stream().map(item -> StringUtils.join(item.getApplicationName() , "_" , item.getAppCode() , "_" , item.getTaskCode())).collect(Collectors.toSet());
    Set<String> mustRestartTaskCodes = mustRestartTasks.stream().map(DynamicTaskSchedulerEntity::getTaskCode).collect(Collectors.toSet());
    Set<String> mustRealStopTaskCodes = Sets.intersection(this.taskMapping.keySet(), mustRestartFullTaskCodes);
    // 交集中的全业务编号对应的任务，就是要真正进行停止的任务(注意，这里的停止没有interrupt中断信号，就是要让可能正在运行的动态任务真正运行完)
    if(!CollectionUtils.isEmpty(mustRealStopTaskCodes)) {
      for (String mustRealStopTaskCode : mustRealStopTaskCodes) {
        LOGGER.info("===== 周期性任务：{} 准备重启 ===== " , mustRealStopTaskCode);
        this.taskMapping.remove(mustRealStopTaskCode).cancel(false);
      }
    }
    // 将这些状态正常的“要求重启”的任务，全部置为“要求运行”状态
    this.dynamicTaskSchedulerRepository.updateReadyRunningByTaskCodesAndApplicationNameAndAppCode(mustRestartTaskCodes.toArray(new String[] {}) , applicationName , appCode);
    this.dynamicTaskParamRepository.flush();
    
    // 2、========
    int[] workingStatuses = new int[] {0};
    Set<DynamicTaskSchedulerEntity> requestStartupDynamicTasks = this.dynamicTaskSchedulerRepository.findByTstatusAndWorkingStatusAndApplicationNameAndAppCode(1, workingStatuses, applicationName, appCode);
    if(!CollectionUtils.isEmpty(requestStartupDynamicTasks)) {
      this.startupTasks(applicationName, appCode, requestStartupDynamicTasks);
    }
  }
  
  /**
   * 如果当一次性任务已经被执行了，再次的loading时，需要检查这个一次性任务是否被执行过
   * true：已执行过
   * false：其他情况
   */
  private boolean checkDynamicOneTimeTask(DynamicTaskSchedulerEntity task){
    Integer workStatus = task.getWorkingStatus();
    return workStatus != null && workStatus == 2;
  }

  @Override
  public void unloading(String applicationName, String appCode) {
    /*
     * 注意停止服务，无需要更改数据库中的状态
     * 以便于其它进程能够进行任务的再次启动 
     * */
    Validate.isTrue(StringUtils.isAnyBlank(applicationName , appCode) , "错误的参数性，请检查!!");
    String matchKey = StringUtils.join(applicationName , "_" , appCode , "_");
    Set<ScheduledFuture<?>> mustStopTasks = taskMapping.entrySet().stream().filter(item -> StringUtils.indexOf(item.getKey(), matchKey) == 0).map(Entry::getValue).collect(Collectors.toSet());
    if(!CollectionUtils.isEmpty(mustStopTasks)) {
      for (ScheduledFuture<?> scheduledFuture : mustStopTasks) {
        scheduledFuture.cancel(true);
      }
    }
  }

  @Override
  public String preModify() {
    String prefix = UUID.randomUUID().toString();
    // 7天后过期
    this.redisMutexService.setMCode(_TASK_GEN_PREFIX, prefix, prefix, TimeUnit.MILLISECONDS.convert(7 ,  TimeUnit.DAYS));
    return prefix;
  }

  @Override
  @Transactional
  public DynamicTaskSchedulerVo create(DynamicTaskSchedulerVo task , String scriptContent) {
    this.validationPrefix(task.getPrefix());
    // 开始进行添加操作
    return this.createIgnorePrefix(task, scriptContent);
  }
  
  /**
   * 验证与操作标记
   * @param orderInfo
   */
  private void validationPrefix(String prefix) {
    Validate.notBlank(prefix , "错误的订单预添加标记，请检查!!");
    Validate.isTrue(StringUtils.isNotBlank(this.redisMutexService.getMCode(_TASK_GEN_PREFIX, prefix)) , "没有发现动态任务的预操作标记，可能是因为重复操作的原因!!");
    boolean isLock = false;
    try { 
      if(isLock = this.redisMutexService.tryLock(prefix, TimeUnit.MILLISECONDS, 1)) {
        this.redisMutexService.setMCode(_TASK_GEN_PREFIX, prefix, prefix, 1l);
      } else { 
        throw new IllegalArgumentException("请不要重复进行动态任务的写操作!!");
      } 
    } finally { 
      if(isLock) {
        this.redisMutexService.unlock(prefix);
      } 
    }
  }

  @Override
  @Transactional
  public DynamicTaskSchedulerVo createIgnorePrefix(DynamicTaskSchedulerVo task, String scriptContent) {
    SecurityContext securityContext = SecurityContextHolder.getContext();
    String userAccount = "admin";
    if(securityContext.getAuthentication() != null) {
      userAccount = securityContext.getAuthentication().getName();
    }
    String appCode = TenantUtils.getAppCode();
    String applicationName = dynamicTaskProperties.getApplicationName();
    Date currentTime = new Date();
    
    // 1、=========首先进行动态任务基本信息的添加
    Validate.notNull(task , "进行当前操作时，动态任务对象必须传入!!");
    // 设置自填充值
    task.setCreateTime(currentTime);
    task.setCreateAccount(userAccount);
    task.setModifyTime(currentTime);
    task.setModifyAccount(userAccount);
    // 设定为有效任务
    task.setTstatus(1);
    // 设定当前状态为“要求启动”
    task.setWorkingStatus(0);
    // 当前的应用程序名
    task.setAppCode(appCode);
    task.setApplicationName(applicationName);
    // 进行对象信息验证
    this.createValidation(task);

    // 2、=======然后进行动态任务脚本内容的创建(如果是脚本性质的动态任务)
    if(task.getInvokeType() == 1) {
      Validate.notBlank(scriptContent , "当任务类型为groovy脚本时，脚本内容必须传入!!");
      ScriptVo script = new ScriptVo();
      String scriptName = "dynamicTask_" + appCode + "_" + task.getTaskCode();
      script.setLanguage("groovy");
      script.setName(scriptName);
      script = this.scriptService.create(script, scriptContent);
      task.setScriptName(scriptName);
    }
    
    // 3、======
    // 进行新增
    DynamicTaskSchedulerEntity taskEntity = this.nebulaToolkitService.copyObjectByWhiteList(task, DynamicTaskSchedulerEntity.class, LinkedHashSet.class, ArrayList.class);
    this.dynamicTaskSchedulerRepository.save(taskEntity);
    task.setId(taskEntity.getId());
    
    // 4、======如果存在关联的参数信息，则要进行新增处理
    List<DynamicTaskParamVo> taskParams = task.getParams();
    this.createTaskParams(taskEntity , taskParams);
    return task;
  }
  
  /**
   * 该私有方法用于创建全动态任务下的参数信息（只适用于全动态任务）</br>
   * TODO 这段代码要放置到DynamicTaskParamService中
   * @param taskParams
   */
  private void createTaskParams(DynamicTaskSchedulerEntity taskEntity , List<DynamicTaskParamVo> taskParams) {
    if(!CollectionUtils.isEmpty(taskParams)) {
      // 首先找到默认转换器
      DynamicTaskParamValueConverter defaultDynamicTaskParamValueConverter = dynamicTaskParamValueConverters.stream().filter(item -> item.isDefault()).findFirst().orElse(null);
      // 检查index的正确性，并按照index顺序进行排序
      for (DynamicTaskParamVo taskParam : taskParams) {
        Class<?> paramType = taskParam.getParamType();
        Object value = taskParam.getParamValue();
        Integer index = taskParam.getParamIndex();
        // 验证
        Validate.notNull(paramType , "动态任务执行参数的类型必须传入!!");
        Validate.notNull(value , "动态任务执行参数的值必须传入!!");
        Validate.isTrue(index >= 0 , "动态任务执行参数的索引位如果传入，则必须不为负数!!");
      }
      List<DynamicTaskParamVo> currentTaskParams = taskParams.stream().sorted((s , t) -> s.getParamIndex() - t.getParamIndex()).collect(Collectors.toList());
      
      // 开始进行转换(注意这里的index取值)
      for (int index = 0 ; index < currentTaskParams.size() ; index++) {
        DynamicTaskParamVo currentTaskParam = currentTaskParams.get(index);
        Class<?> paramType = currentTaskParam.getParamType();
        Object value = currentTaskParam.getParamValue();
        boolean canConverter = false;
        DynamicTaskParamValueConverter currntDynamicTaskParamValueConverter = null;
        for (DynamicTaskParamValueConverter dynamicTaskParamValueConverter : dynamicTaskParamValueConverters) {
          if(canConverter = dynamicTaskParamValueConverter.canConverter(index, paramType)) {
            currntDynamicTaskParamValueConverter = dynamicTaskParamValueConverter;
            break;
          }
        }
        // 如果条件成立，则使用默认制转换器
        if(!canConverter) {
          currntDynamicTaskParamValueConverter = defaultDynamicTaskParamValueConverter;
        }
        // 开始进行转换和创建
        String serialize = currntDynamicTaskParamValueConverter.serialize(index, paramType, value);
        DynamicTaskParamEntity dynamicTaskParam = new DynamicTaskParamEntity();
        dynamicTaskParam.setDynamicTask(taskEntity);
        dynamicTaskParam.setParamIndex(index);
        dynamicTaskParam.setParamType(paramType.getName());
        dynamicTaskParam.setParamValue(serialize);
        this.dynamicTaskParamRepository.save(dynamicTaskParam);
        currentTaskParam.setId(dynamicTaskParam.getId());
        currentTaskParam.setParamIndex(index);
      }
    }
  }
  
  @Override
  @Transactional
  public DynamicTaskSchedulerVo createIgnorePrefix(String taskCode , String invokeBeanName , String method , String executeExpression, Date validityTime , String taskDesc , Object... args) {
    // 该方法的本质，就要在边界验证后，对传入的信息进行DynamicTaskSchedulerVo的组合
    Validate.notBlank(invokeBeanName , "错误的Spring ioc中的bean名称，请检查!!");
    Validate.notBlank(method , "错误的方法信息，请检查!!");
    Validate.notBlank(executeExpression , "错误的周期性值设定内容，请检查!!");
    Validate.notBlank(taskDesc , "必须设定任务的说明信息，请检查!!"); 
    Date now = new Date();
    if(validityTime != null) {
      Validate.isTrue(now.getTime() < validityTime.getTime() , "如果设定了过期时间，则过期时间必须大于当前时间");
    }
    String appCode = TenantUtils.getAppCode();
    String applicationName = dynamicTaskProperties.getApplicationName();
    // 确认是在IOC容器中
    try {
      applicationContext.getBean(invokeBeanName);
    } catch(RuntimeException e) {
      throw new IllegalArgumentException("指定的方法并不存在于Spring Ioc容器中，请检查!!");
    }
    
    // 开始组装
    DynamicTaskSchedulerVo dynamicTaskScheduler = new DynamicTaskSchedulerVo();
    dynamicTaskScheduler.setTaskCode(taskCode);
    dynamicTaskScheduler.setAppCode(appCode);
    dynamicTaskScheduler.setApplicationName(applicationName);
    dynamicTaskScheduler.setExecuteExpression(executeExpression);
    dynamicTaskScheduler.setInvokeBeanName(invokeBeanName);
    dynamicTaskScheduler.setInvokeMethod(method);
    // 这种方法只能设定全动态任务
    dynamicTaskScheduler.setInvokeType(3);
    dynamicTaskScheduler.setScriptName("");
    dynamicTaskScheduler.setTaskDesc(taskDesc);
    dynamicTaskScheduler.setExpressionDesc(taskDesc);
    dynamicTaskScheduler.setTstatus(1);
    dynamicTaskScheduler.setValidityTime(validityTime);
    // 要求运行
    dynamicTaskScheduler.setWorkingStatus(1);
    
    // 以下是对于本次调用的参数的设定    
    if(args != null && args.length > 0) {
      List<DynamicTaskParamVo> taskParams = Lists.newArrayList();
      for(int index = 0 ; index < args.length ; index++) {
        Object arg = args[index];
        DynamicTaskParamVo param = new DynamicTaskParamVo();
        if(arg != null) {
          param.setParamType(arg.getClass());
          param.setParamValue(arg);
        } else {
          param.setParamType(Object.class);
        }
        param.setParamIndex(index);
        taskParams.add(param);
      }
      dynamicTaskScheduler.setParams(taskParams);
    }
    
    return this.createIgnorePrefix(dynamicTaskScheduler, null);
  }

  /**
   * 在创建一个新的DynamicTaskSchedulerEntity模型对象之前，检查对象各属性的正确性，其主键属性必须没有值
   */
  private void createValidation(DynamicTaskSchedulerVo task) {
    // 判定那些不能为null的输入值：条件为 caninsert = true，且nullable = false
    Validate.isTrue(StringUtils.isBlank(task.getId()), "添加信息时，当期信息的数据编号（主键）不能有值！");
    task.setId(null);
    // 只能通过接口创建groovy性质的任务（1），以及全动态任务（3）
    Integer invokeType = task.getInvokeType();
    Validate.isTrue(invokeType != null && (invokeType == 3 || invokeType == 1) , "只能创建groovy脚本性质或者全动态性质的任务!!");
    Validate.notBlank(task.getTaskCode(), "任务唯一编号（只能由大小写字母、数字、“-”）不能为空！");
    Validate.isTrue(StringUtils.indexOfAny(task.getTaskCode(), "_") == -1 , "任务唯一编号（只能由大小写字母、数字、“-”）不能为空！");
    Validate.notBlank(task.getTaskDesc(), "任务中文说明不能为空！");
    Validate.notNull(task.getTaskType(), "任务类型不能为空！");
    Validate.notNull(task.getCreateTime(), "创建时间不能为空！");
    Validate.notNull(task.getModifyTime(), "最后修改时间不能为空！");
    Validate.notBlank(task.getApplicationName() , "必须为动态任务设定应用程序名!!");
    Validate.notBlank(task.getAppCode() , "必须为动态任务设定顶级租户信息（appCode）!!");
    // 验证长度，被验证的这些字段符合特征: 字段类型为String，且不为PK （注意连续空字符串的情况） 
    Validate.isTrue(task.getTaskCode() == null || task.getTaskCode().length() < 128 , "任务唯一编号（只能由英文、数字、下杠构成）,填入值超过了限定长度(128)，请检查!");
    Validate.isTrue(task.getTaskDesc() == null || task.getTaskDesc().length() < 256 , "任务中文说明,填入值超过了限定长度(256)，请检查!");
    // 如果当前是一次性任务
    Date currentTime = new Date();
    if(task.getTaskType() == 1) {
      Validate.notNull(task.getExecutePoint(), "一次性任务的开始时间必须填写");
      Validate.isTrue(task.getExecutePoint().getTime() > currentTime.getTime() , "一次性任务的开始时间，必须设定在当前时间点以后");
    }
    // 如果当前是周期性任务
    else if(task.getTaskType() == 2) {
      Validate.notBlank(task.getExecuteExpression() , "周期性任务必须给定有效的corn表达式!!");
      Validate.notBlank(task.getExpressionDesc() , "周期性任务必须传入corn表达式的中文描述!!");
      Validate.isTrue(task.getExecuteExpression() == null || task.getExecuteExpression().length() < 32 , "周期性任务执行表单式,填入值超过了限定长度(32)，请检查!");
      Validate.isTrue(task.getExpressionDesc() == null || task.getExpressionDesc().length() < 256 , "周期性表单时的中文,填入值超过了限定长度(256)，请检查!");
    }
    // 其它情况报错
    else {
      throw new IllegalArgumentException("动态任务只能是一次性任务或者周期性任务，请检查!!");
    }

    if(StringUtils.isNotBlank(task.getExecuteExpression())){
      try{
        new CronExpression(task.getExecuteExpression());
      }catch (IllegalArgumentException e){
        throw new IllegalArgumentException(String.format("cron表达式有误，请检查!【%s】",e.getMessage()));
      }
    }
    DynamicTaskSchedulerEntity currentDynamicTaskSchedulerEntity = this.dynamicTaskSchedulerRepository.findByTaskCodeAndApplicationNameAndAppCode(task.getTaskCode() , task.getApplicationName(), task.getAppCode());
    Validate.isTrue(currentDynamicTaskSchedulerEntity == null, "任务唯一编号（只能由英文、数字、下杠构成）已存在,请检查");
  }

  @Override
  @Transactional
  public DynamicTaskSchedulerVo update(DynamicTaskSchedulerVo task, String scriptContent) {
    this.validationPrefix(task.getPrefix());
    // 开始进行修改操作
    return this.updateIgnorePrefix(task, scriptContent);
  }
  
  @Override
  @Transactional
  public DynamicTaskSchedulerVo updateIgnorePrefix(String taskCode, String executeExpression, Date validityTime, String taskDesc, Object... args) {
    Validate.notBlank(taskCode , "在进行动态任务修改时，任务业务编号必须传入（且必须存在）!");
    Validate.notBlank(executeExpression , "在进行动态任务修改时，任务执行周期表达式必须传入");
    Validate.notBlank(taskDesc , "在进行动态任务修改时，任务描述信息必须传入");
    String appCode = TenantUtils.getAppCode();
    String applicationName = this.dynamicTaskProperties.getApplicationName();
    DynamicTaskSchedulerVo currentTask = this.findByTaskCodeAndApplicationNameAndAppCode(taskCode, applicationName, appCode);
    Validate.notNull(currentTask , "未找到指定的动态任务，请检查!!");
    Validate.isTrue(currentTask.getInvokeType() == 3 , "该方法只能修改invokeType == 3的全动态任务，请确认任务类型");
    
    // 进行正式修改(只有这些值能够修改)
    currentTask.setExecuteExpression(executeExpression);
    currentTask.setTaskDesc(taskDesc);
    currentTask.setValidityTime(validityTime);
    // 以下是对于本次调用的参数的设定    
    if(args != null && args.length > 0) {
      List<DynamicTaskParamVo> taskParams = Lists.newArrayList();
      for(int index = 0 ; index < args.length ; index++) {
        Object arg = args[index];
        DynamicTaskParamVo param = new DynamicTaskParamVo();
        if(arg != null) {
          param.setParamType(arg.getClass());
          param.setParamValue(arg);
        } else {
          param.setParamType(Object.class);
        }
        param.setParamIndex(index);
        taskParams.add(param);
      }
      currentTask.setParams(taskParams);
    }
    return this.updateIgnorePrefix(currentTask, null);
  }

  @Override
  @Transactional
  public DynamicTaskSchedulerVo updateIgnorePrefix(DynamicTaskSchedulerVo task, String scriptContent) {
    /*
     * 21年8月，在新增了动态任务“重启”特性、全动态任务功能后，修改操作的过程变成了：
     * 注意，修改的原则是，只有一部分内容基本信息能够进行修改
     * 而任务的主状态status不允许通过这种方式进行修改
     * 
     * 1、在边界校验后，对动态任务的基本信息进行修改（注意不同类型的任务又有一些特殊属性能够修改）
     * 1.1、如果是脚本任务，则对脚本内容进行可能的修改
     * 1.2、如果是全动态任务，则对参数信息进行修改（实际上就是删除之前的，然后重新增加）
     * 
     * 2、如果当前全动态任务处于“已运行”状态，则修改运行要求为“要求重启”。
     * 注意这一个步骤只有当前正在运行的任务需要这么做，因为完成修改的任务不一定是在运行的
     * */
    
    // 1、=======
    Validate.notNull(task , "进行当前操作时，动态任务对象必须传入!!");
    SecurityContext securityContext = SecurityContextHolder.getContext();
    String userAccount = "admin";
    if(securityContext.getAuthentication() != null) {
      userAccount = securityContext.getAuthentication().getName();
    }
    Date currentTime = new Date();
    
    // 开始重新赋值——一般属性
    // 1、===================更新基本信息
    String currentId = task.getId();
    Optional<DynamicTaskSchedulerEntity> op = this.dynamicTaskSchedulerRepository.findById(currentId);
    DynamicTaskSchedulerEntity currentTask = op.orElse(null);
    Validate.notNull(currentTask ,"未发现指定的原始模型对象信");
    Validate.isTrue(currentTask.getInvokeType() != 2 , "使用注解方式创建的动态任务，不允许进行修改!!");
    Validate.notBlank(currentTask.getApplicationName() , "错误数据，动态任务的applicationName不允许为空（这可能是脏数据造成的）!!");
    Validate.notBlank(currentTask.getAppCode() , "错误数据，动态任务的appCode不允许为空（这可能是脏数据造成的）!!");
    // 进行边界校验
    this.updateValidation(task);
    currentTask.setTaskDesc(task.getTaskDesc());
    currentTask.setExecutePoint(task.getExecutePoint());
    currentTask.setExecuteExpression(task.getExecuteExpression());
    currentTask.setExpressionDesc(task.getExpressionDesc());
    currentTask.setValidityTime(task.getValidityTime());
    currentTask.setModifyTime(currentTime);
    currentTask.setModifyAccount(userAccount);
    // 进行基本信息修改
    this.dynamicTaskSchedulerRepository.save(currentTask);
    this.dynamicTaskSchedulerRepository.flush();
    
    // 1.1、==========更新脚本信息(如果是脚本执行方式)
    if(currentTask.getInvokeType() == 1) {
      String scriptName = currentTask.getScriptName();
      Validate.notBlank(scriptName , "进行更新时，脚本内容必须传入!!");
      ScriptVo script = this.scriptService.findByName(scriptName);
      Validate.notNull(script , "未找到指定的groovy脚本信息，请检查!!");
      script = this.scriptService.update(script, scriptContent);
    }
    // 1.2、==========
    else if(currentTask.getInvokeType() == 3) {
      this.dynamicTaskParamRepository.deleteByDynamicTask(currentId);
      this.dynamicTaskSchedulerRepository.flush();
      this.createTaskParams(currentTask, task.getParams());
    } else {
      throw new IllegalArgumentException("错误的动态任务执行类型，请检查!!");
    }
    
    // 2、=======
    String applicationName = currentTask.getApplicationName();
    String appCode = currentTask.getAppCode();
    // 只要当前状态正确的任务，不处于“已停止”、“要求停止”状态，就需要进行重启（不要指望动态任务在一定在当前进程运行）
    if(currentTask.getTstatus() == 1 && currentTask.getWorkingStatus() != 1 && currentTask.getWorkingStatus() != 3) {
      this.dynamicTaskSchedulerRepository.updateRestartByTaskCodesAndApplicationNameAndAppCode(new String[] {currentTask.getTaskCode()}, applicationName, appCode);
    }
    
    // 转换后进行返回
    DynamicTaskSchedulerVo taskVo = this.nebulaToolkitService.copyObjectByWhiteList(currentTask, DynamicTaskSchedulerVo.class, LinkedHashSet.class, ArrayList.class);
    return taskVo;
  }

  /**
   * 在更新一个已有的DynamicTaskSchedulerVo模型对象之前，该私有方法检查对象各属性的正确性，其id属性必须有值
   */
  private void updateValidation(DynamicTaskSchedulerVo task) {
    Validate.isTrue(!StringUtils.isBlank(task.getId()), "修改信息时，当期信息的数据编号（主键）必须有值！");
    // 基础信息判断，基本属性，需要满足not null
    Validate.notBlank(task.getTaskCode(), "任务唯一编号（只能由英文、数字、下杠构成）不能为空！");
    Validate.notBlank(task.getTaskDesc(), "任务中文说明不能为空！");
    Validate.notNull(task.getTaskType(), "任务类型不能为空！");
    // 如果当前是一次性任务
    Date currentTime = new Date();
    if(task.getTaskType() == 1) {
      Validate.notNull(task.getExecutePoint(), "一次性任务的开始时间必须填写");
      Validate.isTrue(task.getExecutePoint().getTime() > currentTime.getTime() , "一次性任务的开始时间，必须设定在当前时间点以后");
    }
    // 如果当前是周期性任务
    else if(task.getTaskType() == 2) {
      Validate.notBlank(task.getExecuteExpression() , "周期性任务必须给定有效的corn表达式!!");
      Validate.notBlank(task.getExpressionDesc() , "周期性任务必须传入corn表达式的中文描述!!");
      Validate.isTrue(task.getExecuteExpression() == null || task.getExecuteExpression().length() < 32 , "周期性任务执行表单式,填入值超过了限定长度(32)，请检查!");
      Validate.isTrue(task.getExpressionDesc() == null || task.getExpressionDesc().length() < 256 , "周期性表单时的中文,填入值超过了限定长度(256)，请检查!");
    }
    // 其它情况报错
    else {
      throw new IllegalArgumentException("动态任务只能是一次性任务或者周期性任务，请检查!!");
    }
  
    if(StringUtils.isNotBlank(task.getExecuteExpression())){
      try{
        new CronExpression(task.getExecuteExpression());
      }catch (IllegalArgumentException e){
        throw new IllegalArgumentException(String.format("cron表达式有误，请检查!【%s】",e.getMessage()));
      }
    }
    // 重复性判断，基本属性，需要满足unique = true
    String appCode = TenantUtils.getAppCode();
    String applicationName = dynamicTaskProperties.getApplicationName();
    DynamicTaskSchedulerEntity currentForTaskCode = this.dynamicTaskSchedulerRepository.findByTaskCodeAndApplicationNameAndAppCode(task.getTaskCode() , applicationName , appCode);
    Validate.isTrue(currentForTaskCode == null || StringUtils.equals(currentForTaskCode.getId() , task.getId()) , "任务唯一编号（只能由英文、数字、下杠构成）已存在,请检查");
    // 验证长度，被验证的这些字段符合特征: 字段类型为String，且不为PK，且canupdate = true
    Validate.isTrue(task.getTaskCode() == null || task.getTaskCode().length() < 128 , "任务唯一编号（只能由英文、数字、下杠构成）,填入值超过了限定长度(128)，请检查!");
    Validate.isTrue(task.getTaskDesc() == null || task.getTaskDesc().length() < 256 , "任务中文说明,填入值超过了限定长度(256)，请检查!");
    Validate.isTrue(task.getExecuteExpression() == null || task.getExecuteExpression().length() < 32 , "周期性任务执行表单式,填入值超过了限定长度(32)，请检查!");
    Validate.isTrue(task.getExpressionDesc() == null || task.getExpressionDesc().length() < 256 , "周期性表单时的中文,填入值超过了限定长度(256)，请检查!");
  }

  @Override
  @Transactional
  public void save(List<DynamicTaskSchedulerVo> tasks , String prefix, Integer invokeType  , String applicationName, String appCode) {
    this.validationPrefix(prefix);
    // 开始进行修改操作
    this.saveIgnorePrefix(tasks, invokeType, applicationName, appCode);
  }
  
  @Override
  @Transactional
  public void saveIgnorePrefix(List<DynamicTaskSchedulerVo> tasks, Integer invokeType  , String applicationName, String appCode) {
    /*
     * 在判定了入参有效性后，处理过程为：
     * 1、扫描了所有基于DynamicTaskService注解的java method形式的方法并验证后
     * 就可以开始对比和当前数据库中存在的任务数据进行对比，并分离出哪些是新增的定时器任务，哪些是需要删除（真删除）的定时器任务，哪些是需要更新的定时器任务
     * 2、开始对本次扫描发现的基于DynamicTaskService注解的java method形式的方法进行任务处理
     * 包括进行添加、删除或者修改操作
     * */
    Validate.isTrue(!CollectionUtils.isEmpty(tasks) , "需要同步的任务信息必须传入");
    Validate.notBlank(applicationName , "applicationName信息必须传入，请检查!!");
    Validate.notBlank(appCode , "appCode信息必须传入，请检查!!");
    Validate.notNull(invokeType , "invokeType信息必须传入，请检查!!");
    // 检测tasks中的appCode和applicationName
    for (DynamicTaskSchedulerVo taskItem : tasks) {
      Validate.isTrue(StringUtils.equals(taskItem.getApplicationName(), applicationName) , "每一个需要同步任务，都需要保证applicationName(%s)信息一致" , applicationName);
      Validate.isTrue(StringUtils.equals(taskItem.getAppCode(), appCode) , "每一个需要同步任务，都需要保证appCode(%s)信息一致" , appCode);
      Validate.isTrue(invokeType == taskItem.getInvokeType().intValue() ,  "每一个需要同步任务，都需要保证invokeType(%d)信息一致" , invokeType);
    }
    Collection<DynamicTaskSchedulerEntity> currentTasks = this.nebulaToolkitService.copyCollectionByWhiteList(tasks, DynamicTaskSchedulerVo.class, DynamicTaskSchedulerEntity.class, LinkedHashSet.class, ArrayList.class);
    
    // 1、======
    Date currentTime = new Date();
    Set<DynamicTaskSchedulerEntity> exsitDynamicTasks = this.dynamicTaskSchedulerRepository.findByInvokeTypeAndApplicationNameAndAppCode(invokeType, applicationName, appCode);
    if(exsitDynamicTasks == null) {
      exsitDynamicTasks = Sets.newHashSet();
    }
    Set<DynamicTaskSchedulerEntity> deleteCollections = Sets.newHashSet();
    Set<DynamicTaskSchedulerEntity> updateCollections = Sets.newHashSet();
    Set<DynamicTaskSchedulerEntity> createCollections = Sets.newHashSet();
    this.nebulaToolkitService.collectionDiscrepancy(currentTasks, exsitDynamicTasks, DynamicTaskSchedulerEntity::getTaskCode, deleteCollections, updateCollections, createCollections);

    // 2、======
    // 首先处理新增的集合信息
    if(!createCollections.isEmpty()) {
      for (DynamicTaskSchedulerEntity createItem : createCollections) {
        // 默认就是admin
        createItem.setCreateAccount("admin");
        createItem.setCreateTime(currentTime);
        createItem.setModifyAccount("admin");
        createItem.setModifyTime(currentTime);

        // 任务状态：1有效
        createItem.setTstatus(1);
        // 任务运行状态：0要求运行
        createItem.setWorkingStatus(0);
        this.dynamicTaskSchedulerRepository.save(createItem);
      }
    }
    // 然后进行删除操作（真删除）
    if(!deleteCollections.isEmpty()) {
      for (DynamicTaskSchedulerEntity deleteItem : deleteCollections) {
        String dynamicTaskSchedulerId = deleteItem.getId();
        this.dynamicTaskSchedulerLogService.deleteByDynamicTaskId(dynamicTaskSchedulerId);
        this.dynamicTaskSchedulerRepository.flush();
        this.dynamicTaskSchedulerRepository.delete(deleteItem);
      }
    }
    // 最后进行修改操作（修改时，那些无效状态的，人工停止状态的任务，不能改变）
    if(!updateCollections.isEmpty()) {
      // 基于当前入参的currentScanTasks，转成一个Map集合，以便在update操作时使用
      Map<String, DynamicTaskSchedulerEntity> targetTasksMapping = currentTasks.stream().collect(Collectors.toMap(DynamicTaskSchedulerEntity::getTaskCode, item->item ));
      for (DynamicTaskSchedulerEntity updateItem : updateCollections) {
        String taskCode = updateItem.getTaskCode();
        DynamicTaskSchedulerEntity targetItem = targetTasksMapping.get(taskCode);
        updateItem.setExecuteExpression(targetItem.getExecuteExpression());
        updateItem.setExecutePoint(targetItem.getExecutePoint());
        updateItem.setExpressionDesc(targetItem.getExpressionDesc());
        updateItem.setInvokeBeanName(targetItem.getInvokeBeanName());
        updateItem.setInvokeMethod(targetItem.getInvokeMethod());
        updateItem.setTaskDesc(targetItem.getTaskDesc());
        updateItem.setModifyAccount("admin");
        updateItem.setModifyTime(currentTime);
        updateItem.setAppCode(targetItem.getAppCode());
        updateItem.setApplicationName(targetItem.getApplicationName());
        updateItem.setValidityTime(targetItem.getValidityTime());
        this.dynamicTaskSchedulerRepository.save(updateItem);
      }
    }

  }

  @Override
  @Transactional
  public void invalid(String[] taskCodes) {
    Validate.isTrue(taskCodes != null && taskCodes.length > 0 , "变更任务为无效时，至少传入一个动态定时任务的编号code!!");
    Validate.isTrue(taskCodes.length > 0 , "变更任务为无效时，至少传入一个动态定时任务的编号code!!");
    String appCode = TenantUtils.getAppCode();
    String applicationName = dynamicTaskProperties.getApplicationName();
    // 只有有效的已停止任务，才能变更为无效
    Set<DynamicTaskSchedulerEntity> dynamicTasks = this.dynamicTaskSchedulerRepository.findByTaskCodesAndApplicationNameAndAppCode(Lists.newArrayList(taskCodes), applicationName, appCode);
    Set<String> dbDynamicEffectiveTaskCodes = Sets.newHashSet();
    if(!CollectionUtils.isEmpty(dynamicTasks)) {
      dbDynamicEffectiveTaskCodes = dynamicTasks.stream().filter(item -> item.getTstatus() == 1 && item.getWorkingStatus() == 3).map(DynamicTaskSchedulerEntity::getTaskCode).collect(Collectors.toSet());
    }
    Validate.isTrue(Sets.difference(dbDynamicEffectiveTaskCodes, Sets.newHashSet(taskCodes)).isEmpty() , "变更任务为无效时，至少有一个任务的状态不为“有效”或不为“已停止”!!");
    // 开始置为无效
    this.dynamicTaskSchedulerRepository.updateInvalidByTaskCodesAndApplicationNameAndAppCode(taskCodes, applicationName, appCode);
  }

  @Override
  @Transactional
  public void effective(String[] taskCodes) {
    Validate.notNull(taskCodes , "变更任务为有效时，至少传入一个动态定时任务的编号code!!");
    Validate.isTrue(taskCodes.length > 0 , "变更任务为有效时，至少传入一个动态定时任务的编号code!!");
    String appCode = TenantUtils.getAppCode();
    String applicationName = dynamicTaskProperties.getApplicationName();
    // 只有无效任务，才能变更为有效；且变更后的执行状态只能是“已停止”
    Set<DynamicTaskSchedulerEntity> dynamicTasks = this.dynamicTaskSchedulerRepository.findByTaskCodesAndApplicationNameAndAppCode(Lists.newArrayList(taskCodes), applicationName, appCode);
    Set<String> dbDynamicInvalidTaskCodes = Sets.newHashSet();
    if(!CollectionUtils.isEmpty(dynamicTasks)) {
      dbDynamicInvalidTaskCodes = dynamicTasks.stream().filter(item -> item.getTstatus() == 0).map(DynamicTaskSchedulerEntity::getTaskCode).collect(Collectors.toSet());
    }
    Validate.isTrue(Sets.difference(dbDynamicInvalidTaskCodes, Sets.newHashSet(taskCodes)).isEmpty() , "变更任务为有效时，至少有一个任务的状态不为“无效”!!");
    // 开始置为有效
    this.dynamicTaskSchedulerRepository.updateEffectiveByTaskCodesAndApplicationNameAndAppCode(taskCodes, applicationName, appCode);
  }

  @Override
  @Transactional
  public void start(String[] taskCodes) {
    Validate.notNull(taskCodes , "要求任务启动时，至少传入一个动态定时任务的编号code!!");
    Validate.isTrue(taskCodes.length > 0 , "要求任务启动时，至少传入一个动态定时任务的编号code!!");
    String appCode = TenantUtils.getAppCode();
    String applicationName = dynamicTaskProperties.getApplicationName();
    // 只有处于“已停止”的有效任务，才能要求启动
    Set<DynamicTaskSchedulerEntity> dynamicTasks = this.dynamicTaskSchedulerRepository.findByTaskCodesAndApplicationNameAndAppCode(Lists.newArrayList(taskCodes), applicationName, appCode);
    Set<String> dbDynamicStopedTaskCodes = Sets.newHashSet();
    if(!CollectionUtils.isEmpty(dynamicTasks)) {
      dbDynamicStopedTaskCodes = dynamicTasks.stream().filter(item -> item.getTstatus() == 1 && item.getWorkingStatus() == 3).map(DynamicTaskSchedulerEntity::getTaskCode).collect(Collectors.toSet());
    }
    Validate.isTrue(Sets.difference(dbDynamicStopedTaskCodes, Sets.newHashSet(taskCodes)).isEmpty() , "要求任务启动时，至少有一个任务的状态不为“有效”状态或者不为“已停止”状态!!");
    // 置为要求启动状态
    this.dynamicTaskSchedulerRepository.updateReadyRunningByTaskCodesAndApplicationNameAndAppCode(taskCodes, applicationName, appCode);
  }

  @Override
  @Transactional
  public void stop(String[] taskCodes) {
    Validate.notNull(taskCodes , "要求任务停止时，至少传入一个动态定时任务的编号code!!");
    Validate.isTrue(taskCodes.length > 0 , "要求任务停止时，至少传入一个动态定时任务的编号code!!");
    String appCode = TenantUtils.getAppCode();
    String applicationName = dynamicTaskProperties.getApplicationName();
    // 只有处于“运行”状态的任务，才能进入“要求停止”状态
    Set<DynamicTaskSchedulerEntity> dynamicTasks = this.dynamicTaskSchedulerRepository.findByTaskCodesAndApplicationNameAndAppCode(Lists.newArrayList(taskCodes), applicationName, appCode);
    Set<String> dbDynamicRequestStopedTaskCodes = Sets.newHashSet();
    if(!CollectionUtils.isEmpty(dynamicTasks)) {
      dbDynamicRequestStopedTaskCodes = dynamicTasks.stream().filter(item -> item.getWorkingStatus() == 2).map(DynamicTaskSchedulerEntity::getTaskCode).collect(Collectors.toSet());
    }
    Validate.isTrue(Sets.difference(dbDynamicRequestStopedTaskCodes, Sets.newHashSet(taskCodes)).isEmpty() , "要求任务停止时，至少有一个任务的状态不为“已运行”状态!!");
    // 置为“要求停止”状态
    this.dynamicTaskSchedulerRepository.updateReadyStopByTaskCodesAndApplicationNameAndAppCode(taskCodes, applicationName, appCode);
  }

  @Override
  @Transactional
  public void deleteByTaskcodes(String[] taskCodes) {
    // TODO Auto-generated method stub
    Validate.notNull(taskCodes , "删除任务时，至少传入一个动态定时任务的编号code!!");
    Validate.isTrue(taskCodes.length > 0 , "删除任务时，至少传入一个动态定时任务的编号code!!");
    String appCode = TenantUtils.getAppCode();
    String applicationName = dynamicTaskProperties.getApplicationName();
    // 只有处于“已停止”状态的任务，才能被删除
    Set<DynamicTaskSchedulerEntity> dynamicTasks = this.dynamicTaskSchedulerRepository.findByTaskCodesAndApplicationNameAndAppCode(Lists.newArrayList(taskCodes), applicationName, appCode);
    Set<String> dbDynamicStopedTaskCodes = Sets.newHashSet();
    if(!CollectionUtils.isEmpty(dynamicTasks)) {
      dbDynamicStopedTaskCodes = dynamicTasks.stream().filter(item -> item.getWorkingStatus() == 3).map(DynamicTaskSchedulerEntity::getTaskCode).collect(Collectors.toSet());
    }
    Validate.isTrue(Sets.difference(dbDynamicStopedTaskCodes, Sets.newHashSet(taskCodes)).isEmpty() , "删除任务时，至少有一个任务的状态不为“已停止”状态!!");
    // 开始进行真删除
    for (DynamicTaskSchedulerEntity task : dynamicTasks) {
      // TODO 先删除该定时任务的日志信息
      // this.dynamicTaskSchedulerLogService.deleteByDynamicTaskSchedulerId(task.getId());
      this.dynamicTaskSchedulerRepository.delete(task);
    }
  }

  @Override
  public List<DynamicTaskSchedulerVo> findAll() {
    List<DynamicTaskSchedulerEntity> dynamicTaskSchedulers = this.dynamicTaskSchedulerRepository.findAll();
    if(!CollectionUtils.isEmpty(dynamicTaskSchedulers)) {
      return null;
    }
    return Lists.newArrayList(this.nebulaToolkitService.copyCollectionByWhiteList(dynamicTaskSchedulers, DynamicTaskSchedulerEntity.class, DynamicTaskSchedulerVo.class, LinkedHashSet.class, ArrayList.class));
  }

  @Override
  public DynamicTaskSchedulerVo findByTaskCodeAndApplicationNameAndAppCode(String taskCode, String applicationName, String appCode) {
    if(StringUtils.isAnyBlank(taskCode , applicationName , appCode)) {
      return null;
    }
    DynamicTaskSchedulerEntity dynamicTask = this.dynamicTaskSchedulerRepository.findByTaskCodeAndApplicationNameAndAppCode(taskCode, applicationName, appCode);
    if(dynamicTask == null) {
      return null;
    }
    return this.nebulaToolkitService.copyObjectByWhiteList(dynamicTask, DynamicTaskSchedulerVo.class, LinkedHashSet.class, ArrayList.class);
  }

  @Override
  public Page<DynamicTaskSchedulerVo> findByConditions(Pageable pageable, DynamicTaskSchedulerDto conditions) {
    if(pageable == null) {
      pageable = PageRequest.of(0, 50);
    }
    if(conditions == null) {
      conditions = new DynamicTaskSchedulerDto();
    }
    String appCode = TenantUtils.getAppCode();
    String applicationName = dynamicTaskProperties.getApplicationName();
    conditions.setAppCode(appCode);
    conditions.setApplicationName(applicationName);
    
    // 开始查询
    Page<DynamicTaskSchedulerEntity> results = this.dynamicTaskSchedulerRepository.findByConditions(pageable, conditions);
    if(results == null || results.isEmpty()) {
      return null;
    }
    Collection<DynamicTaskSchedulerVo> dynamicTaskSchedulerVos = this.nebulaToolkitService.copyCollectionByWhiteList(results, DynamicTaskSchedulerEntity.class, DynamicTaskSchedulerVo.class, LinkedHashSet.class, ArrayList.class);
    return new PageImpl<>(Lists.newArrayList(dynamicTaskSchedulerVos), pageable, results.getTotalElements());
  }
}
