package com.bizunited.nebula.task.local.service.scheduler;

import com.bizunited.nebula.task.local.configuration.DynamicTaskProperties;
import com.bizunited.nebula.task.local.entity.DynamicTaskSchedulerEntity;
import com.bizunited.nebula.task.service.DynamicTaskParamVoService;
import com.bizunited.nebula.task.service.DynamicTaskSchedulerLogVoService;
import com.bizunited.nebula.task.service.DynamicTaskSchedulerVoService;
import com.bizunited.nebula.task.vo.DynamicTaskParamVo;
import com.bizunited.nebula.task.vo.DynamicTaskSchedulerLogVo;
import com.bizunited.nebula.task.vo.DynamicTaskSchedulerVo;
import com.google.common.collect.Maps;
import com.bizunited.nebula.common.service.NebulaToolkitService;
import com.bizunited.nebula.common.util.tenant.TenantContextHolder;
import com.bizunited.nebula.script.vo.ScriptVo;
import com.bizunited.nebula.script.exception.InvokeProxyException;
import com.bizunited.nebula.script.service.ScriptVoService;

import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.util.ArrayList;
import java.util.Date;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Map;

import org.apache.commons.lang3.Validate;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.NoSuchBeanDefinitionException;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.ApplicationContext;
import org.springframework.context.annotation.Lazy;
import org.springframework.context.annotation.Scope;
import org.springframework.stereotype.Component;
import org.springframework.util.CollectionUtils;

/**
 * 该任务负责正式执行动态任务，根据动态任务类型、执行方式的不一样，执行的处理细节也不一样</br>
 * 请注意，在老版本nebula中，动态日志任务的记录使用InvokeResponseHandle处理方式，这个方式目前被取消掉了，但在后续的迭代中还会恢复相关的处理设计
 * @author yinwenjie
 */
@Component("_DynamicTask")
@Scope("prototype")
@Lazy
public class DynamicTask implements Runnable {
  /**
   * 当期的动态任务
   */
  private DynamicTaskSchedulerEntity currentTask;
  @Autowired
  private DynamicTaskParamVoService dynamicTaskParamVoService;
  @Autowired
  private DynamicTaskSchedulerVoService dynamicTaskSchedulerVoService;
  @Autowired
  private DynamicTaskSchedulerLogVoService dynamicTaskSchedulerLogVoService;
  @Autowired
  private NebulaToolkitService nebulaToolkitService;
  @Autowired
  private ScriptVoService scriptService;
  @Autowired
  private ApplicationContext applicationContext;
  @Autowired
  private DynamicTaskProperties dynamicTaskProperties;
  /**
   * 日志
   */
  private static final Logger LOGGER = LoggerFactory.getLogger(DynamicTask.class);
  
  /**
   * 动态任务周期性执行内容，其核心点是创建一个新的代理责任链，该责任链中有 指定的script脚本执行过程和 日志记录过程
   * @param currentTask 
   * @param ctx 
   */
  public DynamicTask(DynamicTaskSchedulerEntity currentTask) {
    this.currentTask = currentTask;
  }
  
  @Override
  public void run() {
   /*
    * - 这里的执行过程是一个默认的责任链调用，过程包括：
    * 1、调用前准备各种数据
    * 2、正式进行调用
    * 3、正式执行后，视当前任务的执行类型（单次还是多次）
    * 4、调用动态任务的任务记录服务（无论调用是否成功）
    * 以上调用过程不需要记录返回值
    * */
    boolean success = false;
    Date endTime = null;
    Date startTime = null;
    String errorContents = null;
    String appCode = this.currentTask.getAppCode();
    String taskId = this.currentTask.getId();
    String taskCode = currentTask.getTaskCode();
    String applicationName = this.dynamicTaskProperties.getApplicationName();
    try {
      Integer invokeType = currentTask.getInvokeType();
      TenantContextHolder.setApp(appCode);
      startTime = new Date();
    
      // 1、=======
      // 转VO
      DynamicTaskSchedulerVo taskVo = nebulaToolkitService.copyObjectByWhiteList(currentTask, DynamicTaskSchedulerVo.class, LinkedHashSet.class, ArrayList.class);
      List<DynamicTaskParamVo> dynamicTaskParams = this.dynamicTaskParamVoService.findByDynamicTask(taskId);
      if(!CollectionUtils.isEmpty(dynamicTaskParams)) {
        taskVo.setParams(dynamicTaskParams);
      }
      
      // 2、========
      // 如果条件成立，说明是执行一个groovy脚本性质的定时任务
      if(invokeType == 1) {
        this.invokeGroovyScript(taskVo);
      }
      // 如果条件成立，说明是执行一个基于注解的java method定时任务
      else if(invokeType == 2) {
        this.invokeAnnMethod(taskVo);
      } 
      // 如果条件成立，说明是执行一个全动态方法的定时任务
      else if(invokeType == 3) {
        this.invokeDynamicMethod(taskVo);
      } else {
        // 其它情况抛出异常
        throw new NoSuchBeanDefinitionException(DynamicTaskSchedulerEntity.class);
      } 
      endTime = new Date();
      success = true;
    } catch (InvokeProxyException | RuntimeException e) {
      errorContents = e.getMessage();
      endTime = new Date();
      success = false;
      LOGGER.error(errorContents , e);
    } finally {
      // 3、=======
      this.updateInvokeType(currentTask);
      // 4、=======
      DynamicTaskSchedulerLogVo dynamicTaskSchedulerLog = new DynamicTaskSchedulerLogVo();
      dynamicTaskSchedulerLog.setEndTime(endTime);
      dynamicTaskSchedulerLog.setErrorContents(errorContents);
      dynamicTaskSchedulerLog.setStartTime(startTime);
      dynamicTaskSchedulerLog.setSuccess(success);
      dynamicTaskSchedulerLog.setAppCode(appCode);
      dynamicTaskSchedulerLog.setApplicationName(applicationName);
      dynamicTaskSchedulerLog.setDynamicTaskId(taskId);
      dynamicTaskSchedulerLog.setTaskCode(taskCode);
      this.dynamicTaskSchedulerLogVoService.create(dynamicTaskSchedulerLog);
      TenantContextHolder.clean();
    }
  }
  
  /**
   * 在动态任务执行完成后，可能需要变更任务的执行状态信息
   * @param currentTask
   */
  private void updateInvokeType(DynamicTaskSchedulerEntity currentTask) {
    /*
     * 如果是一次性任务，则在执行后将工作状态变更为“要求停止”
     * */
    Integer taskType = currentTask.getTaskType();
    if(taskType == 1) {
      this.dynamicTaskSchedulerVoService.stop(new String[] {currentTask.getTaskCode()});
    }
  }

  /**
   * 该私有方法用于执行Groovy脚本性质的定时器任务
   */
  private void invokeGroovyScript(DynamicTaskSchedulerVo currentTask) throws InvokeProxyException {
    // 脚本信息
    String scriptName = currentTask.getScriptName();
    ScriptVo scriptVo = this.scriptService.findByName(scriptName);
    Validate.notNull(scriptVo , "在进行动态任务执行时，未发现指定的groovy脚本信息，请检查%s!" , currentTask.getId());
    String scriptId = scriptVo.getId();
    // 内容对象信息
    Map<String, Object> params = Maps.newHashMap();
    params.put("ctx", this.applicationContext);
    params.put("task", currentTask);

    // 调用链上下文对象引入，并完成执行
    this.scriptService.invoke(new String[] {scriptId}, params);
  }
  
  /**
   * 该私有方法用于执行基于DynamicTaskService注解的java method方法
   */
  private void invokeAnnMethod(DynamicTaskSchedulerVo currentTask) throws InvokeProxyException {
    /*
     * 处理过程为：
     * 1、首先拿到currentTask之中的基本信息，并判定有效性
     * 2、然后根据其中的得到的bean的名字、方法的名字，开始进行调用
     * 注意其中的传参要求：如果其中要求了InvokeProxyContext参数，还要拼凑参数进行传入
     * */
    // 1、=======
    String taskCode = currentTask.getTaskCode();
    String invokeBeanName = currentTask.getInvokeBeanName();
    Validate.notBlank(invokeBeanName , "执行动态调用任务时，发现基于DynamicTaskService注解的动态执行任务%s，其invokeBeanName信息非法" , taskCode);
    String invokeMethod = currentTask.getInvokeMethod();
    Validate.notBlank(invokeMethod , "执行动态调用任务时，发现基于DynamicTaskService注解的动态执行任务%s，其invokeMethod信息非法" , taskCode);
    Object currentBean = null;
    try {
      currentBean = applicationContext.getBean(invokeBeanName);
    } catch(NoSuchBeanDefinitionException e) {
      throw new IllegalArgumentException(String.format("执行动态调用任务时，未发现名为%s的spring bean，请检查" , invokeBeanName));
    }

    // 2、=======
    Class<?> beanClass = applicationContext.getType(invokeBeanName);
    Method beanClassMethod = null;
    // 开始进行调用
    try {
      beanClassMethod = beanClass.getMethod(invokeMethod, new Class[] {});
      beanClassMethod.invoke(currentBean, new Object[] {});
    } catch(NoSuchMethodException e) {
      throw new IllegalArgumentException(String.format("执行动态调用任务时，未发现名为%s的spring bean中指定的方法%s，请检查" , invokeBeanName , invokeMethod));
    } catch(InvocationTargetException | IllegalAccessException e) {
      throw new IllegalArgumentException("执行动态调用任务时,方法执行错误，请检查" , e);
    }
  }

  /**
   * 该私有方法进行全动态方法的调用，全动态方法调用支持开发者的各种参数传递
   * @param currentTask 
   */
  private void invokeDynamicMethod(DynamicTaskSchedulerVo currentTask) {
    /*
     * 处理过程可参见invokeAnnMethod方法，只是这种调用支持开发人员传递的参数
     * */
    // 1、=======
    String taskCode = currentTask.getTaskCode();
    String invokeBeanName = currentTask.getInvokeBeanName();
    Validate.notBlank(invokeBeanName , "执行动态调用任务时，发现全动态任务%s，其invokeBeanName信息非法" , taskCode);
    String invokeMethod = currentTask.getInvokeMethod();
    Validate.notBlank(invokeMethod , "执行动态调用任务时，发现全动态任务%s，其invokeMethod信息非法" , taskCode);
    Object currentBean = null;
    try {
      currentBean = applicationContext.getBean(invokeBeanName);
    } catch(NoSuchBeanDefinitionException e) {
      throw new IllegalArgumentException(String.format("执行动态调用任务时，未发现名为%s的spring bean，请检查" , invokeBeanName));
    }

    // 2、=======
    Class<?> beanClass = applicationContext.getType(invokeBeanName);
    Method beanClassMethod = null;
    // 开始进行调用(支持可能的参数传入)
    try {
      List<DynamicTaskParamVo> params = currentTask.getParams();
      if(!CollectionUtils.isEmpty(params)) {
        Object[] values = params.stream().map(DynamicTaskParamVo::getParamValue).toArray(Object[]::new);
        Class<?>[] classes = params.stream().map(DynamicTaskParamVo::getParamType).toArray(Class[]::new);
        beanClassMethod = beanClass.getMethod(invokeMethod, classes);
        beanClassMethod.invoke(currentBean, values);
      } else {
        beanClassMethod = beanClass.getMethod(invokeMethod, new Class[] {});
        beanClassMethod.invoke(currentBean, new Object[] {});
      }
    } catch(NoSuchMethodException e) {
      throw new IllegalArgumentException(String.format("执行全动态调用任务时，未发现名为%s的spring bean中指定的方法%s，请检查" , invokeBeanName , invokeMethod));
    } catch(InvocationTargetException | IllegalAccessException e) {
      throw new IllegalArgumentException("执行全动态调用任务时，方法执行错误，请检查" , e);
    }
  }

}
