package com.bizunited.nebula.task.service;

import java.util.Date;
import java.util.List;

import org.springframework.data.domain.Page;
import org.springframework.data.domain.Pageable;

import com.bizunited.nebula.task.dto.DynamicTaskSchedulerDto;
import com.bizunited.nebula.task.vo.DynamicTaskSchedulerVo;

/**
 * 动态任务调度服务
 * @author yinwenjie
 * @since 2021-08-01
 */
public interface DynamicTaskSchedulerVoService {
  /**
   * 当这个进程抢占到了动态调度任务的执行权限后，通过该方法进行“可执行”任务的加载</br>
   * 注意每一次调度加载只关注指定applicationName，指定appCode下面的动态任务
   * @param applicationName 指定的应用程序名
   * @param appCode 指定的顶级租户业务编号
   */
  public void loading(String applicationName , String appCode);
  /**
   * 当这个进程没有抢占到了动态调度任务的执行权限时，通过该方法进行“可执行”任务的卸载
   * @param applicationName 指定的应用程序名 
   * @param appCode 指定的顶级租户业务编号 
   */
  public void unloading(String applicationName, String appCode);
  /**
   * 创建动态任务前，需要使用该方法获得预授权</br>
   * 预授权成功后，才能通过预授权信息进行添加，
   * @return 
   */
  public String preModify();
  /**
   * 创建一个新的动态任务，其中重要的属性为：</br>
   * invokeType：定时器任务内容的执行方式</br>
   * taskType：任务执行方式</br>
   * workingStatus：任务工作状态
   * @param task 其它属性请参见源代码中的说明
   * @param scriptContent 脚本内容，如果不是创建脚本指定任务，则可以忽略传入
   * @return 
   */
  public DynamicTaskSchedulerVo create(DynamicTaskSchedulerVo task , String scriptContent);
  /**
   * 该方法只能添加invokeType == 3执行方式的动态任务（全动态任务）；使得该任务能够按照指定的运行周期 executeExpression进行一次一次的执行</br>
   * 如果该任务创建时设定了validityTime，那么执行周期将在validityTime时间点后失效。
   * @param taskCode 确认的任务编号（必须填写，只支持数字，“-”，大小写字母，特别不支持“_”）
   * @param invokeBeanName Ioc容器中的具体bean的名称
   * @param method 必须指定的周期性调用方法，该方法必须注入了Ioc容器，并且是以public的访问方式被修饰的
   * @param executeExpression 必须指定的cron表达式
   * @param validityTime 可能指定的周期性全动态任务的有效期，有效期可以不指定但一旦指定就需要大于当前时间点。
   * @param taskDesc 必须设定的任务描述信息
   * @param args 方法参数，注意这些参数都必须是能够被串行化的参数，否则系统会报错
   */
  public DynamicTaskSchedulerVo createIgnorePrefix(String taskCode , String invokeBeanName , String method , String executeExpression , Date validityTime , String taskDesc , Object... args);
  /**
   * 该方法只能添加invokeType == 3执行方式的动态任务（全动态任务）；这些被添加的周期性任务根据taskType分为两种场景：</p>
   * taskType == 1时：一次性执行任务，一次性执行任务时，可以通过executeExpression参数或者executePoint参数，确定这个一次性执行任务方式</p>
   * taskType == 2时：周期性执行任务，周期性执行任务时，可以通过executeExpression参数，确定这个周期性执行任务方式</p>
   * @param taskCode 确认的任务编号（必须填写，只支持数字，“-”，大小写字母，特别不支持“_”）
   * @param invokeBeanName Ioc容器中的具体bean的名称
   * @param method 必须指定的周期性调用方法，该方法必须注入了Ioc容器，并且是以public的访问方式被修饰的
   * @param taskType taskType == 1时：一次性执行任务；taskType == 2时：周期性执行任务
   * @param executeExpression 指定的cron表达式，当taskType == 1时必须指定；当taskType == 2时，可能传入
   * @param executePoint 指定的一次性执行时间点，当taskType == 1时必须指定
   * @param validityTime 可能指定的周期性全动态任务的有效期，有效期可以不指定但一旦指定就需要大于当前时间点。
   * @param taskDesc 必须设定的任务描述信息
   * @param args 方法参数，注意这些参数都必须是能够被串行化的参数，否则系统会报错
   */
  public DynamicTaskSchedulerVo createIgnorePrefix(String taskCode , String invokeBeanName , String method , Integer taskType , String executeExpression , Date executePoint  , Date validityTime , String taskDesc , Object... args);
  /**
   * 创建一个新的动态任务——在忽略修改令牌的场景下 
   * @return 
   */
  public DynamicTaskSchedulerVo createIgnorePrefix(DynamicTaskSchedulerVo task , String scriptContent);
  /**
   * 更新一个指定的task的基本信息，只有状态同时处于“有效”的“未运行”任务，才能进行更新操作
   * @param scriptContent 脚本内容，如果不是修改脚本指定任务中的脚本内容，则可以忽略传入
   * @return
   */
  public DynamicTaskSchedulerVo update(DynamicTaskSchedulerVo task, String scriptContent);
  /**
   * 针对已经存在的动态任务（任务业务编号），对动态任务的基本信息进行修改，只能修改执行周期信息、描述信息、过期和参数信息，对于执行方法不能进行修改
   * @param taskCode 指定的taskCode，如果在当前applicationName和appCode下不存在指定的taskCode，则会抛出异常
   * @param executeExpression 新指定的执行表达式（必须传入）
   * @param validityTime 新指定的过期时间（如果不传入，则表示删除过期时间）
   * @param taskDesc 新指定的任务描述信息（必须传入）
   * @param args 新指定的各类参数信息
   * @return 
   */
  public DynamicTaskSchedulerVo updateIgnorePrefix(String taskCode , String executeExpression , Date validityTime , String taskDesc , Object... args);
  /**
   * 更新一个指定的task的基本信息，只有状态同时处于“有效”的“未运行”任务，才能进行更新操作，</br>
   * 在忽略修改令牌的场景下。
   * @return
   */
  public DynamicTaskSchedulerVo updateIgnorePrefix(DynamicTaskSchedulerVo task, String scriptContent); 
  /**
   * 该方法对当前已存在于系统中的动态任务，参照tasks进行更新处理</br>
   * 对于已存在的任务进行修改、对于未存在的任务进行添加、对于不存在的任务进行删除</br>
   * 该任务只在指定的applicationName、appCode范围内起作用
   * @param tasks 需要进行同步的动态任务信息
   * @param prefix 由于是多个任务信息，所以令牌需要作为统一参数进行传入
   * @param invokeType 动态任务的执行方式范围：1：groovy脚本执行器；2：基于注解的Java method；3、全动态调用任务
   * @param applicationName 指定的applicationName范围
   * @param appCode 指定的一级租户（appCode）范围
   */
  public void save(List<DynamicTaskSchedulerVo> tasks , String prefix , Integer invokeType , String applicationName, String appCode);
  /**
   * 该方法的逻辑可参见save，只是该方法不再验证操作令牌
   */
  public void saveIgnorePrefix(List<DynamicTaskSchedulerVo> tasks , Integer invokeType , String applicationName, String appCode);
  /**
   * 将指定的一个或者多个动态任务的状态变更为“无效”，“无效”状态的动态任务不允许再运行，正在运行的也会被停止
   * @param taskCodes 可以一次传入多个需要操作的动态任务编号
   */
  public void invalid(String[] taskCodes);
  /**
   * 将指定的一个或者多个动态任务的状态从“失效”变为“有效”任务，只有有效任务才能被启动运行
   * @param taskCodes 可以一次传入多个需要操作的动态任务编号
   */
  public void effective(String[] taskCodes);
  /**
   * 要求启动指定的任务（一个或者多个），由于当前的“启动请求”可能不来自于master节点，所以这里只是更改任务状态，以便master节点下一次任务状态扫描时会启动该节点</br>
   * 被启动的任务只能是“有效”的任务。
   * @param taskCodes 可以一次传入多个需要操作的动态任务编号
   */
  public void start(String[] taskCodes);
  /**
   * 要求停止动态任务（一个或者多个），被停止的任务必须是已运行或者是要求运行的</br>
   * 被停止的任务，还是“有效”任务；除非后来被设定为“无效任务”，否则停止的动态任务都可以再被启动
   * @param taskCodes 可以一次传入多个需要操作的动态任务编号
   */
  public void stop(String[] taskCodes);
  /**
   * 根据taskcode删除动态任务（一个或者多个）接口，被删除的任务只能是“无效任务”
   * @param taskCodes 可以一次传入多个需要操作的动态任务编号
   */
  public void deleteByTaskcodes(String[] taskCodes);
  /**
   * 查询当前已设定所有动态任务，无论其状态如何；并且返回信息按照创建时间倒序排列。
   * 查询信息还包括了其所有的直接关联信息
   * @return
   */
  public List<DynamicTaskSchedulerVo> findAll();
  /**
   * 查询任务编号对应的动态任务信息，也包括可能的创建者、修改者信息
   * @param applicationName 应用程序
   * @param taskCode 指定的任务编号
   * @return
   */
  public DynamicTaskSchedulerVo findByTaskCodeAndApplicationNameAndAppCode(String taskCode, String applicationName , String appCode);
  /**
   * 该方法用于按照分页标准，对动态任务列表进行查询
   * @param pageable 分页信息
   * @param conditions 查询条件
   */
  public Page<DynamicTaskSchedulerVo> findByConditions(Pageable pageable , DynamicTaskSchedulerDto conditions);
}