package com.bizunited.nebula.venus.service.local.service.internal;

import com.bizunited.nebula.common.service.NebulaToolkitService;
import com.bizunited.nebula.common.util.FileUtils;
import com.bizunited.nebula.common.util.ZipFileUtils;
import com.bizunited.nebula.venus.sdk.config.SimpleVenusProperties;
import com.bizunited.nebula.venus.sdk.constant.Constants;
import com.bizunited.nebula.venus.sdk.event.VenusFileEventListener;
import com.bizunited.nebula.venus.sdk.service.document.DocumentService;
import com.bizunited.nebula.venus.sdk.service.image.ImageHandleService;
import com.bizunited.nebula.venus.sdk.vo.DocumentVo;
import com.bizunited.nebula.venus.service.local.elasticsearch.repository.DocumentEsRepository;
import com.bizunited.nebula.venus.service.local.entity.DocumentEntity;
import com.bizunited.nebula.venus.service.local.repository.DocumentRepository;
import com.google.common.collect.Lists;
import com.google.common.collect.Sets;
import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.io.ObjectInputStream;
import java.io.ObjectOutputStream;
import java.nio.charset.StandardCharsets;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Date;
import java.util.HashMap;
import java.util.HashSet;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Map;
import java.util.Random;
import java.util.Set;
import java.util.UUID;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.util.stream.Collectors;
import java.util.zip.ZipEntry;
import java.util.zip.ZipFile;
import java.util.zip.ZipOutputStream;
import javax.transaction.Transactional;
import org.apache.commons.io.IOUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.security.core.Authentication;
import org.springframework.security.core.context.SecurityContext;
import org.springframework.security.core.context.SecurityContextHolder;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;
import org.springframework.web.multipart.MultipartFile;


/**
 * 文档服务实现</br>
 * TODO 该服务模块没有被验证过，需要重新进行功能梳理
 * @author: pangdajin yinwenjie
 */
@Service("DocumentServiceImpl")
public class DocumentServiceImpl implements DocumentService {

  private static final Logger LOGGER = LoggerFactory.getLogger(DocumentServiceImpl.class);

  /**
   * 文档文件后缀
   */
  private static final String MD_FILE_SUFFIX = "txt";
  /**
   * zip文档的文件名
   */
  private static final String ZIP_DOCUMENT_FILE_NAME = "documents.in";
  /**
   * 默认排序
   */
  private static final Integer DEFAULT_SORT = 1000;
  @Autowired
  private SimpleVenusProperties venusProperties;
  @Autowired
  private DocumentRepository documentRepository;
  @Autowired
  private NebulaToolkitService nebulaToolkitService;
  @Autowired
  private DocumentEsRepository documentEsRepository;
  @Autowired
  private ImageHandleService imageViewService;
  @Autowired
  private VenusFileEventListener venusFileEventListener;

  /**
   * 保存文档
   * @param documentVo
   * @param principal
   */
  @Transactional
  @Override
  public DocumentVo create(DocumentVo documentVo) {
    //验证创建人信息
    String account = this.findAccount();
    this.createValidation(documentVo);
    DocumentEntity document = nebulaToolkitService.copyObjectByWhiteList(documentVo, DocumentEntity.class, HashSet.class, ArrayList.class, new String[] {"parent"});
    document.setCreateAccount(account);
    document.setModifyAccount(account);
    Date nowDate = new Date();
    document.setCreateTime(nowDate);
    document.setModifyTime(nowDate);
    Integer sort = documentVo.getSort();
    if (sort == null){
      document.setSort(DEFAULT_SORT);
    }
    this.saveDucomentContent(document);
    this.documentRepository.saveAndFlush(document);
    this.saveEsDocument(document);
    return nebulaToolkitService.copyObjectByWhiteList(document, DocumentVo.class, HashSet.class, ArrayList.class);
  }

  /**
   * 验证保存文档
   * @param documentVo
   */
  private void createValidation(DocumentVo documentVo) {
    //验证文档
    Validate.notNull(documentVo, "文档对象必须传入，请检查!!");
    Validate.isTrue(documentVo.getId() == null, "新增文档时，不能传入ID！");
    this.commonValidation(documentVo, null);
  }

  /**
   * 验证保存文档
   * @param document
   */
  private void createValidation(DocumentEntity document) {
    //验证文档
    Validate.notNull(document, "文档对象必须传入，请检查!!");
    DocumentVo documentVo = nebulaToolkitService.copyObjectByWhiteList(document, DocumentVo.class, null, null, "parent");
    this.commonValidation(documentVo, null);
  }

  /**
   * 验证同级title不能重复
   * @param documentVo
   */
  private void commonValidation(DocumentVo documentVo, String currentTitle) {
    String title = documentVo.getTitle();
    Validate.notBlank(title,"标题不能为空");
    //验证同级title不能重复
    DocumentVo parent = documentVo.getParent();
    if (parent != null){
      Validate.notBlank(parent.getId(), "父id不能为空");
      DocumentEntity parentDocument = this.documentRepository.findById(parent.getId()).orElse(null);
      Validate.notNull(parentDocument,"父文档不存在！");
    }
    if(currentTitle == null || !currentTitle.equals(title)){
      if (parent == null){
        long count = this.documentRepository.countByNullParentAndTitle(documentVo.getTitle());
        Validate.isTrue(count == 0 ,"title不能重名，请重新传入！！");
      }else{
        long count = this.documentRepository.countByParentAndTitle(documentVo.getTitle(), parent.getId());
        Validate.isTrue(count == 0 ,"title不能重名，请重新传入！！");
      }
    }

  }

  /**
   * 修改文档
   * @param documentVo
   * @param principal
   */
  @Transactional
  @Override
  public DocumentVo update(DocumentVo documentVo) {
    //验证更新人信息
    String account = this.findAccount();
    //验证文档
    this.updateValidation(documentVo);
    DocumentEntity document = this.documentRepository.findById(documentVo.getId()).orElse(null);
    Validate.notNull(document, "要修改的文档对象不存在!!");
    this.commonValidation(documentVo, document.getTitle());
    //如果标题和原标题不相同则修改标题,同时执行文档内容的保存或更新
    document.setTitle(documentVo.getTitle());
    document.setContent(documentVo.getContent());
    document.setModifyAccount(account);
    document.setModifyTime(new Date());
    document.setSort(documentVo.getSort());

    DocumentVo parentVo = documentVo.getParent();
    if (parentVo != null ) {
      Validate.isTrue(!parentVo.getId().equals(documentVo.getId()), "禁止将该文档本身设置为上级");
      DocumentEntity parent = new DocumentEntity();
      parent.setId(parentVo.getId());
      document.setParent(parent);
      Set<String> documentStack = new HashSet<>();
      documentStack.add(parentVo.getId());
      this.validateCircular(document, documentStack);
    } else {
      document.setParent(null);
    }

    //保存文档内容
    this.saveDucomentContent(document);
    //最后更新数据库
    this.documentRepository.saveAndFlush(document);
    this.saveEsDocument(document);
    return nebulaToolkitService.copyObjectByWhiteList(document, DocumentVo.class, HashSet.class, ArrayList.class);
  }

  /**
   *判断是否形成循环依赖
   * @param document
   * @param documentStack
   */
  private void validateCircular(DocumentEntity document, Set<String> documentStack) {
    List<DocumentEntity> children = documentRepository.findByParent(document.getId());
    if(CollectionUtils.isEmpty(children)) {
      return;
    }
    for (DocumentEntity child : children) {
      Validate.isTrue(!documentStack.contains(child.getId()), "形成循环依赖，更新失败，请检查！");
      documentStack.add(child.getId());
      this.validateCircular(child, documentStack);
    }
  }

  /**
   * 更新验证
   * @param documentVo
   */
  private void updateValidation(DocumentVo documentVo) {
    //验证文档
    Validate.notNull(documentVo, "文档对象必须传入，请检查!!");
    String id = documentVo.getId();
    Validate.notBlank(id, "文档id必须传入，请检查!!");
  }

  /**
   * 保存文档内容
   * @param document
   * @return
   */
  private void saveDucomentContent(DocumentEntity document) {
    if(document.getContent() == null) {
      return;
    }
    byte[] documentContent;
    String fileName = document.getFileName();
    String relativeLocal = document.getRelativeLocal();
    documentContent = document.getContent().getBytes(StandardCharsets.UTF_8);
    //查询数据库记录若有有文件信息，则修改文档内容
    if (StringUtils.isAllBlank(fileName, relativeLocal)) {
      //文件重命名
      fileName = UUID.randomUUID().toString() + "." + MD_FILE_SUFFIX;
      relativeLocal = this.randomRelativeLocal();
    }
    this.venusFileEventListener.onSaveFile(relativeLocal, fileName, fileName, documentContent);
    document.setRelativeLocal(relativeLocal);
    document.setFileName(fileName);
  }

  /**
   * 生成文件夹路径
   * @return
   */
  private String randomRelativeLocal() {
    String folderName = new SimpleDateFormat("yyyyMMdd").format(new Date());
    return StringUtils.join("/document/", folderName, "/", (new Random().nextInt(100) % 10));
  }


  /**
   * 查询树
   * @return
   */
  @Override
  public Set<DocumentVo> findTree() {
    Set<DocumentVo> documentSet = new LinkedHashSet<>();
    Set<DocumentEntity> rootDocuments = documentRepository.findByNullParent();
    if(CollectionUtils.isEmpty(rootDocuments)){
      return Sets.newHashSet();
    }
    for(DocumentEntity documentEntity : rootDocuments){
      DocumentVo documentVo = nebulaToolkitService.copyObjectByWhiteList(documentEntity, DocumentVo.class, LinkedHashSet.class, ArrayList.class, new String[] {"parent"});
      documentVo.setDocuments(this.findChildren(documentEntity));
      documentSet.add(documentVo);
    }
    return documentSet;
  }

  /**
   * 查询子级树
   * @param documentEntity
   * @return
   */
  private Set<DocumentVo> findChildren(DocumentEntity documentEntity) {
   Set<DocumentVo> documentVoSet = new LinkedHashSet<>();
    //  查询子级结构
    Set<DocumentEntity> children = documentEntity.getDocuments();
    if(CollectionUtils.isEmpty(children)){
      return Sets.newHashSet();
    }
    for(DocumentEntity child : children){
      DocumentVo documentVo = this.nebulaToolkitService.copyObjectByWhiteList(child, DocumentVo.class, HashSet.class, ArrayList.class,  new String[] {"parent"});
      documentVo.setDocuments(this.findChildren(child));
      documentVoSet.add(documentVo);
    }
    return documentVoSet;
  }

  /**
   * 删除文档，该文档下若有子文档则一并删除
   * @param id
   */
  @Transactional
  @Override
  public void deleteById(String id) {
    Validate.notBlank(id , "删除时文档唯一主键id不能为空，请检查!!");
    DocumentEntity document = this.documentRepository.findById(id).orElse(null);
    if (document != null){
      this.deleteDocument(document);
    }
  }

  /**
   * 删除当前节点开始的整个树
   * @param document
   */
  private void deleteDocument(DocumentEntity document) {
    Set<DocumentEntity> children = document.getDocuments();
    if(!CollectionUtils.isEmpty(children)){
      for(DocumentEntity documentEntity : children){
        this.deleteDocument(documentEntity);
      }
    }
    //没有儿子直接删除自己
    this.documentRepository.delete(document);
    documentEsRepository.delete(document);
    //删除对应的上传文件
    this.deleteFile(document);
  }

  /**
   * 删除子级对应的文件
   * @param document
   */
  private void deleteFile(DocumentEntity document) {
    String fileRename = document.getFileName();
    String relativeLocal = document.getRelativeLocal();
    if (!StringUtils.isAllBlank(fileRename, relativeLocal)) {
      this.venusFileEventListener.onDeleteFile(relativeLocal, fileRename, fileRename);
    }
  }

  /**
   * 读取文档内容
   * @param id
   * @return
   */
  @Override
  public DocumentVo findDetailsById(String id) {
    DocumentEntity document = documentRepository.findById(id).orElse(null);
    if (document == null){
      return null;
    }
    String fileRename = document.getFileName();
    String relativeLocal = document.getRelativeLocal();
    //文档内容读取
    if(!StringUtils.isEmpty(fileRename) && !StringUtils.isEmpty(relativeLocal)) {
      byte[] bytes = this.venusFileEventListener.onReadFileContent(relativeLocal, fileRename);
      String content = bytes == null ? "" : new String(bytes, StandardCharsets.UTF_8);
      document.setContent(content);
    }
    return nebulaToolkitService.copyObjectByWhiteList(document, DocumentVo.class, HashSet.class, ArrayList.class);
  }

  /**
   * 根据关键字查询文档，支持模糊
   * @return
   */
  @Override
  public List<DocumentVo> findByKeyword(String keyword) {
    if(StringUtils.isBlank(keyword)){
      return Lists.newArrayList();
    }
    List<DocumentEntity> documents = documentEsRepository.findByKeyword(keyword);
    if(CollectionUtils.isEmpty(documents)) {
      return Lists.newArrayList();
    }
    Collection<DocumentVo> documentVos = nebulaToolkitService.copyCollectionByWhiteList(documents, DocumentEntity.class, DocumentVo.class, HashSet.class, ArrayList.class);
    return Lists.newArrayList(documentVos);
  }

  @Override
  public byte[] export() {
    List<DocumentEntity> documents = documentRepository.findAll();
    if(CollectionUtils.isEmpty(documents)) {
      documents = new ArrayList<>();
    }
    Collection<DocumentEntity> collection = nebulaToolkitService.copyCollectionByWhiteList(documents, DocumentEntity.class,
        DocumentEntity.class, HashSet.class, ArrayList.class, "parent");
    documents = Lists.newArrayList(collection);
    try (ByteArrayOutputStream baos = new ByteArrayOutputStream();
         ZipOutputStream zos = new ZipOutputStream(baos)) {
      try (ByteArrayOutputStream objBaos = new ByteArrayOutputStream();
           ObjectOutputStream oos = new ObjectOutputStream(objBaos)) {
        oos.writeObject(documents);
        ZipFileUtils.writeZipFile(zos, ZIP_DOCUMENT_FILE_NAME, objBaos.toByteArray());
      }
      Map<String, DocumentEntity> map = new HashMap<>();
      for (DocumentEntity document : documents) {
        String relativeLocal = document.getRelativeLocal();
        String fileName = document.getFileName();
        if(StringUtils.isAnyBlank(relativeLocal, fileName)) {
          continue;
        }
        byte[] fileContent = this.venusFileEventListener.onReadFileContent(relativeLocal, fileName);
        ZipFileUtils.writeZipFile(zos, relativeLocal, fileName, fileContent);
        //图片  ![image.png](http://192.168.2.9:6500/v1/venus/images/files/20200727/4/2c4a1e16-8151-4f03-a276-89014aef8991.png)
        String regex = "(?<=images).*?(?=\\))";
        Pattern p = Pattern.compile(regex);
        Matcher m = p.matcher(new String(fileContent, StandardCharsets.UTF_8));
        while(m.find()) {
          //  str = /files/20200727/4/2c4a1e16-8151-4f03-a276-89014aef8991.png
          String str = m.group();
          if(map.containsKey(str)){
            continue;
          }
          map.put(str, document);
          int index = str.lastIndexOf("/");
          String imageRelativeLocal = str.substring(0, index);
          String imageFileName = str.substring(index + 1);
          String[] arr = imageFileName.split("\\.");
          byte[] imageBytes = imageViewService.imageQuery(imageRelativeLocal , arr[0], arr[1], null);
          ZipFileUtils.writeZipFile(zos, imageRelativeLocal, imageFileName, imageBytes);
        }
      }
      zos.finish();
      return baos.toByteArray();
    } catch (IOException e) {
      LOGGER.error(e.getMessage(), e);
      throw new IllegalArgumentException(e.getMessage(), e);
    }
  }

  @Override
  @Transactional
  @SuppressWarnings("unchecked")
  public void importDoc(MultipartFile file) {
    Validate.notNull(file, "导入的文件为空");
    File tmpFile;
    try {
      tmpFile = FileUtils.writeLocalFile(file.getBytes(), venusProperties.getFileRoot(), Constants.ZIP_FILE_SUBFIX);
    } catch (IOException e) {
      LOGGER.error(e.getMessage(), e);
      throw new IllegalArgumentException(String.format("访问本地磁盘出错：%s", e.getMessage()));
    }
    try {
      try (ZipFile zipFile = new ZipFile(tmpFile);) {
        ZipEntry entry = zipFile.getEntry(ZIP_DOCUMENT_FILE_NAME);
        Validate.notNull(entry, "未找到文档文件，请上传正确的文件！！");
        try (InputStream is = zipFile.getInputStream(entry);
             ObjectInputStream ois  = new ObjectInputStream(is)) {
          List<DocumentEntity> documents = (List<DocumentEntity>) ois.readObject();
          if(CollectionUtils.isEmpty(documents)) {
            return;
          }
          List<DocumentEntity> tree = this.convert2Tree(documents);
          this.importDocs(zipFile, tree, null);
        } catch (ClassNotFoundException e) {
          throw new IllegalArgumentException("导入失败，未找到相关类", e);
        }
      }
    } catch (IOException e) {
      LOGGER.error(e.getMessage(), e);
      throw new IllegalArgumentException("读取文件失败，请重新上传！");
    } finally {
      tmpFile.delete();
    }
  }

  /**
   * 导入文档数据,documents是树结构的数据
   * @param zipFile
   * @param documents
   * @param parent
   */
  private void importDocs(ZipFile zipFile, List<DocumentEntity> documents, DocumentEntity parent) {
    if(CollectionUtils.isEmpty(documents)) {
      return;
    }
    for (DocumentEntity document : documents) {
      DocumentEntity dbDocument = this.findByTitleAndParent(document.getTitle(), parent);
      if(dbDocument == null) {
        document.setId(null);
        document.setParent(parent);
        this.createValidation(document);
        documentRepository.save(document);
        this.importDocContent(zipFile, document);
        this.saveEsDocument(document);
        dbDocument = document;
      }
      if(!CollectionUtils.isEmpty(document.getDocuments())) {
        List<DocumentEntity> children = document.getDocuments().stream().collect(Collectors.toList());
        this.importDocs(zipFile, children, dbDocument);
      }
    }
  }

  /**
   * 导入文档内容
   * @param zipFile
   * @param document
   */
  private void importDocContent(ZipFile zipFile, DocumentEntity document) {
    String relativeLocal = document.getRelativeLocal();
    String fileName = document.getFileName();
    if(!StringUtils.isAnyBlank(relativeLocal, fileName)) {
      try {
        byte[] bytes = ZipFileUtils.readZipFile(zipFile, relativeLocal, fileName);
        //处理图片
        byte[] newBytes = this.importImage(zipFile, bytes);
        this.venusFileEventListener.onSaveFile(relativeLocal, fileName, fileName, newBytes);
        document.setContent(new String(bytes, StandardCharsets.UTF_8));
      } catch (IOException e) {
        LOGGER.error(e.getMessage(), e);
        throw new IllegalArgumentException(String.format("读取文档内容失败：%s", document.getTitle()), e);
      }
    }
  }

  /**
   * 处理图片
   * @param zipFile
   * @param bytes
   */
  private byte[] importImage(ZipFile zipFile, byte[] bytes) {
    String regex = "(?<=images).*?(?=\\))";
    Pattern p = Pattern.compile(regex);
    Matcher m = p.matcher(new String(bytes, StandardCharsets.UTF_8));
    String content = new String(bytes, StandardCharsets.UTF_8);
    while (m.find()) {
      //  str = /files/20200727/4/2c4a1e16-8151-4f03-a276-89014aef8991.png
      String str = m.group();
      int index = str.lastIndexOf("/");
      String imageRelativeLocal = str.substring(0, index);
      String imageFileName = str.substring(index + 1);
      String[] arr = imageFileName.split("\\.");
      ZipEntry image = ZipFileUtils.getZipEntry(zipFile, imageRelativeLocal, imageFileName);
      InputStream inputStream = null;
      try {
        inputStream = zipFile.getInputStream(image);
        byte[] imageBytes = IOUtils.toByteArray(inputStream);
        String folderName = new SimpleDateFormat("yyyyMMdd").format(new Date());
        String uuid = UUID.randomUUID().toString();
        String fileRename = StringUtils.join(uuid + "." + arr[1]);
        String relativePath = StringUtils.join("/files/", folderName, "/", (new Random().nextInt(100) % 10));
        this.venusFileEventListener.onSaveFile(relativePath, fileRename, fileRename, imageBytes);
        content = content.replace(str, relativePath + "/" + fileRename);
      } catch (IOException e) {
        e.printStackTrace();
      }
    }
    return content.getBytes();
  }

  /**
   * 验证标题是否存在
   * @param title
   * @param parent
   * @return
   */
  private DocumentEntity findByTitleAndParent(String title, DocumentEntity parent) {
    DocumentEntity document;
    if(parent == null) {
      document = documentRepository.findByNullParentAndTitle(title);
    } else {
      document = documentRepository.findByParentAndTitle(title, parent.getId());
    }
    return document;
  }

  /**
   * 将文档list数据转换成树结构
   * @param documents
   * @return
   */
  private List<DocumentEntity> convert2Tree(List<DocumentEntity> documents) {
    Map<String, DocumentEntity> documentMap = documents.stream().collect(Collectors.toMap(DocumentEntity::getId, d -> d));
    Map<String, DocumentEntity> rootMap = new HashMap<>();
    for (DocumentEntity document : documents) {
      DocumentEntity parent = document.getParent();
      if(parent == null) {
        rootMap.put(document.getId(), document);
        continue;
      }
      DocumentEntity parentDoc = documentMap.get(parent.getId());
      if(parentDoc == null) {
        rootMap.put(document.getId(), document);
        continue;
      }
      Set<DocumentEntity> children = parentDoc.getDocuments() == null ? Sets.newHashSet() : parentDoc.getDocuments();
      children.add(document);
      parentDoc.setDocuments(children);
    }
    return Lists.newArrayList(rootMap.values());
  }

  /**
   * 保存数据到elasticsearch
   * @param document
   */
  private void saveEsDocument(DocumentEntity document) {
    if(StringUtils.isBlank(document.getContent())
        && StringUtils.isNotBlank(document.getRelativeLocal())
        && StringUtils.isNotBlank(document.getFileName())) {
      byte[] bytes = this.venusFileEventListener.onReadFileContent(document.getRelativeLocal(), document.getFileName());
      document.setContent(new String(bytes, StandardCharsets.UTF_8));
    }
    DocumentEntity documentEs = nebulaToolkitService.copyObjectByWhiteList(document, DocumentEntity.class, null, null);
    documentEsRepository.save(documentEs);
  }
  
  private String findAccount() {
    SecurityContext securityContext;
    Authentication authentication;
    if((securityContext = SecurityContextHolder.getContext()) == null) {
      return "admin";
    }
    if((authentication =  securityContext.getAuthentication()) == null) {
      return "admin";
    }
    String account = authentication.getName();
    return account;
  }
}