package com.bizunited.nebula.venus.service.local.service.internal;

import com.bizunited.nebula.common.service.NebulaToolkitService;
import com.bizunited.nebula.common.util.tenant.TenantUtils;
import com.bizunited.nebula.venus.sdk.service.file.FileRelativeTemplate;
import com.bizunited.nebula.venus.sdk.config.SimpleVenusProperties;
import com.bizunited.nebula.venus.sdk.event.VenusFileEventListener;
import com.bizunited.nebula.venus.sdk.service.file.FileHandleService;
import com.bizunited.nebula.venus.sdk.vo.OrdinaryFileVo;
import com.bizunited.nebula.venus.service.local.entity.OrdinaryFileEntity;
import com.bizunited.nebula.venus.service.local.repository.OrdinaryFileRepository;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.util.Base64Utils;
import org.springframework.web.multipart.MultipartFile;

import javax.transaction.Transactional;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Calendar;
import java.util.Collection;
import java.util.Collections;
import java.util.Date;
import java.util.LinkedHashSet;
import java.util.LinkedList;
import java.util.List;
import java.util.Optional;
import java.util.UUID;

/**
 * 文件上传服务
 * @author yinwenjie
 */
public class FileHandleServiceImpl implements FileHandleService {
  /**
   * 打印日志.
   */
  private static final Logger LOGGER = LoggerFactory.getLogger(FileHandleServiceImpl.class);
  @Autowired
  private SimpleVenusProperties venusProperties;
  @Autowired
  private OrdinaryFileRepository ordinaryFileRepository;
  @Autowired
  private FileRelativeTemplate fileRelativeTemplate;
  @Autowired
  private VenusFileEventListener venusFileEventListener;
  @Autowired
  private NebulaToolkitService nebulaToolkitService;
  
  @Override
  @Transactional
  public List<OrdinaryFileVo> fileUpload(String subSystem, String creator, Integer effective, String[] fileNanmes, String[] base64Contents) {
    Validate.notBlank(creator , "文件的提交人必须传入，请检查!!");
    if(base64Contents == null || base64Contents.length == 0) {
      throw new IllegalArgumentException("错误的base64编码文件内容!!");
    }
    if(fileNanmes == null || fileNanmes.length == 0) {
      throw new IllegalArgumentException("错误的文件名数量信息!!");
    }
    Validate.isTrue(fileNanmes.length == base64Contents.length , "文件内容数量和文件名数量不匹配，请检查!!");
    
    // 一个文件一个文件的保存
    List<OrdinaryFileVo> ordinaryFileVos = new LinkedList<>();
    for(int index = 0 ; index < fileNanmes.length ; index++) {
      String originalFilename = fileNanmes[index];
      this.validatePrefix(originalFilename);
      String base64Content = base64Contents[index];
      byte[] fileContext = Base64Utils.decodeFromString(base64Content);
      OrdinaryFileVo result = this.fileUpload(subSystem, creator, effective, originalFilename, fileContext);
      ordinaryFileVos.add(result);
    }
    return ordinaryFileVos;
  }

  @Override
  @Transactional
  public List<OrdinaryFileVo> fileUpload(String subSystem , String creator , Integer effective , MultipartFile[] files)  {
    Validate.notNull(files != null && files.length > 0 , "上传的文件信息至少有一个，请检查!!");
    Validate.notBlank(creator , "附件的提交人必须传入，请检查!!");
    /*
     * 1、开始保存文件，注意，文件都要重命名。 为了简单起见重命名使用java自带的UUID工具完成即可
     * 2、正式写入文件，如果以上所有步骤都成功，则向上传者返回文件存储的提示信息
     * 注意，这些文件在上传时，由于不清楚这些图片在显示时会使用什么样的特效 所以在文件上传时就进行redis缓存存储是没有意义的。
     * 
     * 最后，本工程没有提供上传的测试页面，测试是使用postman等软件完成的
     */
    List<OrdinaryFileVo> ordinaryFileVos = new LinkedList<>();
    for (MultipartFile file : files) {
      // 文件内容
      byte[] fileContext = null;
      try {
        fileContext = file.getBytes(); 
      } catch(Exception e) {
        LOGGER.error(e.getMessage() , e);
        throw new IllegalArgumentException(e);
      }
      String originalFilename = file.getOriginalFilename();
      this.validatePrefix(originalFilename);
      OrdinaryFileVo result = this.fileUpload(subSystem, creator, effective, originalFilename, fileContext);
      ordinaryFileVos.add(result);
    }
    return ordinaryFileVos;
  }
  
  @Override
  @Transactional
  public OrdinaryFileVo fileUpload(String subsystem, String creator, Integer effective, String fileNanme, byte[] contents) {
    Validate.notBlank(creator , "文件的提交人必须传入，请检查!!");
    if(contents == null || contents.length == 0) {
      throw new IllegalArgumentException("错误的base64编码文件内容!!");
    }
    Validate.notBlank(fileNanme , "错误的文件名数量信息!!");
    // 这里就是保存单个文件
    this.validatePrefix(fileNanme);
    int prefixIndex = fileNanme.lastIndexOf('.');
    String prefix = null;
    if (prefixIndex != -1) {
      prefix = fileNanme.substring(prefixIndex + 1);
      prefix = prefix.toLowerCase();
    }
    // 可以使用日期作为文件夹的名字
    String renameImage = UUID.randomUUID().toString();
    String relativePath = fileRelativeTemplate.generatRelativePath(subsystem, fileNanme);
    String fileRename = StringUtils.join(renameImage , "." , prefix);
    
    OrdinaryFileEntity result = this.fileUpload(relativePath, fileNanme, fileRename, prefix, effective, creator, contents);
    ordinaryFileRepository.save(result);
    OrdinaryFileVo resultVo = nebulaToolkitService.copyObjectByWhiteList(result, OrdinaryFileVo.class, LinkedHashSet.class, ArrayList.class);
    return resultVo;
  }

  @Override
  public OrdinaryFileVo fileUpload(String subSystem, String creator, Integer effective, String fileRename, String fileNanme, byte[] contents) {
    Validate.notBlank(creator , "文件的提交人必须传入，请检查!!");
    if(contents == null || contents.length == 0) {
      throw new IllegalArgumentException("错误的base64编码文件内容!!");
    }
    Validate.notBlank(fileNanme , "错误的文件名数量信息!!");
    // 这里就是保存单个文件
    this.validatePrefix(fileNanme);
    int prefixIndex = fileNanme.lastIndexOf('.');
    String prefix = null;
    if (prefixIndex != -1) {
      prefix = fileNanme.substring(prefixIndex + 1);
      prefix = prefix.toLowerCase();
    }
    String relativePath = fileRelativeTemplate.generatRelativePath(subSystem, fileNanme);
    OrdinaryFileEntity result = this.fileUpload(relativePath, fileNanme, fileRename, prefix, effective, creator, contents);
    ordinaryFileRepository.save(result);
    OrdinaryFileVo resultVo = nebulaToolkitService.copyObjectByWhiteList(result, OrdinaryFileVo.class, LinkedHashSet.class, ArrayList.class);
    return resultVo;
  }
  
  /**
   * 验证当前上环文件的后缀信息
   * @param originalFilename
   */
  private void validatePrefix(String originalFilename) {
    int prefixIndex = originalFilename.lastIndexOf('.');
    Validate.isTrue(prefixIndex != -1 , "当前上传的文件（%s）必须携带文件后缀，请检查!!" , originalFilename);
    String prefix = originalFilename.substring(prefixIndex + 1);
    prefix = prefix.toLowerCase();
    
    // 只有位图文件和在扩展名白名单设定中的文件，才能进行上传管理
    if (StringUtils.isBlank(prefix) || (!StringUtils.equals(prefix, "png") 
        && !StringUtils.equals(prefix, "gif") && !StringUtils.equals(prefix, "jpg") 
        && !StringUtils.equals(prefix, "jpeg") && !StringUtils.equals(prefix, "bmp") 
        && !StringUtils.equalsAnyIgnoreCase(prefix, venusProperties.getWhitePrefixs()))) {
      throw new IllegalArgumentException("只有位图文件(bmp/jpg/png/gif/white prefixs)和在扩展名白名单设定中的文件，才能进行上传管理!");
    }
  }

  /**
   * 该私有方法保存文件内容
   * @param relativePath 指定的相对路径
   * @param originalFilename 原始文件名，包括扩展名
   * @param fileRename 重命名后的文件名，包括扩展名
   * @param effective 指定的文件有效期时长，单位“天”
   * @param fileContext 文件内容
   * @return 文件的对象描述
   */
  private OrdinaryFileEntity fileUpload(String relativePath , String originalFilename, String fileRename , String prefix ,Integer effective , String creator , byte[] fileContext) {
    this.venusFileEventListener.onSaveFile(relativePath , originalFilename, fileRename , fileContext);
    
    // 计算有效期
    SimpleDateFormat simpleDateFormat = new SimpleDateFormat("yyyy-MM-dd HH:mm:ss");
    Date effectiveDate = null;
    try {
      effectiveDate = simpleDateFormat.parse("3999-01-01 00:00:00");
    } catch (ParseException e) {
      LOGGER.error(e.getMessage() , e);
      throw new IllegalArgumentException(e.getMessage());
    }
    // 如果条件成立，则需要按照当前时间从新计算时间
    if(effective != null) {
      Validate.isTrue(effective > 0 , "错误的有效期时长（单位“天”）, 请检查!!");
      Calendar calendar = Calendar.getInstance();
      calendar.add(Calendar.DATE, effective);
      effectiveDate = calendar.getTime();
    }
    // 获取文件大小
    long fileSize = fileContext.length;
    // 构造信息保存数据库，并进行返回
    OrdinaryFileEntity result = new OrdinaryFileEntity();
    Date now = new Date();
    result.setCommitUser(creator);
    result.setCreateAccount(creator);
    result.setCreateTime(now);
    result.setEffectiveDate(effectiveDate);
    result.setFileName(fileRename);
    result.setModifyAccount(creator);
    result.setModifyTime(now);
    result.setOriginalFileName(originalFilename);
    result.setPrefix(prefix);
    result.setRelativeLocal(relativePath);
    result.setTenantCode(TenantUtils.getTenantCode());
    result.setFileSize(fileSize);
    return result;
  }
  
  @Override
  public List<OrdinaryFileVo> findByEffectiveDate(Date currentDate) {
    if(currentDate == null) {
      return Collections.emptyList();
    }
    List<OrdinaryFileEntity> files = this.ordinaryFileRepository.findByEffectiveDateAndTenantCode(currentDate, TenantUtils.getTenantCode());
    Collection<OrdinaryFileVo> collection = nebulaToolkitService.copyCollectionByWhiteList(files, OrdinaryFileEntity.class, OrdinaryFileVo.class, LinkedHashSet.class,ArrayList.class);
    return new ArrayList<>(collection);
  }

  @Override
  public OrdinaryFileVo findByFileNameAndRelativeLocal(String fileName, String relativeLocal) {
    if(relativeLocal == null || StringUtils.isBlank(fileName)) {
      return null;
    }
    OrdinaryFileEntity ordinaryFile = ordinaryFileRepository.findByFileNameAndRelativeLocalAndTenantCode(fileName, relativeLocal, TenantUtils.getTenantCode());
    if(ordinaryFile == null) {
      return null;
    }
    return nebulaToolkitService.copyObjectByWhiteList(ordinaryFile, OrdinaryFileVo.class, LinkedHashSet.class, ArrayList.class);
  }

  @Override
  public byte[] findContentByFilePathAndFileRename(String relativePath, String fileRename) {
    return this.venusFileEventListener.onReadFileContent(relativePath, fileRename);
  }

  @Override
  @Transactional
  public void updateByEffective(String[] fileReNames) {
    if(fileReNames == null || fileReNames.length == 0) {
      return;
    }
    SimpleDateFormat simpleDateFormat = new SimpleDateFormat("yyyy-MM-dd HH:mm:ss");
    Date effectiveDate = null;
    try {
      effectiveDate = simpleDateFormat.parse("3999-01-01 00:00:00");
    } catch (ParseException e) {
      LOGGER.error(e.getMessage() , e);
      throw new IllegalArgumentException(e.getMessage());
    }
    
    // 开始一个一个更新
    for (int index = 0; index < fileReNames.length; index++) {
      String fileReNameItem = fileReNames[index];
      int targetIndex = fileReNameItem.lastIndexOf('/');
      String fileReName = fileReNameItem.substring(targetIndex + 1);
      String relativeLocal = fileReNameItem.substring(0, targetIndex);
      OrdinaryFileEntity currentOrdinaryFile = this.ordinaryFileRepository.findByFileNameAndRelativeLocalAndTenantCode(fileReName, relativeLocal, TenantUtils.getTenantCode());
      if(currentOrdinaryFile == null) {
        continue;
      }
      
      // 文件有效时间更新为3999-01-01 00:00:00
      currentOrdinaryFile.setEffectiveDate(effectiveDate);
      currentOrdinaryFile.setTenantCode(TenantUtils.getTenantCode());
      this.ordinaryFileRepository.save(currentOrdinaryFile);
    }
  }

  @Override
  @Transactional
  public void deleteFiles(String[] fileIds) {
    Validate.notNull(fileIds , "至少传入一个文件的数据库编号信息!!");
    Validate.isTrue(fileIds.length > 0 , "至少传入一个文件的数据库编号信息!!");
    for (int index = 0; index < fileIds.length; index++) {
      String fileId = fileIds[index];
      Optional<OrdinaryFileEntity> op = this.ordinaryFileRepository.findById(fileId);
      OrdinaryFileEntity currentOrdinaryFile = op.orElse(null);
      if(currentOrdinaryFile == null) {
        continue;
      }
      // 首先删除磁盘文件
      this.venusFileEventListener.onDeleteFile(currentOrdinaryFile.getRelativeLocal(), currentOrdinaryFile.getOriginalFileName(), currentOrdinaryFile.getFileName());
      // 然后删除数据库信息
      this.ordinaryFileRepository.deleteById(fileId);
    }
  }

  @Override
  @Transactional
  public void deleteFile(String filePath, String fileName) {
    Validate.notNull(filePath, "文件路径不能为空");
    Validate.notBlank(fileName, "文件名称不能为空");
    OrdinaryFileEntity file = ordinaryFileRepository.findByFileNameAndRelativeLocalAndTenantCode(fileName, StringUtils.trim(filePath), TenantUtils.getTenantCode());
    Validate.notNull(file, "删除的文件不存在");
    // 首先删除磁盘文件
    this.venusFileEventListener.onDeleteFile(file.getRelativeLocal(), file.getOriginalFileName(), file.getFileName());
    // 然后删除数据库信息
    this.ordinaryFileRepository.delete(file);
  }
}
