package com.bizunited.nebula.venus.service.local.service.notifier;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.util.Set;

import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;

import com.bizunited.nebula.venus.sdk.config.SimpleVenusProperties;
import com.bizunited.nebula.venus.sdk.event.VenusFileEventListener;
import com.google.common.collect.Sets;

/**
 * 技术中台，文件服务接口（单纯的文件服务接口）。在不同的业务系统中，涉及的文件管理方式是不一样的。
 * 例如，文件资料可能被保存到本地文件系统，并从本地文件系统中进行下载；业务系统中的文件资料也可能被保存到诸如阿里云对象存储系统中（使用阿里OSS API）；
 * 又或者被保存到Hadoop HDFS上。所以当前还没有完全脱离代码编写工作的当前技术中台版本，必须定义一个文件服务接口，以便技术中台能够提供一个默认的文件保存方式，
 * 以及不同的业务系统根据自己的文件资料保存要求，实现具体的文件资料保存逻辑<p>
 * 
 * 这个监听就是文件截止实质操作的默认实现</p>
 * 
 * 这里说明几点：<br>
 * 1、这个文件资料的保存、读取服务将被广泛用于技术中台服务中的各个服务，包括但不限于：模板草稿的保存和读取、模板样式的保存和读取、表单实例上附件的上传保存和读取、
 * 表单实例上图片的上传保存和读取，等等。<br>
 * 2、技术中台针对该文件服务接口，有一个默认的实现，既是VenusFileServiceSimpleImpl，如果业务系统搭建时，没有特别实现kuiperFileService，
 * 那么VenusFileServiceSimpleImpl的默认实现将其作用，文件资料将被保存在以venus.file.fileRoot配置的根目录下（默认实现）。
 * @author yinwenjie
 */
public class VenusFileEventListenerImpl implements VenusFileEventListener {
  @Autowired
  private SimpleVenusProperties venusProperties;
  private static final Logger LOGGER = LoggerFactory.getLogger(VenusFileEventListenerImpl.class);
  
  @Override
  public void onSaveFile(String relativePath, String fileName, String fileRename, byte[] fileContext) {
    /*
     * 文件保存过程为：
     * 1、验证文件内容的基本面
     * 2、默认取得根路径和可能存在的相对路径，构成文件存储的完成目录路径，并在没有该目录的情况下，建立目录
     * 3、然后进行文件存储，使用重命名后的文件名进行
     * */
    // 1、=====验证文件内容的基本面
    long fileSize = fileContext.length;
    LOGGER.debug("====fiel Upload====originalFilename:{},fileSize:{}" , fileName , fileSize);
    // 如果条件成立，说明大于最大上传大小了
    Validate.isTrue(fileSize < venusProperties.getMaxFileSize() * 1024 * 1024 , "image file should be less than %dMB!" , venusProperties.getMaxFileSize());
    if(!StringUtils.isBlank(relativePath)) {
      Validate.matchesPattern(relativePath, "/{1}[a-z|A-Z|0-9|_|/|\\.]+", "relativePath must begin with chartset '/' or relativePath noncompliance !!");
    }
    // 文件后缀，必须存在后缀，其中txt后缀的文件都是允许的
    int nodeIndex = StringUtils.lastIndexOf(fileName, ".");
    String filePrefix = StringUtils.substring(fileName, nodeIndex + 1, fileName.length());
    Set<String> whitePrefixSet = Sets.newHashSet(venusProperties.getWhitePrefixs());
    whitePrefixSet.add("txt");
    Validate.isTrue(whitePrefixSet.stream().anyMatch(item -> StringUtils.equalsAnyIgnoreCase(item, filePrefix)) , "不支持的文件格式：%s" , fileName);
    
    // 2、=====确定使用的文件夹，没有即创建
    String absoluteDirPath = StringUtils.join(venusProperties.getFileRoot() , relativePath);
    File absoluteDirPathFile = new File(absoluteDirPath);
    if(!absoluteDirPathFile.exists()) {
      Validate.isTrue(absoluteDirPathFile.mkdirs() , "无法创建指定的文件夹，因为没有操作权限[%s]，请检查文件存储的操作目录的权限!" , absoluteDirPath);
    }
    
    // 3、=====开始保存文件
    String fullPath = StringUtils.join(absoluteDirPath , "/" , fileRename);
    try (FileOutputStream fileOut = new FileOutputStream(new File(fullPath));
        ByteArrayInputStream byteIn = new ByteArrayInputStream(fileContext);) {
      int maxLen = 204800;
      byte[] readContext = new byte[maxLen];
      int realen;
      while((realen = byteIn.read(readContext, 0, maxLen)) != -1) {
        fileOut.write(readContext, 0, realen);
      }
    } catch (IllegalStateException | IOException e) {
      throw new IllegalArgumentException(e.getMessage());
    }
  }
  
  @Override
  public void onDeleteFile(String relativePath, String fileName, String fileRename) {
    if(!StringUtils.isBlank(relativePath)) {
      Validate.matchesPattern(relativePath, "/{1}[a-z|A-Z|0-9|_|/|\\.]+", "relativePath must begin with chartset '/' or relativePath noncompliance !!");
    }
    String absoluteDirPath = StringUtils.join(venusProperties.getFileRoot() , relativePath);
    String fullPath = StringUtils.join(absoluteDirPath , "/" , fileRename);
    File fullPathFile = new File(fullPath);
    
    // 如果存在则进行删除
    if(fullPathFile.exists()) {
      Validate.isTrue(fullPathFile.delete() , "无法删除文件，因为没有操作权限[%s]，请检查文件存储的操作目录的权限",fullPath);
    }
  }
  
  @Override
  public byte[] onReadFileContent(String relativePath, String fileRename) {
    /*
     * 读取文件内容的过程包括：
     * 1、检查文件路径、入参的基本面
     * 2、如果指定的完整路径下，文件存在，则进行读取，否则返回null
     * */
    String absoluteDirPath = StringUtils.join(venusProperties.getFileRoot() , relativePath);
    String fullPath = StringUtils.join(absoluteDirPath , "/" , fileRename);
    File fullPathFile = new File(fullPath);
    if (!fullPathFile.exists()) {
      return new byte[0];
    }
    // 如果不是文件，也不能/不需要读取文件内容了
    if(!fullPathFile.isFile()) {
      return new byte[0];
    }

    byte[] fileContents = null;
    try (FileInputStream ois = new FileInputStream(fullPathFile);
        ByteArrayOutputStream out = new ByteArrayOutputStream()) {
      int maxLen = 204800;
      byte[] readContext = new byte[maxLen];
      int realen;
      while((realen = ois.read(readContext, 0, maxLen)) != -1) {
        out.write(readContext, 0, realen);
      }
      fileContents = out.toByteArray();
    } catch (Exception e) {
      throw new IllegalArgumentException(e);
    }
    return fileContents;
  }
}
