package com.bizunited.nebula.venus.service.local.controller;

import com.bizunited.nebula.common.controller.BaseController;
import com.bizunited.nebula.common.controller.model.ResponseModel;
import com.bizunited.nebula.venus.sdk.service.file.FileHandleService;
import com.bizunited.nebula.venus.sdk.vo.OrdinaryFileVo;
import io.swagger.annotations.ApiOperation;
import io.swagger.annotations.ApiParam;

import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.format.annotation.DateTimeFormat;
import org.springframework.security.core.Authentication;
import org.springframework.security.core.context.SecurityContext;
import org.springframework.security.core.context.SecurityContextHolder;
import org.springframework.web.bind.annotation.DeleteMapping;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.RestController;
import org.springframework.web.multipart.MultipartFile;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import java.io.IOException;
import java.util.Date;
import java.util.List;

/**
 * 这个controller用于文件附件的上传，支持由白名单放行的一般附件，以及位图图片文件（jpeg、jpg、gif、png、bmp）
 * @author yinwenjie
 */
@RestController
@RequestMapping("/v1/venus/files")
public class FileController extends BaseController {

  private static final Logger LOGGER = LoggerFactory.getLogger(FileController.class);

  @Autowired
  private FileHandleService fileHandleService;
  
  private static final String UPLOAD_ERROR = "you must upload least one file !";

  /**
   * 这个方法用于上传若干张附件文件，且有子系统区别时使用
   * @param subsystem
   * @param effective
   * @param files
   * @return
   */
  @ApiOperation(value = "这个方法可用于一次上传多个文件附件（支持文件附件上传）", notes = ""
      + "1、目前支持普通文件，普通文件通过配置文件进行后缀名的白名单控制"
      + "2、只能上传大小默认不超过2048KB的文件（可设置），也就是说如果上传的文件较大，那么客户端需要自行压缩一下<br>"
      + "3、上传成功后，这个请求将返回文件在服务器端保存的唯一id还包括了文件对象的其它信息，如果是图片文件，那么在文件访问时还可以加相应特效哦，具体请参见/v1/kuiper/images下的controller api说明信息）")
  @RequestMapping(path = "/fileUpload/{subsystem}", method = RequestMethod.POST)
  public ResponseModel fileUpload(@PathVariable("subsystem") String subsystem,
                                  @RequestParam(name="effective" , required=false) @ApiParam(required=false , name="effective" , value="设定的文件有效期时长，时长单位为“天”。可以不传入，不传入表示永久有效") Integer effective ,
                                  @RequestParam(name="file" , required=true) @ApiParam(required=true , name="file" , value="上传的文件对象，提交的表单信息中，请命名为files（支持多个文件，建议不超过5个文件）") MultipartFile[] files) {
    if (files == null || files.length == 0) {
      throw new IllegalArgumentException(UPLOAD_ERROR);
    }
    // 依次处理每个文件
    try {
      List<OrdinaryFileVo> uploadPojos = this.fileHandleService.fileUpload(subsystem, this.findCurrentAccount(), effective, files);
      return this.buildHttpResult(uploadPojos);
    } catch(Exception e) {
      return this.buildHttpResultForException(e);
    }
  }
  
  /**
   * 文件上传服务，这个文件不一定是图片，只要满足配置信息中规定后缀的文件都可以上传，注意这些文件的有效期为永久。
   * 如果要更改文件有效时间，则需要指定effective参数，该参数是指定文件有效时长，单位为“天”</p>
   * 并且上传的文件内容为base64编码
   * @param subsystem 指代进行文件上传的子系统信息，子系统将单独生成一个文件夹。
   * @param effective 这些文件的有效时长，单位为“天”。如果没有传入则默认这些文件有效期为永久
   * @param fileNanmes 文件上传时的文件名
   * @param base64Contents 文件内容的base64编码描述
   * @return 
   * @throws IllegalArgumentException
   */
  @ApiOperation(value = "这个方法可用于一次上传多个文件附件（支持文件附件上传）", notes = ""
      + "1、目前支持普通文件，普通文件通过配置文件进行后缀名的白名单控制"
      + "2、只能上传大小默认不超过2048KB的文件（可设置），也就是说如果上传的文件较大，那么客户端需要自行压缩一下<br>"
      + "3、上传成功后，这个请求将返回文件在服务器端保存的唯一id还包括了文件对象的其它信息，如果是图片文件，那么在文件访问时还可以加相应特效哦，具体请参见/v1/kuiper/images下的controller api说明信息）")
  @RequestMapping(path = "/fileUploadBase64/{subsystem}", method = RequestMethod.POST)
  public ResponseModel fileUpload(@PathVariable("subsystem") String subsystem , 
                                  @RequestParam(name="effective" , required=false) @ApiParam(required=false , name="effective" , value="设定的文件有效期时长，时长单位为“天”。可以不传入，不传入表示永久有效") Integer effective , 
                                  @RequestParam(name="fileNanmes" , required=true) @ApiParam(required=true , name="fileNanmes" , value="带有文件后缀，但是不带有文件路径的的文件名(一个或者多个)") String[] fileNanmes , 
                                  @RequestParam(name="base64Contents" , required=true) @ApiParam(required=true , name="base64Contents" , value="用base64编码的文件内容（一个或者多个）") String[] base64Contents) {
    try {
      List<OrdinaryFileVo> imageUploadPojos = this.fileHandleService.fileUpload(subsystem, this.findCurrentAccount(), effective, fileNanmes, base64Contents);
      return this.buildHttpResult(imageUploadPojos);
    } catch(Exception e) {
      return this.buildHttpResultForException(e);
    }
  }
  
  @ApiOperation(value = "将指定文件的有效期设置为“永久”，实际上有效期截止时间将设置为3999-01-01 00:00:00", notes = "文件信息是文件存储的相对路径 + 重命名后的文件名。例如/file/20190402/3/e18ab4f4-9fad-46c7-9754-fdfc3f08c488.jpg")
  @RequestMapping(path = "/updatePermanentEffective", method = RequestMethod.PATCH)
  public ResponseModel updatePermanentEffective(@RequestParam(name="fileNanmes" , required=true) @ApiParam(required=true , name="fileNanmes" , value="带有文件后缀，且带有文件相对路径的的文件名(一个或者多个)") String[] fileNanmes) {
    try {
      this.fileHandleService.updateByEffective(fileNanmes);
      return this.buildHttpResult();
    } catch(Exception e) {
      return this.buildHttpResultForException(e);
    }
  }
  
  /**
   * 按照给定的文件路径和文件名，下载一般性文件。
   * @param response
   * @param relativeLocal
   * @param fileName
   * @throws IOException
   */
  @ApiOperation(value = "按照给定的文件路径和文件名，下载一般性文件。")
  @RequestMapping(value = "/download", method = RequestMethod.GET)
  public void download(HttpServletRequest request, HttpServletResponse response ,
                       @ApiParam(name = "relativeLocal", value = "相对路径", required = true) @RequestParam("relativeLocal") String relativeLocal,
                       @ApiParam(name = "fileName", value = "文件名（重命名后的）", required = true) @RequestParam("fileName") String fileName) throws IOException {
    byte[] bytes = fileHandleService.findContentByFilePathAndFileRename(relativeLocal, fileName); 
    if (bytes == null) { 
      return;
    }
    OrdinaryFileVo ordinaryFile = this.fileHandleService.findByFileNameAndRelativeLocal(fileName, relativeLocal);
    if(ordinaryFile == null) {
      return;
    }
    // 附件文件下载
    this.writeResponseFile(request, response, bytes, ordinaryFile.getOriginalFileName());
  }
  
  /**
   * 按照给定的文件路径和文件名，下载一般性文件。
   */
  @ApiOperation(value = "按照给定的文件路径和文件名，下载一般性文件。")
  @RequestMapping(path = {"/download/{subsystem}/{folder}/{randomfolder}/{fileName}"}, method = RequestMethod.GET)
  public void download(HttpServletRequest request, HttpServletResponse response ,
                         @PathVariable("subsystem") String subsystem, 
                         @PathVariable("folder") String folder,
                         @PathVariable("randomfolder") String randomfolder , 
                         @PathVariable("fileName") String fileName) throws  IOException {
    String relativePath = StringUtils.join("/" , subsystem , "/" , folder , "/" , randomfolder);
    this.download(request, response, relativePath, fileName);
  }
  
  /**
   * 按照给定的文件路径和文件名，下载一般性文件。
   */
  @ApiOperation(value = "按照给定的文件路径和文件名，下载一般性文件。")
  @RequestMapping(path = {"/download/{folder}/{randomfolder}/{fileName}"}, method = RequestMethod.GET)
  public void download(HttpServletRequest request, HttpServletResponse response ,
                          @PathVariable("folder") String folder, 
                          @PathVariable("randomfolder") String randomfolder , 
                          @PathVariable("fileName") String fileName) throws  IOException {
    String relativePath = StringUtils.join("/" , folder , "/" , randomfolder);
    this.download(request, response, relativePath, fileName);
  }

  @GetMapping("/findByFileNameAndRelativeLocal")
  @ApiOperation("根据文件名和相对路径查询文件信息")
  public ResponseModel findByFileNameAndRelativeLocal(@ApiParam(name = "relativeLocal", value = "相对路径", required = true) @RequestParam("relativeLocal") String relativeLocal,
                                                      @ApiParam(name = "fileName", value = "文件名", required = true) @RequestParam("fileName") String fileName) {
    try {
      OrdinaryFileVo ordinaryFile = fileHandleService.findByFileNameAndRelativeLocal(fileName, relativeLocal);
      return this.buildHttpResultW(ordinaryFile);
    } catch (RuntimeException e) {
      LOGGER.error(e.getMessage(), e);
      return this.buildHttpResultForException(e);
    }
  }

  /**
   * 根据过期时间查询文件
   * @param date
   * @return
   */
  @GetMapping("findByEffectiveDate")
  @ApiOperation("根据过期时间查询文件")
  public ResponseModel findByEffectiveDate(@RequestParam("date") @DateTimeFormat(pattern = "yyyy-MM-dd") Date date) {
    try {
      List<OrdinaryFileVo> files = fileHandleService.findByEffectiveDate(date);
      return this.buildHttpResultW(files);
    } catch (RuntimeException e) {
      LOGGER.error(e.getMessage(), e);
      return this.buildHttpResultForException(e);
    }
  }

  /**
   * 根据ID删除文件
   * @param fileIds
   * @return
   */
  @ApiOperation("根据ID删除文件")
  @DeleteMapping("deleteFiles")
  public ResponseModel deleteFiles(@RequestParam("fileIds") String[] fileIds) {
    try {
      fileHandleService.deleteFiles(fileIds);
      return this.buildHttpResult();
    } catch (RuntimeException e) {
      LOGGER.error(e.getMessage(), e);
      return this.buildHttpResultForException(e);
    }
  }

  /**
   * 删除文件，只删除文件系统中的文件，不会删除数据库记录
   * @param relativePath
   * @param fileName
   * @param fileRename
   * @return
   */
  @DeleteMapping("deleteFileByFileRename")
  @ApiOperation(value = "删除文件系统文件", notes = "只删除文件系统中的文件，不会删除数据库记录")
  public ResponseModel deleteFile(@RequestParam("relativePath") String relativePath,  @RequestParam("fileRename") String fileRename) { 
    try {
      fileHandleService.deleteFile(relativePath, fileRename);
      return buildHttpResult();
    } catch (RuntimeException e) {
      LOGGER.error(e.getMessage(), e);
      return this.buildHttpResultForException(e);
    }
  }
  
  /**
   * 该私有方法负责从spring-sercutiy组件中获取当前登录人信息
   * @return
   */
  private String findCurrentAccount() {
    SecurityContext securityContext = SecurityContextHolder.getContext();
    if(securityContext == null) {
      return "admin";
    }
    Authentication authentication = securityContext.getAuthentication();
    if(authentication == null) {
      return "admin";
    }
    return authentication.getName();
  }
}