package com.bizunited.nebula.venus.service.local.service.internal;

import com.bizunited.nebula.common.util.tenant.TenantUtils;
import com.bizunited.nebula.venus.sdk.event.VenusFileEventListener;
import com.bizunited.nebula.venus.sdk.service.file.FileHandleService;
import com.bizunited.nebula.venus.sdk.service.image.ImageHandleService;
import com.bizunited.nebula.venus.sdk.vo.OrdinaryFileVo;
import com.bizunited.nebula.venus.service.local.entity.OrdinaryFileEntity;
import com.bizunited.nebula.venus.service.local.repository.OrdinaryFileRepository;
import com.bizunited.nebula.venus.service.local.service.images.ImageHandler;
import com.bizunited.nebula.venus.service.local.service.images.ImageHandlerBuilder;

import java.awt.image.BufferedImage;
import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.util.List;

import javax.transaction.Transactional;

import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.web.multipart.MultipartFile;

/**
 * 注意，这个实现只针对图片
 * @author yinwenjie
 */
public class ImageHandleServiceImpl implements ImageHandleService {
  /**
   * 日志
   * */
  private static final Log LOGGER = LogFactory.getLog(ImageHandleServiceImpl.class);
  private static final String MESS_CUT_IMAGE = "cutimage";
  private static final String MESS_ZOOM_IMAGE = "zoomimage";
  @Autowired
  private OrdinaryFileRepository ordinaryFileRepository;
  @Autowired
  private FileHandleService fileHandleService;
  @Autowired
  private VenusFileEventListener venusFileEventListener;

  @Override
  @Transactional
  public List<OrdinaryFileVo> imageUpload(String subsystem, String creator, Integer effective, MultipartFile[] files) {
    Validate.notNull(files != null && files.length > 0 , "上传的图片文件信息至少有一个，请检查!!");
    Validate.notBlank(creator , "图片文件的提交人必须传入，请检查!!");
    /*
     * 处理过程为：
     * 1、首先判断文件后缀格式是否为支持的格式。 支持jpg、png、gif格式的图片上传
     * 2、为了保证网络畅通，要控制文件大小在1MB以下，所以也要进行控制（当然也可以通过spring mvc的配置实现限制）
     * 其它步骤请参见fileUpload
     * */
    for (MultipartFile file : files) {
      String originalFilename = file.getOriginalFilename();
      String prefix = null;
      int prefixIndex = originalFilename.lastIndexOf('.');
      if (prefixIndex != -1) {
        prefix = originalFilename.substring(prefixIndex + 1);
        prefix = prefix.toLowerCase();
      }
    }
    // 注意，文件的验证工作会自行在调用过程中完成
    return this.fileHandleService.fileUpload(subsystem, creator , effective, files);
  }

  @Override
  public List<OrdinaryFileVo> imageUpload(String subsystem, String creator, Integer effective, String[] fileNanmes, String[] base64Contents) {
    Validate.notBlank(creator, "图片文件的提交人必须传入，请检查!!");
    if (base64Contents == null || base64Contents.length == 0) {
      throw new IllegalArgumentException("错误的base64编码文件内容!!");
    }
    if (fileNanmes == null || fileNanmes.length == 0) {
      throw new IllegalArgumentException("错误的文件名数量信息!!");
    }
    Validate.isTrue(fileNanmes.length == base64Contents.length, "文件内容数量和文件名数量不匹配，请检查!!");

    for (int index = 0; index < fileNanmes.length; index++) {
      String originalFilename = fileNanmes[index];
      String prefix = null;
      int prefixIndex = originalFilename.lastIndexOf('.');
      if (prefixIndex != -1) {
        prefix = originalFilename.substring(prefixIndex + 1);
        prefix = prefix.toLowerCase();
      }
    }
    
    // 注意，文件的验证工作会自行在调用过程中完成
    return this.fileHandleService.fileUpload(subsystem, creator, effective, fileNanmes, base64Contents);
  }
  
  @Override
  public byte[] imageQuery(String relativeLocal, String fileName) throws  IOException {
    if(StringUtils.isAnyBlank(relativeLocal , fileName)) {
      return new byte[] {};
    }
    OrdinaryFileEntity ordinaryFile = this.ordinaryFileRepository.findByFileNameAndRelativeLocalAndTenantCode(fileName, relativeLocal, TenantUtils.getTenantCode());
    if(ordinaryFile == null) {
      return new byte[] {};
    }
    String prefix = ordinaryFile.getPrefix();
    String originalFileName = ordinaryFile.getOriginalFileName();
    return this.fileContentsQuery(relativeLocal, originalFileName, fileName, "", prefix);
  }

  private byte[] fileContentsQuery(String relativeLocal, String fileName , String fileRename , String special ,String prefix) throws  IOException {
    /*
     * TODO 后续再做ecache缓存
     * 磁盘操作 ============= 
     * 1、根据imageRoot和imageFile的信息在文件系统上提取原始信息，如果没有提取到则不再处理
     * 2、根据既定的特效命令格式，分析请求者要求的特效情况 
     * 3、更具这个特效情况，使用特效构造者构造特效
     * 4、显示到页面上
     */
    // 1、========
    byte[] fileBytes = this.venusFileEventListener.onReadFileContent(relativeLocal, fileRename);
    if (fileBytes == null) {
      return new byte[] {}; 
    }
  
    // 2、=======
    // 如果没有特效要求，就不进行特效处理，直接显示原图（原图不保存）,要进行图片特效，当前文件的扩展名必须是图片的扩展名
    if (StringUtils.isBlank(special) || (!StringUtils.equalsIgnoreCase(prefix, "jpeg")
        && !StringUtils.equalsIgnoreCase(prefix, "jpg") && !StringUtils.equalsIgnoreCase(prefix, "gif")
        && !StringUtils.equalsIgnoreCase(prefix, "bmp") && !StringUtils.equalsIgnoreCase(prefix, "png"))) {
      return fileBytes;
    }
    String[] specialSteps = special.split("\\-\\>");
    if (specialSteps.length == 0) {
      return fileBytes;
    }
    BufferedImage bufferedImage = null;
    try {
      bufferedImage = javax.imageio.ImageIO.read(new ByteArrayInputStream(fileBytes));
    } catch (IOException e) {
      LOGGER.error(e.getMessage(), e);
    }
  
    // 3、=======
    // 图片处理器创建者
    ImageHandlerBuilder.Builder builder = new ImageHandlerBuilder.Builder();
    for (String specialStep : specialSteps) {
      String[] params = specialStep.split("\\|");
      if (StringUtils.equals(params[0], MESS_CUT_IMAGE)) {
        this.builderCutAndZoomImageHandle(params, builder, MESS_CUT_IMAGE);
      } else if (StringUtils.equals(params[0], MESS_ZOOM_IMAGE)) {
        this.builderCutAndZoomImageHandle(params, builder, MESS_ZOOM_IMAGE);
      } else if (StringUtils.equals(params[0], "markimage")) {
        this.builderMarkImageHandle(params, builder);
      }
    }
  
    // 4、=======
    ImageHandler imageHandle = builder.build();
    BufferedImage imageResults = imageHandle.dispose(bufferedImage);
    ByteArrayOutputStream out = new ByteArrayOutputStream();
    javax.imageio.ImageIO.write(imageResults, prefix, out);
    return out.toByteArray();
  }
  
  @Override
  public byte[] imageQuery(String folder, String imageFile, String prefix, String special) throws IOException {
    String relativePath = folder;
    String fileName =  imageFile + "." + prefix;
    return this.fileContentsQuery(relativePath, fileName, fileName, special, prefix);
  }

  /**
   * 这个私有方法用于创建水印图片的特效
   * @param params 水印图片的特效参数（实际上只有一个markValue）
   * @param builder 特效创建器
   * */
  private void builderMarkImageHandle(String[] params, ImageHandlerBuilder.Builder builder) {
    String markValue = null;
    Integer fontSize = 20;
    for (String param : params) {
      String[] paramVars = param.split("=");
      if (StringUtils.equals(paramVars[0], "markValue")) {
        markValue = paramVars[1];
      }
      if (StringUtils.equals(paramVars[0], "fontSize")) {
        fontSize = Integer.parseInt(paramVars[1]);
      }
    }

    // 对参数的正确选定进行判断
    if (StringUtils.isEmpty(markValue)) {
      throw new IllegalArgumentException("markValue must be set value!");
    }
    // 设定到创建器
    builder.createMarkHandler(markValue , fontSize);
  }

  /**
   * 这个私有方法用于创建裁剪图片/缩放图片的特效
   * @param params 裁剪图片/缩放图片的特效参数
   * @param builder 特效创建器
   * */
  private void builderCutAndZoomImageHandle(String[] params, ImageHandlerBuilder.Builder builder, String opType) {
    Integer width = null;
    Integer height = null;
    Float ratio = null;
    for (String param : params) {
      String[] paramVars = param.split("=");
      if (StringUtils.equals(paramVars[0], "width")) {
        width = Integer.parseInt(paramVars[1]);
      } else if (StringUtils.equals(paramVars[0], "height")) {
        height = Integer.parseInt(paramVars[1]);
      } else if (StringUtils.equals(paramVars[0], "ratio")) {
        ratio = Float.parseFloat(paramVars[1]);
      }
    }

    // 对参数的正确选定进行判断
    if (ratio != null && (ratio <= 0f || ratio >= 1f)) {
      throw new IllegalArgumentException("ratio must between from 0 to 1!");
    } else if (ratio == null && (width == null || height == null)) {
      throw new IllegalArgumentException("width and height must be set value!");
    }

    // 设定到创建器
    if (ratio != null && StringUtils.equals(opType, MESS_CUT_IMAGE)) {
      builder.createCutHandler(ratio);
    } else if (ratio != null && StringUtils.equals(opType, MESS_ZOOM_IMAGE)) {
      builder.createZoomHandler(ratio);
    } else if (ratio == null && StringUtils.equals(opType, MESS_CUT_IMAGE)) {
      builder.createCutHandler(width, height);
    } else if (ratio == null && StringUtils.equals(opType, MESS_ZOOM_IMAGE)) {
      builder.createZoomHandler(width, height);
    }
  }
}
