package com.bizunited.nebula.venus.sdk.event;

import java.io.File;

/**
 * 当文件内容在存储介质上（本次磁盘、对象存储、共享存储、针对特定云PaaS组件存储介质）进行存储的事件发生时，该事件被促发</br>
 * 在venus本地实现的模块中，该事件有一个默认实现，但是如果应用程序需要进行重新实现，经过spring boot的配置后该监听会被覆盖
 * @author yinwnejie
 */
public interface VenusFileEventListener {
  /**
   * 保存文件服务，当技术中台中涉及到文件附件的各个功能需要进行物理文件内容的保存时，该接口的服务实现将被调用。
   * @param relativePath 由技术中台中各功能模块，进行文件内容保存调用时，自行设定的基于文件保存根目录的相对路径。有的功能模块的文件保存调用时，可能传入，也可能不传入。
   * 传入的格式一定类似于“/relativePath1/relativePath2”
   * @param fileName 页面操作者在进行文件上传时，选定文件的原始名字，包括了文件的后缀名。
   * 注意这种上传性质的文件附件资料，其最大大小、允许的文件后缀，分别必须满足业务系统中设定的venus.file.maxFileSize属性和venus.file.whitePrefixs属性（不区分大小写）
   * @param fileRename 得到上传的文件后，技术中台系统会重命名这个文件名，以保证文件名在系统中的唯一性（但后缀会保持不变）；重命名的文件名会返回给该接口的实现者
   * 注意：这里有一种情况，就是需要保存的文件并不是由页面操作者明确上传的，而是因为技术中台自身的业务逻辑需要将某些信息作为文件内容进行保存的。
   * 例如在表单模板编辑时，对整个模板布局的json内容进行保存。那么这时，fileName参数和fileRename参数反应出来的值是相同的。
   * @param fileContext 文件内容。文件内容一定至少有1个byte。
   * @throws 如果处理过程出现问题，接口实现者只需要向方法外部抛出人任何一种runtimeException异常即可
   */
  public void onSaveFile(String relativePath, String fileName, String fileRename, byte[] fileContext);
  
  /**
   * 删除文件服务，当技术中台中涉及到文件附件的各个功能需要进行物理文件的删除时，该接口的服务实现将被调用。
   * @param relativePath 请参见saveFile方法中，对应的入参说明
   * @param fileName 描述当前删除的文件，在之前的文件上传操作时的原始文件名（包括后缀名）
   * @param fileRename 描述当前删除的文件，在之前的文件上传操作时，被技术中台重命名后的新的文件名（后缀名不会改变）
   * @throws 如果处理过程出现问题，接口实现者只需要向方法外部抛出人任何一种runtimeException异常即可
   */
  public void onDeleteFile(String relativePath, String fileName, String fileRename);
  /**
   * 文件内容读取服务，当技术中台中涉及到文件附件的各个功能需要进行物理文件内容的读取时，该接口的服务实现将被调用。
   * @param relativePath 请参见saveFile方法中，对应的入参说明
   * @param fileRename 描述当前文件(重命名以后的)，在之前的文件上传操作时，被技术中台重命名后的新的文件名（后缀名不会改变）
   * @throws 如果处理过程出现问题，接口实现者只需要向方法外部抛出人任何一种runtimeException异常即可
   */
  public byte[] onReadFileContent(String relativePath, String fileRename);


  /**
   * 获取文件服务，当技术中台中涉及到文件附件的各个功能需要进行文件获取时，该接口的服务实现将被调用。
   * @param relativePath 请参见saveFile方法中，对应的入参说明
   * @param fileRename 描述当前文件(重命名以后的)，在之前的文件上传操作时，被技术中台重命名后的新的文件名（后缀名不会改变）
   * @throws 如果处理过程出现问题，接口实现者只需要向方法外部抛出人任何一种runtimeException异常即可
   */
  default public File onGetFile(String relativePath, String fileRename){ return null;}
}
