package com.bizunited.nebula.venus.sdk.service.file;

import com.bizunited.nebula.venus.sdk.dto.Base64UploadDto;
import com.bizunited.nebula.venus.sdk.vo.OrdinaryFileVo;
import com.google.common.collect.Lists;
import org.springframework.web.multipart.MultipartFile;

import java.io.File;
import java.util.Date;
import java.util.List;

/**
 * 一般附件（文件）的处理定义
 * @author yinwenjie
 */
public interface FileHandleService {
  /**
   * 文件上传服务，这个文件不一定是图片，只要满足配置信息中规定后缀的文件都可以上传，注意这些文件的有效期为永久。
   * 如果要更改文件有效时间，则需要指定effective参数，该参数是指定文件有效时长，单位为“天”
   * @param subsystem 指代进行文件上传的子系统信息，子系统将单独生成一个文件夹。
   * @param creator 创建人
   * @param effective 这些文件的有效时长，单位为“天”。如果没有传入则默认这些文件有效期为永久
   * @param files 多个文件的描述信息
   * @param special 特效参数
   * @return 
   * @throws IllegalArgumentException
   */
  public List<OrdinaryFileVo> fileUpload(String subsystem, String creator, Integer effective, MultipartFile[] files, String special);
  
  public default List<OrdinaryFileVo> fileUpload(String subsystem, String creator, Integer effective, MultipartFile[] files) {
    return fileUpload(subsystem, creator, effective, files, null);
  }
  
  /**
   * 文件上传服务，这个文件不一定是图片，只要满足配置信息中规定后缀的文件都可以上传，注意这些文件的有效期为永久。
   * 如果要更改文件有效时间，则需要指定effective参数，该参数是指定文件有效时长，单位为“天”</p>
   * 并且上传的文件内容为base64编码（以Base64UploadDto对象进行的描述）
   * @param subsystem 指代进行文件上传的子系统信息，子系统将单独生成一个文件夹。
   * @param creator 创建人
   * @param effective 这些文件的有效时长，单位为“天”。如果没有传入则默认这些文件有效期为永久
   * @param fileNames 文件上传时的文件名
   * @param base64Contents 文件内容的base64编码描述
   * @param special 特效参数
   * @return
   * @throws IllegalArgumentException
   */
  public List<OrdinaryFileVo> fileUpload(String subsystem, Base64UploadDto base64UploadDto, String special);
  
  public default List<OrdinaryFileVo> fileUpload(String subsystem, Base64UploadDto base64UploadDto) {
    return fileUpload(subsystem, base64UploadDto, null);
  }
  
  /**
   * 实际上这就是单个文件的保存，参见：</br>
   * fileUpload(String subsystem, String creator, Integer effective, String[] fileNanmes, String[] base64Contents, String special)
   * @return
   */
  public OrdinaryFileVo fileUpload(String subsystem, String creator, Integer effective, String fileName, byte[] contents, String special);
  
  public default OrdinaryFileVo fileUpload(String subsystem, String creator, Integer effective, String fileName, byte[] contents) {
    return fileUpload(subsystem, creator, effective, fileName, contents, null);
  }
  
  /**
   * 这个文件上传方式，不会生成重命名文件名（fileRename），其它逻辑可以参见：</br>
   * fileUpload(String subsystem, String creator, Integer effective, String[] fileNanmes, String[] base64Contents, String special)
   */
  public OrdinaryFileVo fileUpload(String subsystem, String creator, Integer effective, String fileRename, String fileName , byte[] contents, String special);
  
  public default OrdinaryFileVo fileUpload(String subsystem, String creator, Integer effective, String fileRename, String fileName , byte[] contents) {
    return fileUpload(subsystem, creator, effective, fileRename, fileName, contents, null);
  }
  
  /**
   * 根据指定的文件唯一标识（ID），查询文件的基本信息
   * @param id
   * @return
   */
  public OrdinaryFileVo findById(String id);

  /**
   * 根据指定的文件唯一标识（ID）集合，查询文件的基本信息
   * @param ids
   * @return
   */
  public default List<OrdinaryFileVo> findByIds(List<String> ids) {
    return Lists.newArrayList();
  }
  /**
   * 按照当前指定的时间点，查询文件有效时间小于这个时间点所有文件信息
   * @param currentDate 当前指定的时间点
   * @return 
   */
  public List<OrdinaryFileVo> findByEffectiveDate(Date currentDate);
  /**
   * 按照文件名和相对路径进行查询
   * @param fileName
   * @param relativeLocal
   * @return
   */
  public OrdinaryFileVo findByFileNameAndRelativeLocal(String fileName, String relativeLocal);
  /**
   * 按照文件相对路径和文件重命名后的文件名，查询文件内容
   * @param relativePath 
   * @param fileRename 
   * @return
   */
  public byte[] findContentByFilePathAndFileRename(String relativePath, String fileRename);
  /**
   * 将指定文件的有效期设置为“永久”，实际上有效期截止时间将设置为3999-01-01 00:00:00
   * @param fileReNames 文件信息是文件存储的相对路径 + 重命名后的文件名。例如/file/20190402/3/e18ab4f4-9fad-46c7-9754-fdfc3f08c488.jpg
   */
  public void updateByEffective(String[] fileReNames);
  /**
   * 删除指定文件编号下的文件信息，包括数据库信息和文件信息
   * @param fileIds
   */
  public void deleteFiles(String[] fileIds);
  /**
   * 删除指定的文件，包括数据库信息和文件信息
   * @param relativePath
   * @param fileName
   */
  void deleteFile(String relativePath, String fileName);

  /**
   * 通过路径获取文件
   * @param relativePath
   * @param fileRename
   * @return
   */
  File findFileByFilePathAndFileRename(String relativePath, String fileRename);
}