package com.bizunited.platform.dictionary.web.starter.controller;

import com.alibaba.fastjson.JSON;
import com.bizunited.platform.common.controller.BaseController;
import com.bizunited.platform.common.controller.model.ResponseModel;
import com.bizunited.platform.dictionary.common.service.dict.DictService;
import com.bizunited.platform.dictionary.common.vo.DictVo;
import io.swagger.annotations.ApiOperation;
import io.swagger.annotations.ApiParam;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.Pageable;
import org.springframework.data.web.PageableDefault;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.RestController;

import java.util.Set;

/**
 * DictController
 *
 * @description:
 * @author: yanwe
 * @date: 11/Feb/2019 10:32
 */
@RestController
@RequestMapping("/v1/nebula/dicts")
public class DictController extends BaseController {

  private static final Logger LOGGER = LoggerFactory.getLogger(DictController.class);
  @Autowired
  private DictService dictService;

  @ApiOperation(value = "查询所有字典标题组")
  @RequestMapping(value = "/findAll", method = RequestMethod.GET)
  public ResponseModel findAll() {
    try {
      return buildHttpResultW(dictService.findAll(), "");
    } catch (RuntimeException e) {
      LOGGER.error(e.getMessage(), e);
      return buildHttpResultForException(e);
    }
  }

  @ApiOperation(value = "根据启用状态查询字典")
  @RequestMapping(value = "/findByStatus", method = RequestMethod.GET)
  public ResponseModel findByStatus(
      @RequestParam(name = "dictStatus") @ApiParam(name = "dictStatus", value = "是否已上架启用") Boolean dictStatus) {
    try {
      return buildHttpResultW(dictService.findByStatus(dictStatus), "");
    } catch (RuntimeException e) {
      LOGGER.error(e.getMessage(), e);
      return buildHttpResultForException(e);
    }
  }

  @ApiOperation(value = "根据字典的编号,类型,状态查询")
  @RequestMapping(value = "/findByConditions", method = RequestMethod.GET)
  public ResponseModel findByConditions(
      @RequestParam(name = "dictCode") @ApiParam(name = "dictCode", value = "字典的类型编号") String dictCode,
      @RequestParam(name = "dictTitle") @ApiParam(name = "dictTitle", value = "字典的类型标题") String dictTitle,
      @RequestParam(name = "dictType") @ApiParam(name = "dictType", value = "字典的运用范围，暂时限定为 通用/业务") String dictType,
      @RequestParam(name = "dictStatus") @ApiParam(name = "dictStatus", value = "是否已上架启用") Boolean dictStatus,
      @PageableDefault(value = 50) Pageable pageable) {
    try {
      Page<DictVo> page = dictService.findByConditions(dictCode, dictTitle, dictType, dictStatus, pageable);
      return buildHttpResultW(page, "");
    } catch (RuntimeException e) {
      LOGGER.error(e.getMessage(), e);
      return buildHttpResultForException(e);
    }
  }

  @ApiOperation(value = "创建字典类")
  @RequestMapping(value = "", method = RequestMethod.POST)
  public ResponseModel create(@ApiParam(name = "字典信息") @RequestBody DictVo dictVo,
      @RequestParam(name = "categoryId") @ApiParam(name = "categoryId", value = "字典分类id") String categoryId) {
    LOGGER.debug("待新增/修改的字典信息：{}", JSON.toJSONString(dictVo));
    try {
      return this.buildHttpResultW(dictService.create(dictVo, categoryId), "dictItems");
    } catch (RuntimeException e) {
      LOGGER.error(e.getMessage(), e);
      return buildHttpResultForException(e);
    }
  }

  @ApiOperation(value = "修改字典类")
  @RequestMapping(value = "", method = RequestMethod.PATCH)
  public ResponseModel update(@ApiParam(name = "字典信息") @RequestBody DictVo dictVo,
      @RequestParam(name = "categoryId") @ApiParam(name = "categoryId", value = "字典分类id") String categoryId) {
    LOGGER.debug("待新增/修改的字典信息：{}", JSON.toJSONString(dictVo));
    try {
      return this.buildHttpResultW(dictService.update(dictVo, categoryId), "dictItems");
    } catch (RuntimeException e) {
      LOGGER.error(e.getMessage(), e);
      return buildHttpResultForException(e);
    }
  }


  @ApiOperation(value = "字典类复制升级")
  @RequestMapping(value = "/upgrade", method = RequestMethod.POST)
  public ResponseModel upgrade(
      @RequestParam(name = "newDictCode") @ApiParam(name = "newDictCode", value = "新字典编码") String newDictCode,
      @RequestParam(name = "dictCode") @ApiParam(name = "dictCode", value = "字典编码") String dictCode) {
    LOGGER.debug("新字典编码：{},复制的旧字典信息：{}", newDictCode, dictCode);
    try {
      return this.buildHttpResultW(dictService.upgrade(newDictCode, dictCode), "");
    } catch (RuntimeException e) {
      LOGGER.error(e.getMessage(), e);
      return buildHttpResultForException(e);
    }
  }

  @ApiOperation(value = "上架字典")
  @RequestMapping(value = "/update", method = RequestMethod.PATCH)
  public ResponseModel update(
      @RequestParam(name = "dictCode") @ApiParam(name = "dictCode", value = "字典编码") String dictCode) {
    LOGGER.debug("上架字典编码：{}", dictCode);
    try {
      return this.buildHttpResultW(dictService.onshelf(dictCode), "");
    } catch (RuntimeException e) {
      LOGGER.error(e.getMessage(), e);
      return buildHttpResultForException(e);
    }
  }

  @ApiOperation(value = "删除某字典")
  @RequestMapping(value = "/delete", method = RequestMethod.DELETE)
  public ResponseModel delete(
      @RequestParam(name = "dictCode") @ApiParam(name = "dictCode", value = "字典编码") String dictCode) {
    LOGGER.debug("上架字典编码：{}", dictCode);
    try {
      dictService.delete(dictCode);
      return this.buildHttpResult();
    } catch (RuntimeException e) {
      LOGGER.error(e.getMessage(), e);
      return buildHttpResultForException(e);
    }
  }

  @ApiOperation(value = "根据一个或者多个ID信息，查询字典明细信息，包括字典的各种关联信息（但是不包括下级关联信息）")
  @GetMapping("/findDetailsByIds")
  public ResponseModel findDetailsByIds(
          @RequestParam(name = "dictIds") @ApiParam(name = "dictIds", value = "字典ids") String[] dictIds) {
    try {
      Set<DictVo> dictVos = dictService.findDetailsByIds(dictIds);
      return this.buildHttpResult(dictVos);
    } catch (RuntimeException e) {
      LOGGER.error(e.getMessage(), e);
      return buildHttpResultForException(e);
    }
  }

  @ApiOperation(value = "按照指定的数据字典的ID信息，查询最终满足这些id编号的数据条目。")
  @GetMapping("/countByIds")
  public ResponseModel countByIds(
          @RequestParam(name = "ids") @ApiParam(name = "ids", value = "字典ids") String[] ids) {
    try {
      int count = dictService.countByIds(ids);
      return this.buildHttpResult(count);
    } catch (RuntimeException e) {
      LOGGER.error(e.getMessage(), e);
      return buildHttpResultForException(e);
    }
  }

  @ApiOperation(value = "根据编码查询（包括关联信息）")
  @GetMapping("/findDetailsByDictCode")
  public ResponseModel findDetailsByDictCode(
          @RequestParam(name = "dictCode") @ApiParam(name = "dictCode", value = "字典编码") String dictCode) {
    try {
      DictVo dictVo = dictService.findDetailsByDictCode(dictCode);
      return this.buildHttpResultW(dictVo, "category", "dictItems");
    } catch (RuntimeException e) {
      LOGGER.error(e.getMessage(), e);
      return buildHttpResultForException(e);
    }
  }

  @ApiOperation(value = "根据编码查询")
  @GetMapping("/findByDictCode")
  public ResponseModel findByDictCode(
          @RequestParam(name = "dictCode") @ApiParam(name = "dictCode", value = "字典编码") String dictCode) {
    try {
      DictVo dictVo = dictService.findByDictCode(dictCode);
      return this.buildHttpResultW(dictVo);
    } catch (RuntimeException e) {
      LOGGER.error(e.getMessage(), e);
      return buildHttpResultForException(e);
    }
  }
}
