package com.bizunited.platform.dictionary.web.starter.controller;

import com.alibaba.fastjson.JSON;
import com.bizunited.platform.common.controller.BaseController;
import com.bizunited.platform.common.controller.model.ResponseModel;
import com.bizunited.platform.dictionary.common.service.dictItem.DictItemService;
import com.bizunited.platform.dictionary.common.vo.DictItemVo;
import io.swagger.annotations.ApiOperation;
import io.swagger.annotations.ApiParam;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.web.bind.annotation.DeleteMapping;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.PutMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.RestController;

import java.util.Arrays;
import java.util.Set;
import java.util.stream.Collectors;

/**
 * DictItemController
 *
 * @description:
 * @author: yanwe
 * @date: 14/Feb/2019 09:52
 */
@RestController
@RequestMapping("/v1/nebula/dictItems")
public class DictItemController extends BaseController {

  private static final Logger LOGGER = LoggerFactory.getLogger(DictItemController.class);

  @Autowired
  private DictItemService dictItemService;

  @ApiOperation(value = "根据字典编码查询所有子项")
  @RequestMapping(value = "/findByCode", method = RequestMethod.GET)
  public ResponseModel findByCode(@RequestParam(name = "dictCode") @ApiParam(name = "dictCode", value = "指定的字典编码") String dictCode) {
    try {
      return buildHttpResultW(dictItemService.findItemsByCode(dictCode), "");
    } catch (RuntimeException e) {
      LOGGER.error(e.getMessage(), e);
      return buildHttpResultForException(e);
    }
  }

  @ApiOperation(value = "为字典类 修改/添加/删除 明细项")
  @RequestMapping(value = "/update", method = RequestMethod.POST)
  public ResponseModel update(
          @RequestParam(name = "dictCode") @ApiParam(name = "dictCode", value = "字典编码") String dictCode,
          @ApiParam(name = "字典明细信息") @RequestBody Set<DictItemVo> items) {
    LOGGER.debug("待新增/修改的字典编码：{},字典信息：{}", dictCode, JSON.toJSONString(items));
    try {
      return buildHttpResultW(dictItemService.save(dictCode, items), "");
    } catch (RuntimeException e) {
      LOGGER.error(e.getMessage(), e);
      return buildHttpResultForException(e);
    }
  }

  /**
   * 根据字典ID启用字典
   * @param id
   * @return
   */
  @ApiOperation(value = "启用字典")
  @PutMapping("/enable")
  public ResponseModel enable(@ApiParam(name = "字典ID") @RequestParam("id") String id){
    try {
      DictItemVo dictItem = dictItemService.enable(id);
      return buildHttpResultW(dictItem);
    }catch (RuntimeException e){
      LOGGER.error(e.getMessage(), e);
      return buildHttpResultForException(e);
    }
  }

  /**
   * 根据字典ID禁用字典
   * @param id
   * @return
   */
  @ApiOperation(value = "禁用字典")
  @PutMapping("/disable")
  public ResponseModel disable(@ApiParam(name = "字典ID") @RequestParam("id") String id){
    try {
      DictItemVo dictItem = dictItemService.disable(id);
      return buildHttpResultW(dictItem);
    }catch (RuntimeException e){
      LOGGER.error(e.getMessage(), e);
      return buildHttpResultForException(e);
    }
  }

  /**
   * 创建字典明细项(添加单个)
   * @param dictItem
   * @param dictCode
   * @return
   */
  @ApiOperation(value = "创建字典明细项(只能添加单个数据字典明细项)")
  @PostMapping("")
  public ResponseModel create(@ApiParam(name = "创建字典明细项") @RequestBody DictItemVo dictItem,
                              @RequestParam(name = "dictCode") @ApiParam(name = "dictCode", value = "字典编码") String dictCode) {
    try {
      DictItemVo result = dictItemService.create(dictCode, dictItem);
      return this.buildHttpResultW(result);
    } catch (RuntimeException e) {
      LOGGER.error(e.getMessage(), e);
      return buildHttpResultForException(e);
    }
  }

  /**
   * 根据id查询明细
   * @param id
   * @return
   */
  @ApiOperation(value = "创建字典明细项(根据id查询明细)")
  @GetMapping("/findDetailsById")
  public ResponseModel findDetailsById(@ApiParam(name = "字典明细项id") @RequestParam("id") String  id) {
    try {
      DictItemVo result = dictItemService.findDetailsById(id);
      return this.buildHttpResultW(result, "dict");
    } catch (RuntimeException e) {
      LOGGER.error(e.getMessage(), e);
      return buildHttpResultForException(e);
    }
  }

  /**
   * 根据字典dictCode和字典值的key查询一个数据字典值详细
   * @param dictCode,key
   * @return
   */
  @ApiOperation(value = "根据字典dictCode和字典值的key查询一个数据字典值详细")
  @GetMapping("/findByKeyAndDictCode")
  public ResponseModel findByKeyAndDictCode(@ApiParam(name = "字典code") @RequestParam("dictCode") String dictCode,
                                            @ApiParam(name = "字典值key") @RequestParam("key") String key) {
    try {
      DictItemVo result = dictItemService.findByKeyAndDictCode(dictCode, key);
      return this.buildHttpResultW(result, "dictEntity");
    } catch (RuntimeException e) {
      LOGGER.error(e.getMessage(), e);
      return buildHttpResultForException(e);
    }
  }

  /**
   * 根据ids删除信息
   */
  @ApiOperation(value = "根据ids删除信息")
  @DeleteMapping("deleteByIds")
  public ResponseModel deleteByIds(@RequestParam("ids") @ApiParam(name = "ids" , value = "字典明细ids") String[] ids) {
    try {
      Set<String> pks = Arrays.stream(ids).collect(Collectors.toSet());
      dictItemService.deleteByIds(pks);
      return this.buildHttpResult();
    } catch (RuntimeException e) {
      LOGGER.error(e.getMessage(), e);
      return buildHttpResultForException(e);
    }
  }
}
