package com.bizunited.platform.dictionary.service.feign;

import com.bizunited.platform.common.controller.model.ResponseModel;
import com.bizunited.platform.common.interceptor.FeignRequestInterceptor;
import com.bizunited.platform.dictionary.common.vo.DictVo;
import com.bizunited.platform.dictionary.service.feign.fallback.DictFeignClientFallback;
import org.springframework.cloud.openfeign.FeignClient;
import org.springframework.data.domain.Pageable;
import org.springframework.data.web.PageableDefault;
import org.springframework.web.bind.annotation.DeleteMapping;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.PatchMapping;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestParam;

/**
 * 数据字典服务的feign调用
 * @Author: zengxingwang
 * @Date: 2020/3/20 10:16
 */
@FeignClient(qualifier = "DictFeignClient", name = "${dictionary.application.name}", path = "/v1/nebula/dicts", fallback = DictFeignClientFallback.class, configuration = FeignRequestInterceptor.class)
public interface DictFeignClient {

  /**
   * 查询所有字典标题组
   */
  @GetMapping("/findAll")
  ResponseModel findAll();

  /**
   * 根据启用状态查询字典
   * @param dictStatus
   * @return
   */
  @GetMapping("/findByStatus")
  ResponseModel findByStatus(@RequestParam(name = "dictStatus") Boolean dictStatus);

  /**
   * 根据字典的编号,类型,状态查询
   * @param dictCode
   * @param dictTitle
   * @param dictType
   * @param dictStatus
   * @param pageable
   * @return
   */
  @GetMapping("/findByConditions")
  ResponseModel findByConditions(
            @RequestParam(name = "dictCode") String dictCode,
            @RequestParam(name = "dictTitle")  String dictTitle,
            @RequestParam(name = "dictType") String dictType,
            @RequestParam(name = "dictStatus") Boolean dictStatus,
            @PageableDefault(value = 50) Pageable pageable);


  /**
   * 创建字典类
   * @param dictVo
   * @param categoryId
   * @return
   */
  @PostMapping("")
  ResponseModel create(@RequestBody DictVo dictVo, @RequestParam(name = "categoryId") String categoryId);

  /**
   * 修改字典类
   * @param dictVo
   * @param categoryId
   * @return
   */
  @PatchMapping("")
  ResponseModel update(@RequestBody DictVo dictVo, @RequestParam(name = "categoryId") String categoryId);

  /**
   * 字典类复制升级
   * @param newDictCode
   * @param dictCode
   * @return
   */
  @PostMapping("/upgrade")
  ResponseModel upgrade(@RequestParam(name = "newDictCode") String newDictCode, @RequestParam(name = "dictCode") String dictCode);

  /**
   * 上架字典
   * @param dictCode
   * @return
   */
  @PatchMapping("/update")
  ResponseModel update(@RequestParam(name = "dictCode") String dictCode);

  /**
   * 删除某字典
   * @param dictCode
   * @return
   */
  @DeleteMapping("/delete")
  ResponseModel delete(@RequestParam(name = "dictCode") String dictCode);

  /**
   * 根据一个或者多个ID信息，查询字典明细信息，包括字典的各种关联信息（但是不包括下级关联信息）
   * @param dictIds
   * @return
   */
  @GetMapping("/findDetailsByIds")
  ResponseModel findDetailsByIds(@RequestParam(name = "dictIds") String[] dictIds);

  /**
   * 按照指定的数据字典的ID信息，查询最终满足这些id编号的数据条目。
   * @param ids
   * @return
   */
  @GetMapping("/countByIds")
  ResponseModel countByIds(@RequestParam(name = "ids") String[] ids);

  /**
   * 根据编码查询
   * @param dictCode
   * @return
   */
  @GetMapping("/findByDictCode")
  ResponseModel findByDictCode(@RequestParam(name = "dictCode") String dictCode);

  /**
   * 根据编码查询（包括关联信息）
   * @param dictCode
   * @return
   */
  @GetMapping("/findDetailsByDictCode")
  ResponseModel findDetailsByDictCode(@RequestParam(name = "dictCode") String dictCode);
}
