package com.bizunited.platform.dictionary.service.feign.service.internal;

import com.bizunited.platform.common.controller.model.ResponseModel;
import com.bizunited.platform.common.model.MigrateImportModel;
import com.bizunited.platform.common.util.JsonUtils;
import com.bizunited.platform.common.util.PageUtils;
import com.bizunited.platform.dictionary.common.service.dict.DictService;
import com.bizunited.platform.dictionary.common.vo.DictVo;
import com.bizunited.platform.dictionary.service.feign.DictFeignClient;
import com.fasterxml.jackson.core.type.TypeReference;
import com.google.common.collect.Lists;
import com.google.common.collect.Sets;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.PageImpl;
import org.springframework.data.domain.Pageable;
import org.springframework.stereotype.Service;

import java.util.List;
import java.util.Set;

/**
 * 数据字典服务
 * @Author: zengxingwang
 * @Date: 2020/3/20 10:16
 */
@Service("DictServiceImpl")
public class DictServiceImpl implements DictService {

  @Autowired
  private DictFeignClient dictFeignClient;

  /**
   * 查询所有字典标题组
   * @return
   */
  @Override
  public List<DictVo> findAll() {
    ResponseModel response = dictFeignClient.findAll();
    return JsonUtils.convert(response.getSuccessDataOrElse(Lists.newArrayList()), new TypeReference<List<DictVo>>() {});
  }

  /**
   * 根据启用状态查询字典
   * @param dictStatus
   * @return
   */
  @Override
  public List<DictVo> findByStatus(Boolean dictStatus) {
    ResponseModel response = dictFeignClient.findByStatus(dictStatus);
    return JsonUtils.convert(response.getSuccessDataOrElse(Lists.newArrayList()), new TypeReference<List<DictVo>>() {});
  }

  /**
   * 根据字典的编号,类型,状态查询
   * @param dictCode
   * @param dictTitle
   * @param dictType
   * @param dictStatus
   * @param pageable
   * @return
   */
  @Override
  public Page<DictVo> findByConditions(String dictCode, String dictTitle, String dictType, Boolean dictStatus, Pageable pageable) {
    ResponseModel response = dictFeignClient.findByConditions(dictCode, dictTitle, dictType, dictStatus, pageable);
    Object data = response.getSuccessDataOrElse(null);
    if(data == null) {
      return new PageImpl<>(Lists.newArrayList(), pageable, 0);
    }
    return PageUtils.convert2Page(JsonUtils.toJSONObject(data), DictVo.class);
  }

  /**
   * 根据一个或者多个ID信息，查询字典明细信息，包括字典的各种关联信息（但是不包括下级关联信息）
   * @param dictIds
   * @return
   */
  @Override
  public Set<DictVo> findDetailsByIds(String[] dictIds) {
    ResponseModel response = dictFeignClient.findDetailsByIds(dictIds);
    return JsonUtils.convert(response.getSuccessDataOrElse(Sets.newHashSet()), new TypeReference<Set<DictVo>>() {});
  }

  /**
   * 按照指定的数据字典的ID信息，查询最终满足这些id编号的数据条目。
   * @param ids 指定的id信息
   * @return
   */
  @Override
  public int countByIds(String[] ids) {
    ResponseModel response = dictFeignClient.countByIds(ids);
    Validate.isTrue(response.getSuccess(), response.getErrorMsg());
    return (int) response.getData();
  }

  /**
   * 创建字典类
   * @param dictVo
   * @param categoryId
   * @return
   */
  @Override
  public DictVo create(DictVo dictVo, String categoryId) {
    Validate.notNull(dictVo, "输入字典不可为空!");
    // 判断该字典是否已经是启用状态，若为启用，则不可再编辑
    Validate.notBlank(dictVo.getDictCode(), "dict code 不能为空!");
    Validate.notBlank(dictVo.getDictTitle(), "dict title 不能为空!");
    Validate.notBlank(categoryId, "字典分组ID不能为空");
    ResponseModel response = dictFeignClient.create(dictVo, categoryId);
    Validate.isTrue(response.getSuccess(), response.getErrorMsg());
    return JsonUtils.convert(response.getSuccessDataOrElse(null), DictVo.class);
  }

  /**
   * 修改字典类
   * @param dictVo
   * @param categoryId
   * @return
   */
  @Override
  public DictVo update(DictVo dictVo, String categoryId) {
    Validate.notNull(dictVo, "输入字典不可为空!");
    // 判断该字典是否已经是启用状态，若为启用，则不可再编辑
    Validate.notBlank(dictVo.getDictCode(), "dict code 不能为空!");
    Validate.notBlank(dictVo.getDictTitle(), "dict title 不能为空!");
    Validate.notBlank(categoryId, "字典分组ID不能为空");
    ResponseModel response = dictFeignClient.update(dictVo, categoryId);
    Validate.isTrue(response.getSuccess(), response.getErrorMsg());
    return JsonUtils.convert(response.getSuccessDataOrElse(null), DictVo.class);
  }

  /**
   * 字典类复制升级
   * @param newDictCode
   * @param dictCode
   * @return
   */
  @Override
  public DictVo upgrade(String newDictCode, String dictCode) {
    Validate.notBlank(dictCode, "查询字典编码不能为空");
    ResponseModel response = dictFeignClient.upgrade(newDictCode, dictCode);
    Validate.isTrue(response.getSuccess(), response.getErrorMsg());
    return JsonUtils.convert(response.getSuccessDataOrElse(null), DictVo.class);
  }

  /**
   * 上架字典
   * @param dictCode
   * @return
   */
  @Override
  public DictVo onshelf(String dictCode) {
    Validate.notBlank(dictCode, "查询字典编码不能为空");
    ResponseModel response = dictFeignClient.update(dictCode);
    Validate.isTrue(response.getSuccess(), response.getErrorMsg());
    return JsonUtils.convert(response.getSuccessDataOrElse(null), DictVo.class);
  }

  /**
   * 删除某字典
   * @param dictCode
   */
  @Override
  public void delete(String dictCode) {
    Validate.notBlank(dictCode, "查询字典编码不能为空");
    ResponseModel response = dictFeignClient.delete(dictCode);
    Validate.isTrue(response.getSuccess(), response.getErrorMsg());
  }

  /**
   * 根据编码查询
   * @param dictCode
   * @return
   */
  @Override
  public DictVo findByDictCode(String dictCode) {
    ResponseModel response = dictFeignClient.findByDictCode(dictCode);
    return JsonUtils.convert(response.getSuccessDataOrElse(null), DictVo.class);
  }

  /**
   * 根据编码查询（包括关联信息）
   * @param dictCode
   * @return
   */
  @Override
  public DictVo findDetailsByDictCode(String dictCode) {
    ResponseModel response = dictFeignClient.findDetailsByDictCode(dictCode);
    return JsonUtils.convert(response.getSuccessDataOrElse(null), DictVo.class);
  }

  /**
   * 数据字典导入
   * @param importModel
   */
  @Override
  public void importData(MigrateImportModel importModel) {
    throw new UnsupportedOperationException("暂不支持");
  }

  @Override
  public Set<DictVo> findDetailsByDictCodes(String[] dictCodes) {
    ResponseModel response = dictFeignClient.findDetailsByDictCodes(dictCodes);
    return JsonUtils.convert(response.getSuccessDataOrElse(Sets.newHashSet()), new TypeReference<Set<DictVo>>() {});
  }
}
