package com.bizunited.platform.dictionary.service.feign.service.internal;

import com.bizunited.platform.common.controller.model.ResponseModel;
import com.bizunited.platform.common.util.ChineseCharUtils;
import com.bizunited.platform.common.util.JsonUtils;
import com.bizunited.platform.dictionary.common.service.dictCategory.DictCategoryService;
import com.bizunited.platform.dictionary.common.vo.DictCategoryVo;
import com.bizunited.platform.dictionary.service.feign.DictCategoryFeignClient;
import com.fasterxml.jackson.core.type.TypeReference;
import com.google.common.collect.Sets;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import java.util.LinkedHashSet;
import java.util.Map;
import java.util.Set;

/**
 * 数据字典分组服务
 * @Author: zengxingwang
 * @Date: 2020/3/20 10:16
 */
@Service("DictCategoryServiceImpl")
public class DictCategoryServiceImpl implements DictCategoryService {

  @Autowired
  private DictCategoryFeignClient dictCategoryFeignClient;

  /**
   * 新增一个字典分组
   * @param dictCategoryVo 字典信息
   * @param parentId 父类分组ID，如为顶级，则可为空
   * @return
   */
  @Override
  public DictCategoryVo create(DictCategoryVo dictCategoryVo, String parentId) {
    Validate.notNull(dictCategoryVo,"字典分类不能为空！");
    Validate.isTrue(StringUtils.isBlank(dictCategoryVo.getId()),"新增时不能填入Id!");
    Validate.notBlank(dictCategoryVo.getCateName(),"字典分类名称不能为空！");
    Validate.notBlank(dictCategoryVo.getCateCode(),"字典分类编码不能为空！");
    Validate.isTrue(!ChineseCharUtils.hasChinese(dictCategoryVo.getCateCode()),"字典分类编码不能含有中文！");
    ResponseModel response = dictCategoryFeignClient.create(dictCategoryVo, parentId);
    Validate.isTrue(response.getSuccess(), response.getErrorMsg());
    return JsonUtils.convert(response.getSuccessDataOrElse(null), DictCategoryVo.class);
  }

  /**
   * 更新一个字典分组
   * @param dictCategoryVo
   * @return
   */
  @Override
  public DictCategoryVo update(DictCategoryVo dictCategoryVo) {
    Validate.notNull(dictCategoryVo,"字典分类不能为空！");
    Validate.notBlank(dictCategoryVo.getId(),"新增时不能填入Id!");
    ResponseModel response = dictCategoryFeignClient.update(dictCategoryVo);
    Validate.isTrue(response.getSuccess(), response.getErrorMsg());
    return JsonUtils.convert(response.getSuccessDataOrElse(null), DictCategoryVo.class);
  }

  /**
   * 将两个字典分组列为父子级关系
   * @param childId 将要作为子级的字典分组ID
   * @param parentId 将要作为父级的字典分组ID, 如果为空，则设置子级字典为根级
   */
  @Override
  public void bind(String childId, String parentId) {
    Validate.notBlank(childId,"子级字典分类ID不能为空！");
    Validate.notBlank(parentId,"父级字典分类ID不能为空！");
    ResponseModel response = dictCategoryFeignClient.bind(childId, parentId);
    Validate.isTrue(response.getSuccess(), response.getErrorMsg());
  }

  /**
   * 查询所有,并以树形结构列出
   * @return
   */
  @Override
  public Set<DictCategoryVo> findAll() {
    ResponseModel response = dictCategoryFeignClient.findAll();
    return JsonUtils.convert(response.getSuccessDataOrElse(Sets.newLinkedHashSet()), new TypeReference<LinkedHashSet<DictCategoryVo>>() {});
  }

  /**
   * 根据id查询基本信息，没有任何关联信息
   * @param id
   * @return
   */
  @Override
  public DictCategoryVo findById(String id) {
    Validate.notBlank(id,"字典分类ID不能为空！");
    ResponseModel response = dictCategoryFeignClient.findById(id);
    return JsonUtils.convert(response.getSuccessDataOrElse(null), DictCategoryVo.class);
  }

  /**
   * 根据id查询字典分类的基本信息和关联信息（但是不包括字典分类的子级集合关联）
   * @param id
   * @return
   */
  @Override
  public DictCategoryVo findDetailsById(String id) {
    Validate.notBlank(id,"字典分类ID不能为空！");
    ResponseModel response = dictCategoryFeignClient.findDetailsById(id);
    return JsonUtils.convert(response.getSuccessDataOrElse(null), DictCategoryVo.class);
  }

  /**
   * 根据数据字典业务编号code进行查询，不包括任何关联信息
   * @param code 业务编号code在全系统唯一
   * @return
   */
  @Override
  public DictCategoryVo findByCode(String code) {
    Validate.notBlank(code,"字典分类编码不能为空！");
    ResponseModel response = dictCategoryFeignClient.findByCode(code);
    return JsonUtils.convert(response.getSuccessDataOrElse(null), DictCategoryVo.class);
  }

  /**
   * 根据分组id删除分组（不能删除有字典的分组）
   * @param id
   * @return
   */
  @Override
  public void deleteById(String id) {
    Validate.notBlank(id, "字典分类id不能为空！");
    ResponseModel response = dictCategoryFeignClient.deleteById(id);
    Validate.isTrue(response.getSuccess(), response.getErrorMsg());
  }

  /**
   * 导入处理字典分类
   * @param id
   * @param dictCategoryMap
   * @return
   */
  @Override
  public String importCategory(String id, Map<String, DictCategoryVo> dictCategoryMap) {
    throw new UnsupportedOperationException("暂不支持");
  }
}
