package com.bizunited.platform.dictionary.service.feign.service.internal;

import com.bizunited.platform.common.controller.model.ResponseModel;
import com.bizunited.platform.common.util.JsonUtils;
import com.bizunited.platform.dictionary.common.service.dictItem.DictItemService;
import com.bizunited.platform.dictionary.common.vo.DictItemVo;
import com.bizunited.platform.dictionary.service.feign.DictItemFeignClient;
import com.fasterxml.jackson.core.type.TypeReference;
import com.google.common.collect.Lists;
import com.google.common.collect.Sets;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import java.util.List;
import java.util.Set;

/**
 * 数据字典项服务
 * @Author: zengxingwang
 * @Date: 2020/3/20 10:16
 */
@Service("DictItemServiceImpl")
public class DictItemServiceImpl implements DictItemService {

  @Autowired
  private DictItemFeignClient dictItemFeignClient;

  /**
   * 根据字典编码查询所有子项
   * @param dictCode
   * @return
   */
  @Override
  public List<DictItemVo> findItemsByCode(String dictCode) {
    ResponseModel response = dictItemFeignClient.findByCode(dictCode);
    return JsonUtils.convert(response.getSuccessDataOrElse(Lists.newArrayList()), new TypeReference<List<DictItemVo>>() {});
  }

  /**
   * 为字典类 修改/添加/删除 明细项的保存
   * @param dictCode
   * @param items
   * @return
   */
  @Override
  public Set<DictItemVo> save(String dictCode, Set<DictItemVo> items) {
    ResponseModel response = dictItemFeignClient.update(dictCode, items);
    Validate.isTrue(response.getSuccess(), response.getErrorMsg());
    return JsonUtils.convert(response.getSuccessDataOrElse(Sets.newHashSet()), new TypeReference<Set<DictItemVo>>() {});
  }

  /**
   * 根据字典ID启用字典
   * @param dictItemId
   * @return
   */
  @Override
  public DictItemVo enable(String dictItemId) {
    Validate.notBlank(dictItemId, "字典明细ID不能为空");
    ResponseModel response = dictItemFeignClient.enable(dictItemId);
    Validate.isTrue(response.getSuccess(), response.getErrorMsg());
    return JsonUtils.convert(response.getSuccessDataOrElse(null), DictItemVo.class);
  }

  /**
   * 根据字典ID禁用字典
   * @param dictItemId
   * @return
   */
  @Override
  public DictItemVo disable(String dictItemId) {
    Validate.notBlank(dictItemId, "字典明细ID不能为空");
    ResponseModel response = dictItemFeignClient.disable(dictItemId);
    Validate.isTrue(response.getSuccess(), response.getErrorMsg());
    return JsonUtils.convert(response.getSuccessDataOrElse(null), DictItemVo.class);
  }

  /**
   * 创建字典明细项(添加单个)
   * @param dictCode
   * @param dictItem
   * @return
   */
  @Override
  public DictItemVo create(String dictCode, DictItemVo dictItem) {
    Validate.notBlank(dictCode, "字典编码不能为空，请检查");
    Validate.notNull(dictItem, "保存对象不存在!");
    Validate.isTrue(dictItem.getId() == null, "保存数据ID必须为空!");
    Validate.notBlank(dictItem.getDictKey(), "字典名称不能为空，请检查!");
    Validate.notBlank(dictItem.getDictValue(), "字典值不能为空，请检查!");
    ResponseModel response = dictItemFeignClient.create(dictItem, dictCode);
    Validate.isTrue(response.getSuccess(), response.getErrorMsg());
    return JsonUtils.convert(response.getSuccessDataOrElse(null), DictItemVo.class);
  }

  /**
   * 根据id查询明细
   * @param id
   * @return
   */
  @Override
  public DictItemVo findDetailsById(String id) {
    Validate.notNull(id, "查询字典值明细参数为空，请检查!");
    ResponseModel response = dictItemFeignClient.findDetailsById(id);
    Validate.isTrue(response.getSuccess(), response.getErrorMsg());
    return JsonUtils.convert(response.getSuccessDataOrElse(null), DictItemVo.class);
  }

  @Override
  public DictItemVo findByKeyAndDictCode(String dictCode, String key) {
    if (StringUtils.isBlank(dictCode) || StringUtils.isBlank(key)){
      return null;
    }
    ResponseModel response = dictItemFeignClient.findByKeyAndDictCode(dictCode, key);
    Validate.isTrue(response.getSuccess(), response.getErrorMsg());
    return JsonUtils.convert(response.getSuccessDataOrElse(null), DictItemVo.class);
  }

  @Override
  public void deleteByIds(Set<String> ids) {
    Validate.notEmpty(ids,"传入id信息不能为空");
    ResponseModel response = dictItemFeignClient.deleteByIds(ids.toArray(new String[0]));
    Validate.isTrue(response.getSuccess(), response.getErrorMsg());
  }

}
