package com.bizunited.platform.dictionary.service.local.service.internal;

import com.bizunited.platform.common.enums.ImportExecuteModeEnum;
import com.bizunited.platform.common.enums.NormalStatusEnum;
import com.bizunited.platform.common.model.MigrateImportModel;
import com.bizunited.platform.common.service.NebulaToolkitService;
import com.bizunited.platform.common.util.FileUtils;
import com.bizunited.platform.dictionary.common.service.DictImportService;
import com.bizunited.platform.dictionary.common.service.dict.DictService;
import com.bizunited.platform.dictionary.common.vo.DictImportVo;
import com.bizunited.platform.dictionary.service.local.entity.DictImportDetailEntity;
import com.bizunited.platform.dictionary.service.local.entity.DictImportEntity;
import com.bizunited.platform.dictionary.service.local.repository.DictImportDetailRepository;
import com.bizunited.platform.dictionary.service.local.repository.DictImportRepository;
import com.bizunited.platform.rbac.server.util.SecurityUtils;
import com.bizunited.platform.user.common.service.user.UserService;
import com.bizunited.platform.user.common.vo.UserVo;
import com.bizunited.platform.venus.common.service.file.VenusFileService;
import com.bizunited.platform.venus.common.service.image.FileUpdateService;
import com.bizunited.platform.venus.common.vo.OrdinaryFileVo;
import com.google.common.collect.Lists;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.data.domain.Sort;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;
import org.springframework.web.multipart.MultipartFile;

import javax.transaction.Transactional;
import java.io.File;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Date;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.zip.ZipFile;

import static com.bizunited.platform.common.constant.MigrateDataConstants.ZIP_FILE_SUBFIX;

/**
 * 数据字典导入接口实现
 * @Author: zengxingwang
 * @Date: 2020/5/20 9:54
 */
@Service("DictImportServiceImpl")
public class DictImportServiceImpl implements DictImportService {

  private static final Logger LOGGER = LoggerFactory.getLogger(DictImportServiceImpl.class);

  @Value("${venus.file.fileRoot}")
  private String fileRoot;

  @Autowired
  private DictImportRepository dictImportRepository;
  @Autowired
  private NebulaToolkitService nebulaToolkitService;
  @Autowired
  private VenusFileService venusFileService;
  @Autowired
  private DictImportDetailRepository dictImportDetailRepository;
  @Autowired
  private DictService dictService;
  @Autowired
  private FileUpdateService fileUpdateService;
  @Autowired
  private UserService userService;

  /**
   * 上传迁入文件
   * @param file
   * @return
   */
  @Override
  @Transactional
  public DictImportVo upload(MultipartFile file) {
    Validate.notNull(file, "上传文件不能为空！");
    UserVo user = SecurityUtils.getCurrentUser();
    DictImportEntity dictImport = new DictImportEntity();
    dictImport.setCreateTime(new Date());
    dictImport.setExecuted(false);
    dictImport.setCreator(user.getAccount());
    dictImport.setOriginalFileName(file.getOriginalFilename());
    MultipartFile[] files = new MultipartFile[] {file};
    List<OrdinaryFileVo> ordinaryFileEntities = fileUpdateService.fileUpload("migrate", user.getAccount(), null, files);
    Validate.notEmpty(files, "保存上传文件失败，请检查！");
    OrdinaryFileVo fileEntity = ordinaryFileEntities.get(0);
    Validate.notNull(fileEntity,"上次文件保存返回结果不能为空！");
    dictImport.setFileName(fileEntity.getFileName());
    dictImport.setRelativeLocal(fileEntity.getRelativeLocal());
    DictImportEntity dictImportEntity = this.create(dictImport);
    return this.nebulaToolkitService.copyObjectByWhiteList(dictImportEntity, DictImportVo.class, HashSet.class, ArrayList.class, "creator", "executor");
  }

  /**
   * 创建字典导入记录
   * @param dictImport
   * @return
   */
  private DictImportEntity create(DictImportEntity dictImport) {
    Validate.notNull(dictImport.getCreateTime(), "保存记录时，导入时间不能为空！");
    Validate.notBlank(dictImport.getRelativeLocal(), "迁入文件在本地路径不能为空！");
    Validate.notBlank(dictImport.getFileName(), "迁入文件重命名后的文件名字不能为空！");
    Validate.notNull(dictImport.getCreator(), "迁入文件上传人不能为空！");
    Validate.notNull(dictImport.getCreateTime(), "迁入文件上传时间不能为空！");
    Validate.notNull(dictImport.getExecuted(), "迁入文件任务是否被执行不能为空！");
    return dictImportRepository.save(dictImport);
  }

  /**
   * 数据字典导入执行
   * @param id
   * @param mode
   * @return
   */
  @Override
  @Transactional
  public DictImportVo execute(String id, Integer mode) {
    /*
     * 1、验证入参
     * 2、获取导入的文件内容
     * 3、将文件内容存到本地的临时文件中，并加载压缩文件
     * 4、根据数据类型调用对应的导入方法
     * 5、删除临时文件
     * 6、保存明细信息并返回
     *
     * */
    // 1、
    Validate.notBlank(id, "导入记录ID不能为空");
    ImportExecuteModeEnum executeMode;
    if(mode == null) {
      executeMode = ImportExecuteModeEnum.SKIP;
    } else {
      executeMode = ImportExecuteModeEnum.valueOfMode(mode);
      Validate.notNull(executeMode, "不支持的执行模式：%s", mode);
    }
    UserVo user = SecurityUtils.getCurrentUser();
    // 2
    DictImportEntity dictImport = dictImportRepository.findDetailsById(id);
    Validate.notNull(dictImport, "未找到导入的记录");
    Validate.isTrue(!dictImport.getExecuted(), "该导入已经执行过了！！");
    byte[] bytes = venusFileService.readFileContent(dictImport.getRelativeLocal(), dictImport.getFileName());
    DictImportDetailEntity importDetail = dictImport.getDetail();
    if(importDetail == null) {
      importDetail = new DictImportDetailEntity();
    }
    Validate.isTrue(bytes != null && bytes.length > 0, "导入的文件内容为空");
    // 3
    File tmpFile = FileUtils.writeLocalFile(bytes, fileRoot, ZIP_FILE_SUBFIX);
    int executeResult = NormalStatusEnum.ENABLE.getStatus();
    try (ZipFile zipFile = new ZipFile(tmpFile)) {
      MigrateImportModel importModel = new MigrateImportModel(zipFile, executeMode);
      // 4
      try {
        dictService.importData(importModel);
        importModel.appendLine("导入完成！");
      } catch (RuntimeException e) {
        LOGGER.error(e.getMessage(), e);
        importModel.setError(true);
        importModel.append("导入错误：").appendLine(e.getMessage());
      } finally {
        importDetail.setCreateCount(importModel.getCreateCount());
        importDetail.setIgnoreCount(importModel.getSkipCount());
        importDetail.setTotalCount(importModel.getTotalCount());
        importDetail.setUpdateCount(importModel.getUpdateCount());
        importDetail.setExecuteLog(importModel.getExecuteLog().toString());
        if(importModel.isError()) {
          executeResult = NormalStatusEnum.DISABLE.getStatus();
        }
      }
    } catch (IOException e) {
      LOGGER.error(e.getMessage(), e);
      throw new IllegalArgumentException(String.format("读取压缩文件出错：%s", e.getMessage()));
    } finally {
      // 5
      tmpFile.delete();
    }
    // 6
    dictImportDetailRepository.save(importDetail);
    dictImport.setDetail(importDetail);
    dictImport.setExecuted(true);
    dictImport.setExecutor(user.getAccount());
    dictImport.setExecuteTime(new Date());
    dictImport.setExecuteResult(executeResult);
    dictImportRepository.save(dictImport);
    return nebulaToolkitService.copyObjectByWhiteList(dictImport, DictImportVo.class, HashSet.class, ArrayList.class, "detail");
  }

  /**
   * 根据ID查询详情
   * @param id
   * @return
   */
  @Override
  public DictImportVo findDetailsById(String id) {
    if(StringUtils.isBlank(id)) {
      return null;
    }
    DictImportEntity dictImport = dictImportRepository.findDetailsById(id);
    return nebulaToolkitService.copyObjectByWhiteList(dictImport, DictImportVo.class, HashSet.class, ArrayList.class, "detail");
  }

  @Override
  public List<DictImportVo> findAll() {
    List<DictImportEntity> imports = dictImportRepository.findAll(Sort.by(Sort.Order.desc("createTime")));
    if(CollectionUtils.isEmpty(imports)) {
      return Lists.newArrayList();
    }
    List<DictImportEntity> dictImportEntities = this.loadUsers(imports);
    Collection<DictImportVo> dictImportVo = this.nebulaToolkitService.copyCollectionByWhiteList(dictImportEntities, DictImportEntity.class, DictImportVo.class, HashSet.class, ArrayList.class, "detail");
    return Lists.newArrayList(dictImportVo);
  }

  /**
   * 加载用户信息
   * @param imports
   * @return
   */
  private List<DictImportEntity> loadUsers(List<DictImportEntity> imports) {
    Map<String, UserVo> tmp = new HashMap<>();
    for (DictImportEntity anImport : imports) {
      String creator = anImport.getCreator();
      UserVo createUser = tmp.get(creator);
      if(createUser == null) {
        createUser = userService.findByAccount(creator);
        tmp.put(creator, createUser);
      }
      anImport.setCreateUser(createUser);
      String executor = anImport.getExecutor();
      if(StringUtils.isNotBlank(executor)) {
        UserVo executeUser = tmp.get(executor);
        if(executeUser == null) {
          executeUser = userService.findByAccount(executor);
          tmp.put(executor, executeUser);
        }
        anImport.setExecuteUser(executeUser);
      }
    }
    return imports;
  }
}
