package com.bizunited.platform.dictionary.service.local.service.internal;

import com.bizunited.platform.common.constant.PlatformContext;
import com.bizunited.platform.common.service.NebulaToolkitService;
import com.bizunited.platform.common.util.ChineseCharUtils;
import com.bizunited.platform.dictionary.common.service.dictCategory.DictCategoryService;
import com.bizunited.platform.dictionary.common.vo.DictCategoryVo;
import com.bizunited.platform.dictionary.service.local.entity.DictCategoryEntity;
import com.bizunited.platform.dictionary.service.local.repository.DictCategoryRepository;
import com.bizunited.platform.rbac.server.util.SecurityUtils;
import com.bizunited.platform.user.common.vo.UserVo;
import com.google.common.collect.Sets;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;

import javax.transaction.Transactional;
import java.util.ArrayList;
import java.util.Date;
import java.util.HashSet;
import java.util.LinkedHashSet;
import java.util.Map;
import java.util.Optional;
import java.util.Set;

/**
 * DictCategoryServiceImpl
 *
 * @description:
 * @author: yanwe
 * @date: 11/Jun/2019 17:34
 */

@Service("DictCategoryServiceImpl")
public class DictCategoryServiceImpl implements DictCategoryService {

  @Autowired
  private DictCategoryRepository dictCategoryRepository;
  @Autowired
  private NebulaToolkitService nebulaToolkitService;
  @Autowired
  private PlatformContext platformContext;

  @Override
  @Transactional
  public DictCategoryVo create(DictCategoryVo dictCategoryVo, String parentId) {
    Validate.notNull(dictCategoryVo,"字典分类不能为空！");
    Validate.isTrue(StringUtils.isBlank(dictCategoryVo.getId()),"新增时不能填入Id!");
    Validate.notBlank(dictCategoryVo.getCateName(),"字典分类名称不能为空！");
    Validate.notBlank(dictCategoryVo.getCateCode(),"字典分类编码不能为空！");
    Validate.isTrue(!ChineseCharUtils.hasChinese(dictCategoryVo.getCateCode()),"字典分类编码不能含有中文！");
    DictCategoryEntity existCode = dictCategoryRepository.findByCateCode(dictCategoryVo.getCateCode());
    Validate.isTrue(null == existCode,"字典分类编码重复，请检查！");
    if(StringUtils.isBlank(dictCategoryVo.getCreateAccount()) || StringUtils.isBlank(dictCategoryVo.getModifyAccount())) {
      //获取当前登录账户
      UserVo user = SecurityUtils.getCurrentUser();
      dictCategoryVo.setCreateAccount(user.getAccount());
      dictCategoryVo.setModifyAccount(user.getAccount());
    }
    //其他参数
    DictCategoryEntity entity = this.nebulaToolkitService.copyObjectByWhiteList(dictCategoryVo, DictCategoryEntity.class, HashSet.class, ArrayList.class);
    entity.setCreateTime(new Date());
    entity.setModifyTime(new Date());

    // 父级
    if(StringUtils.isNotBlank(parentId)){
      Optional<DictCategoryEntity> op = dictCategoryRepository.findById(parentId);
      DictCategoryEntity parent = op.orElse(null);
      Validate.notNull(parent,"未查询到传入父类字典分类！");
      entity.setParentCategory(parent);
    }
    entity.setProjectName(platformContext.getAppName());
    DictCategoryEntity dictCategory = dictCategoryRepository.saveAndFlush(entity);
    return this.nebulaToolkitService.copyObjectByWhiteList(dictCategory, DictCategoryVo.class, HashSet.class, ArrayList.class);
  }

  @Override
  @Transactional
  public DictCategoryVo update(DictCategoryVo entity) {
    Validate.notNull(entity,"字典分类不能为空！");
    Validate.notBlank(entity.getId(),"新增时不能填入Id!");
    Optional<DictCategoryEntity> op = dictCategoryRepository.findById(entity.getId());
    DictCategoryEntity existId = op.orElse(null);
    Validate.notNull(existId,"未查询到当前字典分类！");
    Validate.notBlank(entity.getCateName(),"字典分类名称不能为空！");
    //获取当前登录账户
    UserVo user = SecurityUtils.getCurrentUser();
    //修改参数
    existId.setCateName(entity.getCateName());
    existId.setCateDesc(entity.getCateDesc());
    // 父级
    if(entity.getParentCategory() != null){
      Optional<DictCategoryEntity> opDictCategory = dictCategoryRepository.findById(entity.getParentCategory().getId());
      DictCategoryEntity parent = opDictCategory.orElse(null);
      Validate.notNull(parent,"未查询到传入父类字典分类！");
      existId.setParentCategory(parent);
    }else {
      existId.setParentCategory(null);
    }
    existId.setModifyTime(new Date());
    existId.setModifyAccount(user.getAccount());
    DictCategoryEntity dictCategory = dictCategoryRepository.save(existId);
    return this.nebulaToolkitService.copyObjectByWhiteList(dictCategory, DictCategoryVo.class, HashSet.class, ArrayList.class);
  }

  @Override
  public void bind(String childId, String parentId) {
    Validate.notBlank(childId,"子级字典分类ID不能为空！");
    Optional<DictCategoryEntity> op = dictCategoryRepository.findById(childId);
    DictCategoryEntity child = op.orElse(null);
    Validate.notNull(child,"未查询到子级字典分类！");

    child.setParentCategory(null);
    if(StringUtils.isNotEmpty(parentId)){
      Validate.notBlank(parentId,"父级字典分类ID不能为空！");
      Optional<DictCategoryEntity> optional = dictCategoryRepository.findById(parentId);
      Validate.isTrue(optional.isPresent(),"未查询到父级字典分类！");
      DictCategoryEntity parent = optional.get();
      child.setParentCategory(parent);
    }
    dictCategoryRepository.saveAndFlush(child);
  }

  @Override
  public DictCategoryVo findById(String id) {
    if(StringUtils.isBlank(id)) {
      return null;
    }
    Optional<DictCategoryEntity> op = dictCategoryRepository.findById(id);
    DictCategoryEntity dictCategory = op.orElse(null);
    return this.nebulaToolkitService.copyObjectByWhiteList(dictCategory, DictCategoryVo.class, HashSet.class, ArrayList.class);
  }

  @Override
  public DictCategoryVo findDetailsById(String id) {
    if(StringUtils.isBlank(id)) {
      return null;
    }
    DictCategoryEntity dictCategory = this.dictCategoryRepository.findDetailsById(id);
    return this.nebulaToolkitService.copyObjectByWhiteList(dictCategory, DictCategoryVo.class, HashSet.class, ArrayList.class, "parentCategory");
  }

  @Override
  public DictCategoryVo findByCode(String code) {
    if(StringUtils.isBlank(code)) {
      return null;
    }
    DictCategoryEntity dictCategory = this.dictCategoryRepository.findByCateCode(code);
    if(dictCategory == null) {
      return null;
    }
    return this.nebulaToolkitService.copyObjectByWhiteList(dictCategory, DictCategoryVo.class, HashSet.class, ArrayList.class);
  }

  /**
   * 根据分组id删除分组（不能删除有字典的分组）
   * @param id
   * @return
   */
  @Override
  @Transactional
  public void deleteById(String id) {
    Validate.notBlank(id, "字典分类id不能为空！");
    DictCategoryEntity dictCategory = dictCategoryRepository.findById(id).orElse(null);
   if(dictCategory == null){
     return;
   }
    this.isExistDicts(dictCategory);
    dictCategoryRepository.delete(dictCategory);
  }

  /**
   * 验证分组下是否存在字典
   * @param dictCategory
   */
  private void isExistDicts(DictCategoryEntity dictCategory) {
    //1.分组下有字典
    Validate.isTrue(CollectionUtils.isEmpty(dictCategory.getDicts()), "字典分组下存在字典，不能删除");
    if(CollectionUtils.isEmpty(dictCategory.getChildCategory())){
      return;
    }
    //2.分组下的子分组有字典
    for(DictCategoryEntity child : dictCategory.getChildCategory()){
      this.isExistDicts(child);
      dictCategoryRepository.delete(child);
    }
  }

  @Override
  public Set<DictCategoryVo> findAll() {
    LinkedHashSet<DictCategoryVo> dictCategoryVos = new LinkedHashSet<>();
    Set<DictCategoryEntity> dictCategorys = dictCategoryRepository.findByTreeAndProjectName(platformContext.getAppName());
    for(DictCategoryEntity dictCategory : dictCategorys){
      DictCategoryVo dictCategoryVo = this.nebulaToolkitService.copyObjectByWhiteList(dictCategory, DictCategoryVo.class, HashSet.class, ArrayList.class, "dicts", "dicts.dictItems");
      dictCategoryVo.setChildCategory(this.findChildren(dictCategory));
      dictCategoryVos.add(dictCategoryVo);
    }
    return dictCategoryVos;
  }


  /**
   * 查询子角色
   * @param dictCategory
   * @return
   */
  public Set<DictCategoryVo> findChildren(DictCategoryEntity dictCategory) {
    LinkedHashSet<DictCategoryVo> dictCategoryVos = new LinkedHashSet<>();
    //  查询子级结构
    Set<DictCategoryEntity> children = dictCategory.getChildCategory();
    if(CollectionUtils.isEmpty(children)){
      return Sets.newHashSet();
    }
    for(DictCategoryEntity child : children){
      DictCategoryVo dictCategoryVo = this.nebulaToolkitService.copyObjectByWhiteList(child, DictCategoryVo.class, HashSet.class, ArrayList.class, "dicts", "dicts.dictItems");
      dictCategoryVo.setChildCategory(this.findChildren(child));
      dictCategoryVos.add(dictCategoryVo);
    }
    return dictCategoryVos;
  }

  /**
   * 导入处理字典分类
   * @param id
   * @param dictCategoryMap
   * @return
   */
  @Override
  public String importCategory(String id, Map<String, DictCategoryVo> dictCategoryMap) {
    DictCategoryVo category = dictCategoryMap.get(id);
    String categoryId = null;
    Validate.notNull(category, "导入数据字典未找到分类信息，请检查！");
    DictCategoryEntity dictCategory = dictCategoryRepository.findByCateCode(category.getCateCode());
    if(dictCategory == null){
      category.setId(null);
      DictCategoryVo dictCategoryVo = this.create(category, null);
      categoryId = dictCategoryVo.getId();
      //处理上级
      category.setId(dictCategoryVo.getId());
      this.hanldParent(category, dictCategoryMap);
    }else {
      categoryId = dictCategory.getId();
    }
    return categoryId;
  }

  /**
   * 字典导入分组上下级关系处理
   * @param dictCategoryVo
   * @param dictCategoryMap
   */
  private void hanldParent(DictCategoryVo dictCategoryVo, Map<String, DictCategoryVo> dictCategoryMap) {
    if(dictCategoryVo.getParentCategory() == null){
      return;
    }
    String parentcategoryId = dictCategoryVo.getParentCategory().getId();
    DictCategoryVo category = dictCategoryMap.get(parentcategoryId);
    DictCategoryEntity dictCategory = dictCategoryRepository.findByCateCode(category.getCateCode());
    if(dictCategory == null){
      category.setId(null);
      DictCategoryVo dictCategoryParent = this.create(category, null);
      dictCategoryRepository.updateParentId(dictCategoryVo.getId(),dictCategoryParent.getId());
      category.setId(dictCategoryParent.getId());
      this.hanldParent(category, dictCategoryMap);
    }else {
      dictCategoryRepository.updateParentId(dictCategoryVo.getId(),dictCategory.getId());
    }
  }
}
