package com.bizunited.platform.dictionary.service.local.service.internal;

import com.bizunited.platform.common.constant.PlatformContext;
import com.bizunited.platform.common.enums.MigrateDataTypeEnum;
import com.bizunited.platform.common.service.NebulaToolkitService;
import com.bizunited.platform.common.util.ZipFileUtils;
import com.bizunited.platform.dictionary.common.service.DictExportService;
import com.bizunited.platform.dictionary.common.service.dict.DictService;
import com.bizunited.platform.dictionary.common.service.dictCategory.DictCategoryService;
import com.bizunited.platform.dictionary.common.vo.DictCategoryVo;
import com.bizunited.platform.dictionary.common.vo.DictExportVo;
import com.bizunited.platform.dictionary.common.vo.DictVo;
import com.bizunited.platform.dictionary.service.local.entity.DictExportEntity;
import com.bizunited.platform.dictionary.service.local.repository.DictExportRepository;
import com.bizunited.platform.rbac.server.util.SecurityUtils;
import com.bizunited.platform.user.common.service.user.UserService;
import com.bizunited.platform.user.common.vo.UserVo;
import com.google.common.collect.Lists;
import com.google.common.collect.Sets;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.ObjectOutputStream;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Date;
import java.util.HashMap;
import java.util.HashSet;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.zip.ZipOutputStream;
import javax.transaction.Transactional;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.context.ApplicationContext;
import org.springframework.data.domain.Sort;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;

import static com.bizunited.platform.common.constant.MigrateDataConstants.DICT_FILENAME;

/**
 * 数据字典导出接口实现
 * @Author: zengxingwang
 * @Date: 2020/5/20 21:23
 */
@Service("DictExportServiceImpl")
public class DictExportServiceImpl implements DictExportService {

  @Value("${venus.file.fileRoot}")
  private String fileRoot;

  @Autowired
  private DictService dictService;
  @Autowired
  private NebulaToolkitService nebulaToolkitService;
  @Autowired
  private DictCategoryService dictCategoryService;
  @Autowired
  private ApplicationContext applicationContext;
  @Autowired
  private DictExportRepository dictExportRepository;
  @Autowired
  private UserService userService;
  @Autowired
  private PlatformContext platformContext;

  @Override
  @Transactional
  public DictExportVo save(DictExportVo export) {
    UserVo user = SecurityUtils.getCurrentUser();
    export.setCreateTime(new Date());
    export.setCreateUser(user);
    export.setCreator(user.getAccount());
    DictExportEntity dictExportEntity = nebulaToolkitService.copyObjectByWhiteList(export, DictExportEntity.class, HashSet.class, ArrayList.class);
    dictExportEntity.setProjectName(platformContext.getAppName());
    dictExportRepository.save(dictExportEntity);
    export.setId(dictExportEntity.getId());
    return export;
  }

  @Override
  public List<DictExportVo> findAll() {
    List<DictExportEntity> exports = dictExportRepository.findAll(Sort.by(Sort.Order.desc("createTime")));
    if(CollectionUtils.isEmpty(exports)) {
      return Lists.newArrayList();
    }
    Map<String, UserVo> userCache = new HashMap<>();
    for (DictExportEntity export : exports) {
      String creator = export.getCreator();
      UserVo user = userCache.get(creator);
      if(user == null) {
        user = userService.findByAccount(creator);
        userCache.put(creator, user);
      }
      export.setCreateUser(user);
    }
    Collection<DictExportVo> dictExportVos = nebulaToolkitService.copyCollectionByWhiteList(exports, DictExportEntity.class, DictExportVo.class, HashSet.class, ArrayList.class, "createUser");
    return Lists.newArrayList(dictExportVos);
  }

  /**
   * 该私有方法用于对指定的数据字典进行迁出
   * @param dictIds
   */
  @Override
  public byte[] exportDict(String[] dictIds) {
    List<DictVo> dicts;
    if(dictIds == null || dictIds.length == 0) {
      dicts = dictService.findAll();
    } else {
      int count = dictService.countByIds(dictIds);
      Validate.isTrue(count == dictIds.length, "指定导出的编码规则信息与数据库存储的信息不符，请检查");
      Set<DictVo> detailsByIds = dictService.findDetailsByIds(dictIds);
      dicts = Lists.newArrayList(detailsByIds);
    }
    if(CollectionUtils.isEmpty(dicts)) {
      return new byte[0];
    }

    Set<DictCategoryVo> allJpaLeafDictCategorys = Sets.newHashSet();
    for (DictVo dictVo : dicts) {
      allJpaLeafDictCategorys.add(dictVo.getCategory());
    }
    Collection<DictCategoryVo> allLeafDictCategorys = this.nebulaToolkitService.copyCollectionByWhiteList(allJpaLeafDictCategorys, DictCategoryVo.class, DictCategoryVo.class, HashSet.class, ArrayList.class);
    /*
     * 3、========
     * .那么我们最终要形成的链条是
     * .第一个字典分类的根节点<->第一个字典分类的下层父节<->…………<->第一个字典分类
     * <-> 第二个字典分类的最顶层父节点（且没有在之前出现过）<->…………<->第二个字典分类
     * <-> ………………………… <-> 最后一个字典分类
     * ,另外注意：这里每做一次递归，都会进行一次中拷，性能比较差，但是迁移功能使用度很低，所以性能问题的优先级不高
     * */
    LinkedList<DictCategoryVo> allDictCategorys = Lists.newLinkedList();
    for (DictCategoryVo leafDictCategoryItem : allLeafDictCategorys) {
      LinkedList<DictCategoryVo> stackDictCategorys = Lists.newLinkedList();
      String leafDictId = leafDictCategoryItem.getId();
      DictCategoryVo current = this.dictCategoryService.findDetailsById(leafDictId);
      DictCategoryVo currentEntity = this.nebulaToolkitService.copyObjectByWhiteList(current, DictCategoryVo.class, HashSet.class, ArrayList.class, "parentCategory");
      // 添加当前叶子节点（为什么不适用leafDictCategoryItem直接添加呢？因为里面没有父级节点的信息）
      stackDictCategorys.addFirst(currentEntity);
      DictCategoryVo parentDictCategory = currentEntity.getParentCategory();
      if(parentDictCategory == null) {
        allDictCategorys.addAll(stackDictCategorys);
        continue;
      }

      // 如果条件成立，说明要进行记录，记录是按照栈的特性进行
      final String parentDictCategoryIdf = parentDictCategory.getId();
      while(parentDictCategory != null && !allDictCategorys.stream().filter(item -> StringUtils.equals(item.getId(), parentDictCategoryIdf)).findAny().isPresent()) {
        final String parentDictCategoryId = parentDictCategory.getId();
        // 为什么还要查询一次呢？因为里面可能没有父级节点的信息
        DictCategoryVo parentDictCategoryVo = this.dictCategoryService.findDetailsById(parentDictCategoryId);
        parentDictCategory = this.nebulaToolkitService.copyObjectByWhiteList(parentDictCategoryVo, DictCategoryVo.class, HashSet.class, ArrayList.class, "parentCategory");
        stackDictCategorys.addFirst(parentDictCategory);
        parentDictCategory = parentDictCategory.getParentCategory();
      }
      // 将本次循环得到的各级字典分类节点放入结果集合，
      allDictCategorys.addAll(stackDictCategorys);
    }
    // 接着将所有数据脱离JPA对象特性
    Collection<DictVo> allDicts = this.nebulaToolkitService.copyCollectionByWhiteList(dicts, DictVo.class, DictVo.class, HashSet.class, ArrayList.class, "category" , "dictItems");

    try (ByteArrayOutputStream fullZipBis = new ByteArrayOutputStream();
         ZipOutputStream zipf = new ZipOutputStream(fullZipBis);) {
      try(ByteArrayOutputStream bos = new ByteArrayOutputStream();
          ObjectOutputStream oos = new ObjectOutputStream(bos);) {
        oos.writeObject(allDictCategorys);
        oos.writeObject(allDicts);
        ZipFileUtils.writeZipFile(zipf, DICT_FILENAME, bos.toByteArray());
      }
      zipf.finish();
      this.save(MigrateDataTypeEnum.DICT.getType(), StringUtils.join(dictIds, ","));
      return fullZipBis.toByteArray();
    } catch(IOException e) {
      throw new IllegalArgumentException(e.getMessage());
    }
  }

  /**
   * 保存导出日志
   * @param dataType
   * @param datas
   * @return
   */
  private DictExportEntity save(Integer dataType, String datas) {
    Validate.notNull(dataType, "数据类型不能为空");
    DictExportEntity export = new DictExportEntity();
    export.setDatas(datas);
    DictExportService dictExportService = applicationContext.getBean(DictExportService.class);
    DictExportVo dictExportVo = nebulaToolkitService.copyObjectByWhiteList(export, DictExportVo.class, HashSet.class, ArrayList.class);
    DictExportVo save = dictExportService.save(dictExportVo);
    export.setId(save.getId());
    return export;
  }
}
