package com.bizunited.platform.dictionary.service.local.service.internal;

import com.bizunited.platform.common.constant.PlatformContext;
import com.bizunited.platform.common.enums.ImportExecuteModeEnum;
import com.bizunited.platform.common.model.MigrateImportModel;
import com.bizunited.platform.common.service.NebulaToolkitService;
import com.bizunited.platform.dictionary.common.service.dict.DictService;
import com.bizunited.platform.dictionary.common.service.dictCategory.DictCategoryService;
import com.bizunited.platform.dictionary.common.service.dictItem.DictItemService;
import com.bizunited.platform.dictionary.common.vo.DictCategoryVo;
import com.bizunited.platform.dictionary.common.vo.DictItemVo;
import com.bizunited.platform.dictionary.common.vo.DictVo;
import com.bizunited.platform.dictionary.service.local.entity.DictCategoryEntity;
import com.bizunited.platform.dictionary.service.local.entity.DictEntity;
import com.bizunited.platform.dictionary.service.local.entity.DictItemEntity;
import com.bizunited.platform.dictionary.service.local.repository.DictItemRepository;
import com.bizunited.platform.dictionary.service.local.repository.DictRepository;
import com.google.common.collect.Sets;
import java.io.IOException;
import java.io.InputStream;
import java.io.ObjectInputStream;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Date;
import java.util.HashMap;
import java.util.HashSet;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.stream.Collectors;
import java.util.zip.ZipEntry;
import java.util.zip.ZipFile;
import javax.transaction.Transactional;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.PageImpl;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Pageable;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;

import static com.bizunited.platform.common.constant.MigrateDataConstants.DICT_FILENAME;
import static javax.transaction.Transactional.TxType.REQUIRES_NEW;

/**
 * DictServiceImpl
 *
 * @description:
 * @author: yanwe
 * @date: 11/Feb/2019 10:33
 */
@Service("dictServiceImpl")
public class DictServiceImpl implements DictService {
  private static final Logger LOGGER = LoggerFactory.getLogger(DictServiceImpl.class);
  private static final String ERROR_CODE_MESS = "查询字典编码不能为空！";
  private static final String ERROR_MESS = "未找到该编码匹配字典！";
  @Autowired 
  private DictRepository dictRepository;
  @Autowired 
  private DictItemRepository dictItemRepository;
  @Autowired 
  private DictCategoryService dictCategoryService;
  @Autowired
  private DictItemService dictItemService;
  @Autowired
  private NebulaToolkitService nebulaToolkitService;
  @Autowired
  private PlatformContext platformContext;

  @Override
  public List<DictVo> findAll() {
    List<DictEntity> dictEntities = dictRepository.findAll();
    if(!CollectionUtils.isEmpty(dictEntities)){
      dictEntities = dictEntities.stream().filter(dictEntity -> StringUtils.equals(platformContext.getAppName(), dictEntity.getProjectName())).collect(Collectors.toList());
    }
    Collection<DictVo> dictVos = this.nebulaToolkitService.copyCollectionByWhiteList(dictEntities, DictEntity.class, DictVo.class, LinkedHashSet.class, ArrayList.class);
    return new ArrayList<>(dictVos);
  }

  @Override
  public List<DictVo> findByStatus(Boolean dictStatus) {
    List<DictEntity> dictEntities = dictRepository.findByDictStatusAndProjectName(dictStatus, platformContext.getAppName());
    Collection<DictVo> dictVos = this.nebulaToolkitService.copyCollectionByWhiteList(dictEntities, DictEntity.class, DictVo.class, LinkedHashSet.class, ArrayList.class);
    return new ArrayList<>(dictVos);
  }

  @Override
  public Page<DictVo> findByConditions(
      String dictCode, String dictTitle, String dictType, Boolean dictStatus, Pageable pageable) {
    Map<String, Object> conditions = new HashMap<>();
    if (StringUtils.isNotBlank(dictCode)) {
      conditions.put("dictCode", dictCode);
    }
    if (StringUtils.isNotBlank(dictTitle)) {
      conditions.put("dictTitle", dictTitle);
    }
    if (StringUtils.isNotBlank(dictType)) {
      conditions.put("dictType", dictType);
    }
    // 如果为空则默认查询上架状态的字典
    if (dictStatus != null) {
      conditions.put("dictStatus", dictStatus);
    }

    if (pageable == null) {
      pageable = PageRequest.of(0, 50);
    }
    Page<DictEntity> page = dictRepository.queryPage(pageable, conditions);
    Collection<DictVo> dictVos = this.nebulaToolkitService.copyCollectionByWhiteList(page.getContent(), DictEntity.class, DictVo.class, LinkedHashSet.class, ArrayList.class);
    return new PageImpl<DictVo>(new ArrayList<>(dictVos), page.getPageable(), page.getTotalElements());
  }

  @Override
  public DictVo findDetailsByCodeAndStatus(String dictCode, Boolean showStatus) {
    if(StringUtils.isBlank(dictCode)){
      return null;
    }
    DictEntity entity = dictRepository.findDetailsByCodeAndStatus(dictCode,showStatus);
    if(entity == null){
      return null;
    }
    return this.nebulaToolkitService.copyObjectByWhiteList(entity, DictVo.class, HashSet.class, ArrayList.class, "category", "dictItems");
  }

  @Override
  public Set<DictVo> findDetailsByIds(String[] dictIds) {
    if(dictIds == null || dictIds.length == 0) {
      return Sets.newHashSet();
    }

    Set<DictEntity> dictEntities = this.dictRepository.findDetailsByIds(dictIds);
    Collection<DictVo> dictVos = this.nebulaToolkitService.copyCollectionByWhiteList(dictEntities, DictEntity.class, DictVo.class, LinkedHashSet.class, ArrayList.class, "category" , "dictItems");
    return Sets.newLinkedHashSet(dictVos);
  }

  @Override
  public int countByIds(String[] ids) {
    if(ids == null || ids.length == 0) {
      return 0;
    }
    return this.dictRepository.countByIds(ids);
  }

  @Override
  @Transactional
  public DictVo create(DictVo dictVo, String categoryId) {
    this.validEntity(dictVo, categoryId);
    DictCategoryVo categoryVo = dictCategoryService.findById(categoryId);
    // 判断CODE是否有重复
    DictEntity existDict = dictRepository.findByDictCode(dictVo.getDictCode());
    Validate.isTrue(null == existDict, "该编码已有重复！");
    DictEntity dictEntity = this.nebulaToolkitService.copyObjectByWhiteList(dictVo, DictEntity.class, HashSet.class, ArrayList.class);

    // 新增字典默认明细项数量为0
    dictEntity.setItemNum(0);
    // 默认启用状态为true
    dictEntity.setDictStatus(true);
    DictCategoryEntity categoryEntity = this.nebulaToolkitService.copyObjectByWhiteList(categoryVo, DictCategoryEntity.class, HashSet.class, ArrayList.class);
    dictEntity.setCategory(categoryEntity);
    dictEntity.setCreateTime(new Date());
    dictEntity.setModifyTime(new Date());
    dictEntity.setProjectName(platformContext.getAppName());
    DictEntity dict = dictRepository.saveAndFlush(dictEntity);
    //新增明细项
    dictItemService.save(dictEntity.getDictCode(), dictVo.getDictItems());
    return this.nebulaToolkitService.copyObjectByWhiteList(dict, DictVo.class, HashSet.class, ArrayList.class, "dictItems");
  }

  @Override
  @Transactional
  public DictVo update(DictVo dictVo, String categoryId) {
    this.validEntity(dictVo, categoryId);
    DictCategoryVo categoryVo = dictCategoryService.findById(categoryId);
    Optional<DictEntity> op = dictRepository.findById(dictVo.getId());
    DictEntity existDict = op.orElse(null);
    Validate.notNull(existDict, "该字典不存在，请检查！");

    Validate.notNull(existDict, "未找到改修改字典");
    existDict.setDescription(dictVo.getDescription());
    existDict.setDictType(dictVo.getDictType());
    existDict.setDictTitle(dictVo.getDictTitle());
    DictCategoryEntity categoryEntity = this.nebulaToolkitService.copyObjectByWhiteList(categoryVo, DictCategoryEntity.class, HashSet.class, ArrayList.class);
    existDict.setCategory(categoryEntity);
    existDict.setModifyTime(new Date());
    existDict.setValueType(dictVo.getValueType());
    //更新明细项
    Set<DictItemVo> dictItems = dictVo.getDictItems();
    dictItemService.save(dictVo.getDictCode(), dictItems);
    DictEntity entity = dictRepository.saveAndFlush(existDict);
    return this.nebulaToolkitService.copyObjectByWhiteList(entity, DictVo.class, HashSet.class, ArrayList.class);
  }

  @Override
  @Transactional
  public DictVo upgrade(String newDictCode, String dictCode) {
    Validate.notBlank(dictCode, ERROR_CODE_MESS);
    DictEntity dictEntity = dictRepository.findByDictCode(dictCode);
    Validate.notNull(dictEntity, ERROR_MESS);
    // 复制字典
    DictEntity newDict = new DictEntity();
    newDict.setDictCode(newDictCode);
    newDict.setDictTitle(dictEntity.getDictTitle());
    newDict.setItemNum(dictEntity.getItemNum());
    newDict.setDescription(dictEntity.getDescription());
    newDict.setDictType(dictEntity.getDictType());
    newDict.setDictStatus(false);
    newDict.setDictItems(new HashSet<>());
    newDict.setCategory(dictEntity.getCategory());
    newDict.setCreateTime(new Date());
    newDict.setModifyTime(new Date());
    newDict.setProjectName(platformContext.getAppName());
    newDict = dictRepository.saveAndFlush(newDict);
    // 复制字典明细
    Set<DictItemEntity> newItems = new HashSet<>();
    for (DictItemEntity item : dictEntity.getDictItems()) {
      DictItemEntity newItem = new DictItemEntity();
      newItem.setDictSort(item.getDictSort());
      newItem.setDictKey(item.getDictKey());
      newItem.setDictValue(item.getDictValue());
      newItem.setDictEntity(newDict);
      newItem.setCreateTime(new Date());
      newItem.setProjectName(platformContext.getAppName());
      newItems.add(newItem);
    }
    dictItemRepository.saveAll(newItems);
    return this.nebulaToolkitService.copyObjectByWhiteList(newDict, DictVo.class, HashSet.class, ArrayList.class);
  }

  @Override
  @Transactional
  public DictVo onshelf(String dictCode) {
    Validate.notBlank(dictCode, ERROR_CODE_MESS);
    DictEntity dictEntity = dictRepository.findByDictCode(dictCode);
    Validate.notNull(dictEntity, ERROR_MESS);
    dictEntity.setDictStatus(true);
    DictEntity entity = dictRepository.saveAndFlush(dictEntity);
    return this.nebulaToolkitService.copyObjectByWhiteList(entity, DictVo.class, HashSet.class, ArrayList.class);
  }

  @Override
  @Transactional
  public void delete(String dictCode) {
    Validate.notBlank(dictCode, ERROR_CODE_MESS);
    DictEntity dictEntity = dictRepository.findByDictCode(dictCode);
    Validate.notNull(dictEntity, ERROR_MESS);
    Validate.isTrue(!dictEntity.getDictStatus(), "已上架的字典无法删除！");
    Set<DictItemEntity> dictItemEntities = dictEntity.getDictItems();
    dictItemRepository.deleteAll(dictItemEntities);
    dictRepository.delete(dictEntity);
  }

  /**
   * 根据编码查询
   * @param dictCode
   * @return
   */
  @Override
  public DictVo findByDictCode(String dictCode) {
    if(StringUtils.isBlank(dictCode)){
      return null;
    }
    DictEntity entity = dictRepository.findByDictCode(dictCode);
    if(entity == null) {
      return null;
    }
    return this.nebulaToolkitService.copyObjectByWhiteList(entity, DictVo.class, HashSet.class, ArrayList.class,"category");
  }

  /**
   * 根据编码查询（包括关联信息）
   * @param dictCode
   * @return
   */
  @Override
  public DictVo findDetailsByDictCode(String dictCode) {
    if(StringUtils.isBlank(dictCode)){
      return null;
    }
    DictEntity entity = dictRepository.findDetailsByDictCode(dictCode);
    if(entity == null) {
      return null;
    }
    return this.nebulaToolkitService.copyObjectByWhiteList(entity, DictVo.class, LinkedHashSet.class, ArrayList.class, "category", "dictItems");
  }

  /**
   * 批量查询（包括关联信息）
   * @param dictCodes
   * @return
   */
  @Override
  public Set<DictVo> findDetailsByDictCodes(String[] dictCodes) {
    if(dictCodes == null || dictCodes.length == 0) {
      return null;
    }
    Set<DictEntity> entities = dictRepository.findDetailsByDictCodes(dictCodes);
    Collection<DictVo> dictVos = this.nebulaToolkitService.copyCollectionByWhiteList(entities, DictEntity.class, DictVo.class, HashSet.class, ArrayList.class, "category", "dictItems");
    return Sets.newLinkedHashSet(dictVos);
  }


  /**
   * 验证实体
   *
   * @param dictVo
   * @param categoryId
   */
  private void validEntity(DictVo dictVo, String categoryId) {
    Validate.notNull(dictVo, "输入字典不可为空!");
    // 判断该字典是否已经是启用状态，若为启用，则不可再编辑
    Validate.notBlank(dictVo.getDictCode(), "dict code 不能为空!");
    Validate.notBlank(dictVo.getDictTitle(), "dict title 不能为空!");
    Validate.notBlank(categoryId, "字典分组ID不能为空");
    Validate.notBlank(dictVo.getValueType(), "字典值类型不能为空");
    DictCategoryVo categoryVo = dictCategoryService.findById(categoryId);
    Validate.notNull(categoryVo, "未查询到字典分组");
  }

  /**
   * 数据字典导入执行
   * @param importModel
   */
  @SuppressWarnings("unchecked")
  @Override
  @Transactional(REQUIRES_NEW)
  public void importData(MigrateImportModel importModel) {
    Validate.notNull(importModel, "导入信息不能为空");
    ZipFile zipFile = importModel.getZipFile();
    Validate.notNull(zipFile, "导入文件不能为空");
    Validate.notNull(importModel.getExecuteMode(), "执行模式不能为空");
    importModel.appendLine("开始导入数据");
    ZipEntry dataViewEntry = zipFile.getEntry(DICT_FILENAME);
    if(dataViewEntry == null) {
      importModel.appendLine("导入压缩包中未发现数据文件，请检查");
    }
    if(dataViewEntry != null) {
      try (InputStream is = zipFile.getInputStream(dataViewEntry);
           ObjectInputStream ois = new ObjectInputStream(is)) {
        List<DictCategoryVo> dictCategorys = (List<DictCategoryVo>) ois.readObject();
        List<DictVo> dicts = (List<DictVo>) ois.readObject();
        if(CollectionUtils.isEmpty(dicts)) {
          importModel.appendLine("导入的数据为空");
        } else {
          this.importData(dicts, dictCategorys, importModel);
        }
      } catch (IOException | ClassNotFoundException e) {
        LOGGER.error(e.getMessage(), e);
        importModel.append("读取业务数据失败：").appendLine(e.getMessage());
      }
    }
  }

  /**
   * 字典导入-总数与顺序处理
   * @param dicts
   * @param dictCategorys
   * @param importModel
   */
  private void importData(List<DictVo> dicts, List<DictCategoryVo> dictCategorys, MigrateImportModel importModel) {
    importModel.setTotalCount(dicts.size());
    Map<String, DictCategoryVo> dictCategoryMap = this.getDictCategoryMapById(dictCategorys);
    for (int i = 0; i < dicts.size(); i++) {
      DictVo dict = dicts.get(i);
      importModel.appendLine(StringUtils.join("--------[", i + 1, "]----------"));
      this.importData(dict, importModel, dictCategoryMap);
    }
  }

  /**
   * 把字典的分组存入map中
   * @param dictCategorys
   * @return
   */
  private Map<String, DictCategoryVo> getDictCategoryMapById(List<DictCategoryVo> dictCategorys) {
    Map<String, DictCategoryVo> dictCategoryMap = new HashMap<>();
    for(DictCategoryVo dictCategoryVo : dictCategorys){
      dictCategoryMap.put(dictCategoryVo.getId(), dictCategoryVo);
    }
    return dictCategoryMap;
  }

  /**
   * 导入字典-开始处理数据
   * @param dict
   * @param importModel
   * @param dictCategoryMap
   */
  private void importData(DictVo dict, MigrateImportModel importModel,  Map<String, DictCategoryVo> dictCategoryMap) {
    importModel.append("开始处理数据：").appendLine(dict.getDictTitle());
    ImportExecuteModeEnum executeMode = importModel.getExecuteMode();
    DictEntity dbDict = dictRepository.findByDictCode(dict.getDictCode());
    this.handleImportDataId(dict);
    if(dbDict != null && ImportExecuteModeEnum.SKIP == executeMode) {
      importModel.appendLine("数据字典已存在，跳过");
      importModel.addSkipCount();
      return;
    }
    if(dbDict != null && ImportExecuteModeEnum.UPDATE == executeMode) {
      importModel.appendLine("数据字典已存在，进行更新覆盖");
      this.handleUpdateData(dict, dbDict, importModel);
      return;
    }
    if(dbDict == null) {
      this.handleCreateData(dict, dbDict, importModel, dictCategoryMap);
      return;
    }
    importModel.appendLine("暂不支持的执行方式");
    importModel.addSkipCount();

  }

  /**
   * 导入字典的新增处理
   * @param dict
   * @param dbDict
   * @param importModel
   * @param dictCategoryMap
   */
  private void handleCreateData(DictVo dict, DictEntity dbDict, MigrateImportModel importModel, Map<String, DictCategoryVo> dictCategoryMap) {
    importModel.appendLine("开始新增数据字典");
    importModel.appendLine("导入数据字典");
    //处理分类
    String ategoryId = dictCategoryService.importCategory(dict.getCategory().getId(), dictCategoryMap);
    this.create(dict, ategoryId);
    importModel.addCreateCount();
  }

  /**
   * 导入字典的更新处理
   * @param dict
   * @param dbDict
   * @param importModel
   */
  private void handleUpdateData(DictVo dict, DictEntity dbDict, MigrateImportModel importModel) {
    DictVo dictVo = nebulaToolkitService.copyObjectByWhiteList(dbDict, DictVo.class, HashSet.class, ArrayList.class);
    dictVo.setDictTitle(dict.getDictTitle());
    dictVo.setDescription(dict.getDescription());
    dictVo.setDictType(dict.getDictType());
    dictVo.setValueType(dict.getValueType());
    dictVo.setDictItems(dict.getDictItems());
    this.update(dictVo, dbDict.getCategory().getId());
    importModel.appendLine("更新成功");
    importModel.addUpdateCount();
  }

  /**
   * 去掉导入数据字典和字典项的id
   * @param dict
   */
  private void handleImportDataId(DictVo dict) {
    dict.setId(null);
    Set<DictItemVo> dictItems = dict.getDictItems();
    if(!CollectionUtils.isEmpty(dictItems)) {
      dictItems.forEach(f -> f.setId(null));
    }
  }

}
