package com.bizunited.platform.dictionary.service.local.service.internal;

import com.bizunited.platform.common.constant.PlatformContext;
import com.bizunited.platform.common.service.NebulaToolkitService;
import com.bizunited.platform.dictionary.common.service.dict.DictService;
import com.bizunited.platform.dictionary.common.service.dictItem.DictItemService;
import com.bizunited.platform.dictionary.common.vo.DictItemVo;
import com.bizunited.platform.dictionary.common.vo.DictVo;
import com.bizunited.platform.dictionary.service.local.entity.DictEntity;
import com.bizunited.platform.dictionary.service.local.entity.DictItemEntity;
import com.bizunited.platform.dictionary.service.local.repository.DictItemRepository;
import com.bizunited.platform.dictionary.service.local.repository.DictRepository;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.google.common.collect.Sets;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Date;
import java.util.HashMap;
import java.util.HashSet;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.function.Function;
import java.util.stream.Collectors;
import javax.transaction.Transactional;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;

@Service("DictItemServiceImpl")
public class DictItemServiceImpl implements DictItemService {

  private static final String ERROR_MESS = "字典明细ID不能为空！";
  @Autowired 
  private DictRepository dictRepository;
  @Autowired 
  private DictItemRepository dictItemRepository;
  @Autowired
  private DictService dictService;
  @Autowired
  private NebulaToolkitService nebulaToolkitService;
  @Autowired
  private PlatformContext platformContext;

  private static final String ADD = "add";
  private static final String UPDATE = "update";

  @Override
  public List<DictItemVo> findItemsByCode(String dictCode) {
    DictEntity dictEntity = dictRepository.findByDictCode(dictCode);
    if (dictEntity == null) {
      return Lists.newArrayList();
    }
    List<DictItemEntity> dictItems = dictItemRepository.findByDictEntityOrderByDictSort(dictEntity);
    Collection<DictItemVo> dictItemVos = this.nebulaToolkitService.copyCollectionByWhiteList(dictItems, DictItemEntity.class, DictItemVo.class, LinkedHashSet.class, ArrayList.class);
    return Lists.newArrayList(dictItemVos);
  }

  /**
   * 明细的更新目前只涉及新增和更新，删除暂不开发
   */
  @Override
  @Transactional
  public Set<DictItemVo> save(String dictCode, Set<DictItemVo> items) {
    if(CollectionUtils.isEmpty(items)){
      return Sets.newHashSet();
    }
    Validate.notBlank(dictCode,"字典编码不能为空");
    DictVo dictVo = dictService.findDetailsByDictCode(dictCode);
    Validate.notNull(dictVo, "未找到该编码匹配字典！");
    DictEntity dbDictEntity = this.nebulaToolkitService.copyObjectByWhiteList(dictVo, DictEntity.class, HashSet.class, ArrayList.class);
    List<DictItemEntity> dbDictItemsList = dictItemRepository.findByDictEntity(dbDictEntity);
    //校验分析（区分新增和更新，且验重）
    Map<String,Set<DictItemVo>> datas = this.analysisAndValidation(dbDictEntity, items, dbDictItemsList);
    Collection<DictItemEntity> needAdds = nebulaToolkitService.copyCollectionByWhiteList(datas.get(ADD),DictItemVo.class,DictItemEntity.class,LinkedHashSet.class,ArrayList.class);
    Collection<DictItemEntity> needUpdates = nebulaToolkitService.copyCollectionByWhiteList(datas.get(UPDATE),DictItemVo.class,DictItemEntity.class,LinkedHashSet.class,ArrayList.class);

    if(!CollectionUtils.isEmpty(needAdds)){
      Date now = new Date();
      needAdds.forEach(dictItem -> {
        dictItem.setId(null);
        dictItem.setCreateTime(now);
        dictItem.setDictSort(dictItem.getDictSort() == null ? 1 : dictItem.getDictSort());
        dictItem.setDictEntity(dbDictEntity);
        dictItem.setDictItemStatus(dictItem.getDictItemStatus() == null ? Boolean.TRUE : dictItem.getDictItemStatus());
        dictItem.setProjectName(platformContext.getAppName());
      });
      dictItemRepository.saveAll(needAdds);
    }
    if(!CollectionUtils.isEmpty(needUpdates)){
      //如果字典名称相互置换，需要先删除后新增。所以此处一律先删除，后新增
      //注意：该更新逻辑其实是：先删除后新增
      this.deleteByIds(needUpdates.stream().map(DictItemEntity::getId).collect(Collectors.toSet()));
      needUpdates.forEach(dictItem -> {
        dictItem.setId(null);
        dictItem.setDictSort(dictItem.getDictSort() == null ? 1 : dictItem.getDictSort());
        dictItem.setDictEntity(dbDictEntity);
        dictItem.setDictItemStatus(dictItem.getDictItemStatus() == null ? Boolean.TRUE : dictItem.getDictItemStatus());
        dictItem.setProjectName(platformContext.getAppName());
      });
      dictItemRepository.saveAll(needUpdates);
    }

    Set<DictItemEntity> unions = Sets.union(Sets.newHashSet(needAdds),Sets.newHashSet(needUpdates));
    Collection<DictItemVo> dictItemVos = this.nebulaToolkitService.copyCollectionByWhiteList(unions, DictItemEntity.class, DictItemVo.class, LinkedHashSet.class, ArrayList.class);
    return Sets.newLinkedHashSet(dictItemVos);
  }

  @Override
  @Transactional
  public DictItemVo create(String dictCode, DictItemVo dictItem) {
    //参数校验
    this.createValidation(dictCode, dictItem);

    DictVo dict = dictService.findByDictCode(dictCode);
    dict.setItemNum(dict.getItemNum() + 1);
    DictEntity dictEntity = this.nebulaToolkitService.copyObjectByWhiteList(dict, DictEntity.class, HashSet.class, ArrayList.class, "category");
    dictEntity.setProjectName(platformContext.getAppName());
    dictRepository.save(dictEntity);

    DictItemEntity dictItemEntity = this.nebulaToolkitService.copyObjectByWhiteList(dictItem, DictItemEntity.class, HashSet.class, ArrayList.class);
    dictItemEntity.setCreateTime(new Date());
    dictItemEntity.setDictSort(dictItem.getDictSort() == null ? 1 : dictItem.getDictSort());
    dictItemEntity.setDictEntity(dictEntity);
    dictItemEntity.setDictItemStatus(dictItem.getDictItemStatus() == null ? Boolean.TRUE : dictItem.getDictItemStatus());
    dictItemEntity.setProjectName(platformContext.getAppName());
    DictItemEntity entity = dictItemRepository.save(dictItemEntity);
    return this.nebulaToolkitService.copyObjectByWhiteList(entity, DictItemVo.class, HashSet.class, ArrayList.class);
  }

  @Override
  public DictItemVo findDetailsById(String id) {
    Validate.notNull(id, "查询字典值明细参数为空，请检查!");
    DictItemEntity entity = dictItemRepository.findDetailById(id);
    return this.nebulaToolkitService.copyObjectByWhiteList(entity, DictItemVo.class, HashSet.class, ArrayList.class,  "dict");
  }

  /**
   * 根据字典dictCode和字典值的key查询一个数据字典值详细
   *
   * @param dictCode
   * @param key
   * @return
   */
  @Override
  public DictItemVo findByKeyAndDictCode(String dictCode, String key) {
    if (StringUtils.isBlank(dictCode) || StringUtils.isBlank(key)){
      return null;
    }
    DictItemEntity dictItem = this.dictItemRepository.findByKeyAndDictCode(dictCode, key);
    if (dictItem == null){
      return null;
    }
    return this.nebulaToolkitService.copyObjectByWhiteList(dictItem, DictItemVo.class, HashSet.class, ArrayList.class,  "dictEntity");
  }

  @Override
  @Transactional
  public void deleteByIds(Set<String> ids) {
    if(CollectionUtils.isEmpty(ids)){
      return;
    }
    List<DictItemEntity> existItems = dictItemRepository.findAllById(ids);
    if(!CollectionUtils.isEmpty(existItems)){
      dictItemRepository.deleteInBatch(existItems);
    }
  }

  /**
   * 新增-参数校验
   * @param dictCode
   * @param dictItem
   */
  private void createValidation(String dictCode, DictItemVo dictItem) {
    Validate.notBlank(dictCode, "字典编码不能为空，请检查");
    DictVo dict = dictService.findByDictCode(dictCode);
    Validate.notNull(dict, "未找到该编码匹配字典！");

    Validate.notNull(dictItem, "保存对象不存在!");
    Validate.isTrue(dictItem.getId() == null, "保存数据ID必须为空!");
    Validate.notBlank(dictItem.getDictKey(), "字典名称不能为空，请检查!");
    Validate.notBlank(dictItem.getDictValue(), "字典值不能为空，请检查!");

    long countByDictKey = dictItemRepository.countByDictKeyAndDictId(dictItem.getDictKey(), dict.getId());
    Validate.isTrue(countByDictKey == 0L, "字典名称重复，请检查!");
    long countByDictValue = dictItemRepository.countByDictValueAndDictId(dictItem.getDictValue(), dict.getId());
    Validate.isTrue(countByDictValue == 0L, "字典值重复，请检查!");
  }

  @Override
  @Transactional
  public DictItemVo enable(String dictItemId) {
    Validate.notBlank(dictItemId, ERROR_MESS);
    DictItemEntity dictItemEntity = dictItemRepository.findDetailById(dictItemId);
    Validate.notNull(dictItemEntity, ERROR_MESS);
    dictItemEntity.setDictItemStatus(true);
    DictItemEntity dictItem = dictItemRepository.save(dictItemEntity);
    return this.nebulaToolkitService.copyObjectByWhiteList(dictItem, DictItemVo.class, HashSet.class, ArrayList.class);
  }

  @Override
  @Transactional
  public DictItemVo disable(String dictItemId) {
    Validate.notBlank(dictItemId, ERROR_MESS);
    DictItemEntity dictItemEntity = dictItemRepository.findDetailById(dictItemId);
    Validate.notNull(dictItemEntity, ERROR_MESS);
    dictItemEntity.setDictItemStatus(false);
    DictItemEntity dictItem = dictItemRepository.save(dictItemEntity);
    return this.nebulaToolkitService.copyObjectByWhiteList(dictItem, DictItemVo.class, HashSet.class, ArrayList.class);
  }


  /**
   * 参数校验
   * 重复性验证：value只需要验证传递过来的是否重复，key还需要验证与数据库里是否重复
   * @param dbDictEntity 所对应的字典
   * @param items 前端传递过来的
   * @param dbDictItemsList 数据库里原有的
   */
  private Map<String,Set<DictItemVo>> analysisAndValidation(DictEntity dbDictEntity, Set<DictItemVo> items, List<DictItemEntity> dbDictItemsList) {
    for(DictItemVo item : items){
      Validate.notBlank(item.getDictKey(), "字典名称不能为空，请检查!");
      Validate.notBlank(item.getDictValue(), "字典值不能为空，请检查!");
    }

    Map<String,DictItemVo> sources = items.stream().filter(e -> StringUtils.isNotBlank(e.getId())).collect(Collectors.toMap(DictItemVo::getId, Function.identity()));
    //1.先判断筛选新增
    //注：由于历史版本原因，新增时，前端对id进行了赋值，这里需要区别分析，兼容以前的设计
    // 考虑到后期可能前端突然不对id进行赋值，那么这里需要兼容考虑
    Set<String> dbIds = dbDictItemsList.stream().map(DictItemEntity::getId).collect(Collectors.toSet());
    //这里是全有值的情况
    Set<String> ids = items.stream().filter(e -> StringUtils.isNotBlank(e.getId())).map(DictItemVo::getId).collect(Collectors.toSet());
    //这里是可能没有id的字典明细
    Set<DictItemVo> noIds = items.stream().filter(e -> StringUtils.isBlank(e.getId())).collect(Collectors.toSet());

    Set<String> needAddIds = Sets.difference(ids,dbIds);
    Set<DictItemVo> needAdds = Sets.newHashSet(noIds);
    Set<DictItemVo> needUpdates = Sets.newHashSet();
    for(String needAddId : needAddIds){
      needAdds.add(sources.get(needAddId));
    }

    //2.更新的情况
    Set<String> needUpdateIds = Sets.intersection(dbIds,ids);
    for(String needUpdateId : needUpdateIds){
      needUpdates.add(sources.get(needUpdateId));
    }

    //3.判断重复
    Set<DictItemVo> unions = Sets.union(needAdds,needUpdates);
    //因为暂不考虑删除功能，所以itemNum的值一定是前端传过来的，且有效的明细项个数值
    dbDictEntity.setItemNum(unions.size());
    Validate.isTrue(unions.stream().map(DictItemVo::getDictKey).distinct().count() == unions.size(),"字典名称重复，请检查!");
    Validate.isTrue(unions.stream().map(DictItemVo::getDictValue).distinct().count() == unions.size(),"字典值重复，请检查!");

    Map<String,Set<DictItemVo>> result = Maps.newHashMap();
    result.put(ADD,needAdds);
    result.put(UPDATE,needUpdates);
    return result;
  }
}
