package com.bizunited.platform.dictionary.service.local.service.internal;

import com.alibaba.fastjson.JSONObject;
import com.bizunited.platform.common.service.NebulaToolkitService;
import com.bizunited.platform.dictionary.common.service.DictItemOrgMappingService;
import com.bizunited.platform.dictionary.common.service.dictItem.DictItemService;
import com.bizunited.platform.dictionary.common.vo.DictItemOrgMappingVo;
import com.bizunited.platform.dictionary.common.vo.DictItemVo;
import com.bizunited.platform.dictionary.service.local.entity.DictItemEntity;
import com.bizunited.platform.dictionary.service.local.entity.DictItemOrgMappingEntity;
import com.bizunited.platform.dictionary.service.local.repository.DictItemOrgMappingRepository;
import com.bizunited.platform.user.common.service.organization.OrganizationExtendService;
import com.bizunited.platform.user.common.service.organization.OrganizationService;
import com.bizunited.platform.user.common.vo.OrganizationExtendVo;
import com.bizunited.platform.user.common.vo.OrganizationVo;
import org.apache.commons.compress.utils.Lists;
import org.apache.commons.lang3.ObjectUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.PageImpl;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Pageable;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;

import javax.transaction.Transactional;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.HashSet;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Map;

@Service("DictItemOrgMappingServiceImpl")
public class DictItemOrgMappingServiceImpl implements DictItemOrgMappingService {

  @Autowired
  private OrganizationService organizationService;
  @Autowired
  private DictItemService dictItemService;
  @Autowired
  private NebulaToolkitService nebulaToolkitService;
  @Autowired
  private DictItemOrgMappingRepository dictItemOrgMappingRepository;
  @Autowired
  private OrganizationExtendService organizationExtendService;

  @Override
  @Transactional
  public OrganizationExtendVo createAndBindDictItem(OrganizationExtendVo vo, String id) {
    Validate.notNull(vo, "扩展组织机构为空，请检查!");
    Validate.notNull(id, "数据字典值为空请检查!");
    Validate.notNull(vo.getOrganization(), "组织机构为空请检查!");
    OrganizationExtendVo organizationExtendVo = organizationExtendService.create(vo);
    this.bindOrgDictItem(vo.getOrganization().getCode(), id);
    return organizationExtendVo;
  }

  @Override
  @Transactional
  public OrganizationExtendVo updateAndBindDictItem(OrganizationExtendVo vo, String id) {
    Validate.notNull(vo, "扩展组织机构为空，请检查!");
    Validate.notNull(id, "数据字典值为空请检查!");
    Validate.notNull(vo.getOrganization(), "组织机构为空请检查!");
    DictItemOrgMappingEntity entity = this.dictItemOrgMappingRepository.findByOrgCode(vo.getOrganization().getCode());
    DictItemVo dictItem = dictItemService.findDetailsById(id);
    DictItemEntity dictItemEntity = nebulaToolkitService.copyObjectByWhiteList(dictItem, DictItemEntity.class, HashSet.class, ArrayList.class);
    entity.setDictItem(dictItemEntity);
    entity.setOrgCode(vo.getOrganization().getCode());
    this.dictItemOrgMappingRepository.deleteByOrgCode(vo.getOrganization().getCode());
    this.dictItemOrgMappingRepository.save(entity);
    OrganizationExtendVo organizationExtendVo = organizationExtendService.update(vo);
    this.bindOrgDictItem(vo.getOrganization().getCode(), id);
    return organizationExtendVo;
  }

  @Override
  @Transactional
  public void bindOrgDictItem(String code, String dictItemId) {
    Validate.notBlank(code, "进行组织机构-层级绑定时，必须指定当前的组织机构信息!!");
    Validate.notBlank(dictItemId, "进行组织机构-层级绑定时,至少传入数据字典值一个信息!!");
    OrganizationVo organization = organizationService.findByCode(code);
    Validate.notNull(organization, "未找到组织机构：%s", code);
    DictItemVo vo = this.dictItemService.findDetailsById(dictItemId);
    Validate.notNull(vo, "未找到数据字典值");
    DictItemOrgMappingEntity mapping = new DictItemOrgMappingEntity();
    DictItemEntity current = new DictItemEntity();
    current.setId(dictItemId);
    current.setDictKey(vo.getDictKey());
    current.setDictValue(vo.getDictValue());
    current.setDictSort(vo.getDictSort());
    mapping.setDictItem(current);
    mapping.setOrgCode(code);
    dictItemOrgMappingRepository.save(mapping);
  }

  @Override
  public Page<OrganizationVo> queryPage(String dictItemId, Pageable pageable) {
    if (StringUtils.isBlank(dictItemId)){
      return new PageImpl<>(Lists.newArrayList(), pageable, 0l);
    }
    DictItemVo vo = this.dictItemService.findDetailsById(dictItemId);
    Integer sort = vo.getDictSort()-1;
    if (sort < 1){
      return new PageImpl<>(Lists.newArrayList(), pageable, 0l);
    }
    Page<DictItemOrgMappingEntity> entityPage = dictItemOrgMappingRepository.queryPage(sort, pageable);
    List<DictItemOrgMappingEntity> content = entityPage.getContent();
    List<OrganizationVo> vos = new ArrayList<>();
    Page<OrganizationVo> page;
    if (!content.isEmpty()){
      for (DictItemOrgMappingEntity entity : content) {
        OrganizationVo organizationVo = organizationService.findByCode(entity.getOrgCode());
        vos.add(organizationVo);
      }
      page = new PageImpl<>(new ArrayList<>(vos), pageable, entityPage.getTotalElements());
    }else {
      page = new PageImpl<>(Lists.newArrayList(), pageable, 0l);
    }
    return page;
  }

  @Override
  public Page<JSONObject> findByConditions(String id, String orgName, String code, String createUser, Integer tstatus, Pageable pageable) {
    pageable = ObjectUtils.defaultIfNull(pageable, PageRequest.of(0, 50));
    Map<String, Object> parameters = new HashMap<>();
    parameters.put("id", id);
    parameters.put("code", code);
    parameters.put("tstatus", tstatus);
    parameters.put("orgName", orgName);
    parameters.put("createUser", createUser);
    Page<DictItemOrgMappingEntity> entityPage = dictItemOrgMappingRepository.findByConditions(parameters, pageable);
    List<DictItemOrgMappingEntity> content = entityPage.getContent();
    if (CollectionUtils.isEmpty(content)) {
      return Page.empty(pageable);
    }
    List<JSONObject> jsonObjects = new ArrayList<>();
    Collection<DictItemOrgMappingVo> vos = this.nebulaToolkitService.copyCollectionByWhiteList(content, DictItemOrgMappingEntity.class, DictItemOrgMappingVo.class, LinkedHashSet.class, ArrayList.class, "dictItem");
    for (DictItemOrgMappingVo vo : vos){
      jsonObjects.add(this.buildProperty(vo));
    }
    return new PageImpl<>(new ArrayList<>(jsonObjects), pageable, entityPage.getTotalElements());
  }

  @Override
  public List<OrganizationVo> findOrgByConditions(String dictItemId, String name) {
    if (StringUtils.isBlank(dictItemId)) {
      return Lists.newArrayList();
    }
    DictItemVo vo = this.dictItemService.findDetailsById(dictItemId);
    Integer sort = Integer.parseInt(vo.getDictValue());
    Map<String, Object> parameters = new HashMap<>();
    if (sort <= 1) {
      return Lists.newArrayList();
    }
    Integer dictValue = sort-1;
    parameters.put("dictValue", dictValue);
    parameters.put("name", name);
    List<DictItemOrgMappingEntity> entityList = dictItemOrgMappingRepository.findOrgByConditions(parameters);
    List<OrganizationVo> vos = new ArrayList<>();
    if (!entityList.isEmpty()) {
      for (DictItemOrgMappingEntity entity : entityList) {
        OrganizationVo organizationVo = organizationService.findByCode(entity.getOrgCode());
        vos.add(organizationVo);
      }
    }
    return vos;
  }

  @Override
  public JSONObject findDetailByCode(String code) {
    Validate.notNull(code, "根据组织编码查询组织信息");
    DictItemOrgMappingEntity entity = this.dictItemOrgMappingRepository.findByOrgCode(code);
    DictItemOrgMappingVo vo = this.nebulaToolkitService.copyObjectByWhiteList(entity, DictItemOrgMappingVo.class, LinkedHashSet.class, ArrayList.class, "dictItem");
    return this.buildProperty(vo);
  }

  /**
   * 赋值json对象
   * @param vo
   * @return
   */
  private JSONObject buildProperty(DictItemOrgMappingVo vo){
    OrganizationVo organization = this.organizationService.findByCode(vo.getOrgCode());
    Validate.notNull(organization, "不存在code为[%s]的组织机构,请检查!", vo.getOrgCode());
    OrganizationVo org = this.organizationService.findDetailsById(organization.getId());
    JSONObject jsonitem = new JSONObject();
    jsonitem.put("id", org.getId());
    jsonitem.put("code", org.getCode());
    jsonitem.put("orgName", org.getOrgName());
    jsonitem.put("dictItemId", vo.getDictItem().getId());
    jsonitem.put("dictValue", vo.getDictItem().getDictKey());
    if (org.getParent() != null) {
      jsonitem.put("parentName", org.getParent().getOrgName());
    }else {
      jsonitem.put("parentName", null);
    }
    jsonitem.put("tstatus", org.getTstatus());
    jsonitem.put("createUser", org.getCreateUser());
    jsonitem.put("createTime", org.getCreateTime());
    jsonitem.put("updateTime", org.getUpdateTime());
    jsonitem.put("updateUser", org.getUpdateUser());
    return jsonitem;
  }
}
