package com.bizunited.platform.dictionary.service.local.service.internal;

import com.bizunited.platform.common.service.NebulaToolkitService;
import com.bizunited.platform.common.util.ApplicationContextUtils;
import com.bizunited.platform.dictionary.common.service.dict.DictService;
import com.bizunited.platform.dictionary.common.service.dictItem.DictItemService;
import com.bizunited.platform.dictionary.common.vo.DictItemVo;
import com.bizunited.platform.dictionary.common.vo.DictVo;
import com.bizunited.platform.dictionary.service.local.entity.DictEntity;
import com.bizunited.platform.dictionary.service.local.entity.DictItemEntity;
import com.bizunited.platform.dictionary.service.local.repository.DictItemRepository;
import com.bizunited.platform.dictionary.service.local.repository.DictRepository;
import com.google.common.collect.Lists;
import com.google.common.collect.Sets;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;

import javax.transaction.Transactional;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Date;
import java.util.HashMap;
import java.util.HashSet;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;

@Service("DictItemServiceImpl")
public class DictItemServiceImpl implements DictItemService {

  private static final String ERROR_MESS = "字典明细ID不能为空！";
  @Autowired 
  private DictRepository dictRepository;
  @Autowired 
  private DictItemRepository dictItemRepository;
  @Autowired
  private DictService dictService;
  @Autowired
  private NebulaToolkitService nebulaToolkitService;

  @Override
  public List<DictItemVo> findItemsByCode(String dictCode) {
    DictEntity dictEntity = dictRepository.findByDictCode(dictCode);
    if (dictEntity == null) {
      return Lists.newArrayList();
    }
    List<DictItemEntity> dictItems = dictItemRepository.findByDictEntityOrderByDictSort(dictEntity);
    Collection<DictItemVo> dictItemVos = this.nebulaToolkitService.copyCollectionByWhiteList(dictItems, DictItemEntity.class, DictItemVo.class, LinkedHashSet.class, ArrayList.class);
    return Lists.newArrayList(dictItemVos);
  }

  @Override
  @Transactional
  public Set<DictItemVo> save(String dictCode, Set<DictItemVo> items) {
    Map<String, DictItemVo> map = new HashMap<>();
    for(DictItemVo dictItemVo : items ){
      map.put(dictItemVo.getDictValue(), dictItemVo);
    }

    DictVo dict = dictService.findDetailsByDictCode(dictCode);
    Validate.notNull(dict, "未找到该编码匹配字典！");
    DictEntity entity = this.nebulaToolkitService.copyObjectByWhiteList(dict, DictEntity.class, HashSet.class, ArrayList.class);
    List<DictItemEntity> dictItemsList = dictItemRepository.findByDictEntity(entity);
    if(items == null) {
      items = new HashSet<>();
    }
    //参数校验
    this.saveValidation(dictCode, items, dictItemsList);
    // TODO: 2020/1/10 目前的业务只有新增的情况，更新删除待后期开发(如果后期业务需要)
    //可能的更新 更新Value对应的key(显示字段)
    Set<DictItemEntity> needInsertDictItems = new HashSet<>();
    Set<DictItemEntity> needUpdateDictItems = new HashSet<>();
    Set<DictItemEntity> needDeleteDictItems = new HashSet<>();
    Collection<DictItemEntity> dictItemEntities = this.nebulaToolkitService.copyCollectionByWhiteList(items, DictItemVo.class, DictItemEntity.class, LinkedHashSet.class, ArrayList.class);
    nebulaToolkitService.collectionDiscrepancy(dictItemEntities, dictItemsList, DictItemEntity::getDictValue, needDeleteDictItems, needUpdateDictItems, needInsertDictItems);

    int itemNum;
    if (CollectionUtils.isEmpty(needUpdateDictItems)){
      itemNum = items.size() + dictItemsList.size();
    }else {
      itemNum = items.size() - needUpdateDictItems.size() + dictItemsList.size();
    }
    dict.setItemNum(itemNum);
    dict.setDictItems(null);
    DictEntity dictEntity = this.nebulaToolkitService.copyObjectByWhiteList(dict, DictEntity.class, HashSet.class, ArrayList.class, "category");
    dictEntity.setProjectName(ApplicationContextUtils.getProjectName());
    dictRepository.saveAndFlush(dictEntity);
    dictItemRepository.flush();

    needInsertDictItems.stream().forEach(dictItem -> {
      dictItem.setId(null);
      dictItem.setCreateTime(new Date());
      dictItem.setDictSort(dictItem.getDictSort() == null ? 1 : dictItem.getDictSort());
      dictItem.setDictEntity(dictEntity);
      dictItem.setDictItemStatus(dictItem.getDictItemStatus() == null ? Boolean.TRUE : dictItem.getDictItemStatus());
      dictItem.setProjectName(ApplicationContextUtils.getProjectName());
    });
    dictItemRepository.saveAll(needInsertDictItems);

    for(DictItemEntity dictItem : needUpdateDictItems){
      DictItemVo mapItem = map.get(dictItem.getDictValue());
      dictItem.setDictItemStatus(mapItem.getDictItemStatus());
      dictItem.setDictKey(mapItem.getDictKey());
      dictItem.setDictSort(mapItem.getDictSort());
      dictItemRepository.save(dictItem);
    }
    Collection<DictItemVo> dictItemVos = this.nebulaToolkitService.copyCollectionByWhiteList(needInsertDictItems, DictItemEntity.class, DictItemVo.class, LinkedHashSet.class, ArrayList.class);
    return Sets.newLinkedHashSet(dictItemVos);
  }

  @Override
  @Transactional
  public DictItemVo create(String dictCode, DictItemVo dictItem) {
    //参数校验
    this.createValidation(dictCode, dictItem);

    DictVo dict = dictService.findByDictCode(dictCode);
    dict.setItemNum(dict.getItemNum() + 1);
    DictEntity dictEntity = this.nebulaToolkitService.copyObjectByWhiteList(dict, DictEntity.class, HashSet.class, ArrayList.class);
    dictRepository.save(dictEntity);

    DictItemEntity dictItemEntity = this.nebulaToolkitService.copyObjectByWhiteList(dictItem, DictItemEntity.class, HashSet.class, ArrayList.class);
    dictItemEntity.setCreateTime(new Date());
    dictItemEntity.setDictSort(dictItem.getDictSort() == null ? 1 : dictItem.getDictSort());
    dictItemEntity.setDictEntity(dictEntity);
    dictItemEntity.setDictItemStatus(dictItem.getDictItemStatus() == null ? Boolean.TRUE : dictItem.getDictItemStatus());

    DictItemEntity entity = dictItemRepository.save(dictItemEntity);
    return this.nebulaToolkitService.copyObjectByWhiteList(entity, DictItemVo.class, HashSet.class, ArrayList.class);
  }

  @Override
  public DictItemVo findDetailsById(String id) {
    Validate.notNull(id, "查询字典值明细参数为空，请检查!");
    DictItemEntity entity = dictItemRepository.findDetailById(id);
    return this.nebulaToolkitService.copyObjectByWhiteList(entity, DictItemVo.class, HashSet.class, ArrayList.class,  "dict");
  }

  /**
   * 根据字典dictCode和字典值的key查询一个数据字典值详细
   *
   * @param dictCode
   * @param key
   * @return
   */
  @Override
  public DictItemVo findByKeyAndDictCode(String dictCode, String key) {
    if (StringUtils.isBlank(dictCode) || StringUtils.isBlank(key)){
      return null;
    }
    DictItemEntity dictItem = this.dictItemRepository.findByKeyAndDictCode(dictCode, key);
    if (dictItem == null){
      return null;
    }
    return this.nebulaToolkitService.copyObjectByWhiteList(dictItem, DictItemVo.class, HashSet.class, ArrayList.class,  "dictEntity");
  }

  /**
   * 新增-参数校验
   * @param dictCode
   * @param dictItem
   */
  private void createValidation(String dictCode, DictItemVo dictItem) {
    Validate.notBlank(dictCode, "字典编码不能为空，请检查");
    DictVo dict = dictService.findByDictCode(dictCode);
    Validate.notNull(dict, "未找到该编码匹配字典！");

    Validate.notNull(dictItem, "保存对象不存在!");
    Validate.isTrue(dictItem.getId() == null, "保存数据ID必须为空!");
    Validate.notBlank(dictItem.getDictKey(), "字典名称不能为空，请检查!");
    Validate.notBlank(dictItem.getDictValue(), "字典值不能为空，请检查!");

    long countByDictKey = dictItemRepository.countByDictKeyAndDictId(dictItem.getDictKey(), dict.getId());
    Validate.isTrue(countByDictKey == 0L, "字典名称重复，请检查!");
    long countByDictValue = dictItemRepository.countByDictValueAndDictId(dictItem.getDictValue(), dict.getId());
    Validate.isTrue(countByDictValue == 0L, "字典值重复，请检查!");
  }

  @Override
  @Transactional
  public DictItemVo enable(String dictItemId) {
    Validate.notBlank(dictItemId, ERROR_MESS);
    DictItemEntity dictItemEntity = dictItemRepository.findDetailById(dictItemId);
    Validate.notNull(dictItemEntity, ERROR_MESS);
    dictItemEntity.setDictItemStatus(true);
    DictItemEntity dictItem = dictItemRepository.save(dictItemEntity);
    return this.nebulaToolkitService.copyObjectByWhiteList(dictItem, DictItemVo.class, HashSet.class, ArrayList.class);
  }

  @Override
  @Transactional
  public DictItemVo disable(String dictItemId) {
    Validate.notBlank(dictItemId, ERROR_MESS);
    DictItemEntity dictItemEntity = dictItemRepository.findDetailById(dictItemId);
    Validate.notNull(dictItemEntity, ERROR_MESS);
    dictItemEntity.setDictItemStatus(false);
    DictItemEntity dictItem = dictItemRepository.save(dictItemEntity);
    return this.nebulaToolkitService.copyObjectByWhiteList(dictItem, DictItemVo.class, HashSet.class, ArrayList.class);
  }

  /**
   * 参数校验
   * 重复性验证：value只需要验证传递过来的是否重复，key还需要验证与数据库里是否重复
   * @param dictCode
   * @param items 前端传递过来的
   * @param dictItemsList 数据库里原有的
   */
  private void saveValidation(String dictCode, Set<DictItemVo> items, List<DictItemEntity> dictItemsList) {
    Validate.notBlank(dictCode, "字典编码不能为空，请检查");
    Collection<DictItemEntity> dictItemCollection = nebulaToolkitService.copyCollectionByWhiteList(dictItemsList, DictItemEntity.class, DictItemEntity.class, HashSet.class, ArrayList.class);
    List<DictItemEntity> oldItems = Lists.newArrayList(dictItemCollection);

    Set<DictItemEntity> needInsertItems = new HashSet<>();
    Collection<DictItemEntity> dictItems = nebulaToolkitService.copyCollectionByWhiteList(items, DictItemVo.class, DictItemEntity.class, HashSet.class, ArrayList.class);
    needInsertItems.addAll(dictItems);
    Map<String, DictItemEntity>  oldItemsMap = oldItems.stream().collect(Collectors.toMap(DictItemEntity::getDictValue, e -> e));

    dictItems.forEach(item -> {
      //能找到就是更新
      DictItemEntity dictItemEntity = oldItemsMap.get(item.getDictValue());
      if (null != dictItemEntity){
        //移除需要更新的字典明细项
        needInsertItems.remove(item);
        dictItemEntity.setDictKey(item.getDictKey());
      }
    });

    Set<String> dictItemValues = new HashSet<>();
    Set<String> dictItemKeys = new HashSet<>();

    dictItems.forEach(item -> {
      Validate.notNull(item, "保存对象不存在!");
      Validate.notBlank(item.getDictKey(), "字典名称不能为空，请检查!");
      Validate.notBlank(item.getDictValue(), "字典值不能为空，请检查!");
      //判断前端传递过来的字典Value和字典Key是否重复
      Validate.isTrue(!dictItemValues.contains(item.getDictValue()), "字典值重复，请检查!");
      Validate.isTrue(!dictItemKeys.contains(item.getDictKey()), "字典名称重复，请检查!");
      dictItemValues.add(item.getDictValue());
      dictItemKeys.add(item.getDictKey());
    });

    needInsertItems.addAll(oldItems);
    Set<String> dbDictItemKeys = new HashSet<>();
    needInsertItems.forEach(item -> {
      Validate.isTrue(!dbDictItemKeys.contains(item.getDictKey()), "字典名称重复，请检查!");
      dbDictItemKeys.add(item.getDictKey());
    });
  }
}
