package com.bizunited.platform.kuiper.starter.common.excel;

import com.bizunited.platform.kuiper.starter.common.enums.ExcelTypeEnum;
import com.bizunited.platform.kuiper.starter.common.excel.exception.ExcelWraperException;
import com.bizunited.platform.kuiper.starter.common.excel.reader.IExcelReader;
import com.bizunited.platform.kuiper.starter.common.excel.reader.impl.DomExcelReader;
import com.bizunited.platform.kuiper.starter.common.excel.reader.impl.NewExcelEventReader;
import com.bizunited.platform.kuiper.starter.common.excel.reader.impl.OldExcelEventReader;
import org.apache.commons.lang3.Validate;
import org.apache.poi.openxml4j.exceptions.OpenXML4JException;
import org.apache.poi.poifs.filesystem.FileMagic;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.xml.sax.SAXException;

import java.io.File;
import java.io.IOException;
import java.io.InputStream;

/**
 * @author Keller
 * @create 2020/8/24
 */
public class ExcelImportWrapperBuilder {

  public static final Logger LOGGER = LoggerFactory.getLogger(ExcelImportWrapperBuilder.class);

  /**
   * excel读取文件
   */
  private File fileName;

  /**
   * excel类型
   */
  private ExcelTypeEnum excelTypeEnum;
  /**
   * excel文件输入流
   */
  private InputStream input;
  /**
   * dom读取文件方式开关（默认false）
   * dom方式不适合读取大文件数据
   */
  private boolean isDomReader = false;

  /**
   * ExcelImportWrapper构造器 通过输入流
   *
   * @param input
   * @return
   */
  public static ExcelImportWrapperBuilder create(InputStream input, File fileName) {
    return new ExcelImportWrapperBuilder(input, fileName);
  }

  private ExcelImportWrapperBuilder(InputStream input, File fileName) {
    this.input = input;
    this.fileName = fileName;
  }

  public ExcelImportWrapperBuilder setDomReader(boolean isDomReader) {
    this.isDomReader = isDomReader;
    return this;
  }

  /**
   * 通过文件头字节判断文件格式（不使用文件后缀）
   *
   * @return
   * @throws ExcelWraperException
   */
  public ExcelImportWrapper builder() {
    Validate.notNull(input, "输入流不能为空");
    try (InputStream is = FileMagic.prepareToCheckMagic(input)) {
      FileMagic fileMagic = FileMagic.valueOf(is);
      switch (fileMagic) {
        case OLE2:
          excelTypeEnum = ExcelTypeEnum.XLS;
          break;
        case OOXML:
          excelTypeEnum = ExcelTypeEnum.XLSX;
          break;
        default:
          throw new ExcelWraperException("excel格式解析错误");
      }
      IExcelReader excelReader = null;
      if (isDomReader) {
        excelReader = new DomExcelReader(excelTypeEnum, input);
      } else {
        switch (excelTypeEnum) {
          case XLS:
            excelReader = new OldExcelEventReader(is);
            break;
          case XLSX:
            excelReader = new NewExcelEventReader(is, fileName);
            break;
          default:
            break;
        }
      }
      return new ExcelImportWrapper(excelReader);
    } catch (IOException e) {
      LOGGER.error(e.getMessage(), e);
      throw new ExcelWraperException("excel文件读取错误，请检查", e);
    } catch (SAXException e) {
      LOGGER.error(e.getMessage(), e);
      throw new ExcelWraperException("excel文件格式读取错误，请检查", e);
    } catch (OpenXML4JException e) {
      LOGGER.error(e.getMessage(), e);
      throw new ExcelWraperException("excel文件格式读取错误，请检查", e);
    }
  }
}
